//-*-c++-*-
#ifndef INCLUDED_WalkNode_h_
#define INCLUDED_WalkNode_h_

#include "Behaviors/StateNode.h"
#include "Motion/MotionManager.h"
#include "Motion/WalkMC.h"
#include "Motion/MMAccessor.h"
#include "Events/LocomotionEvent.h"
#include "Events/EventRouter.h"

//! A StateNode for walking in a direction
class WalkNode : public StateNode {
public:
	//! lets us interpret values as either distances or velocities
  enum WalkMode_t {
		VelocityWalkMode, //!< #x, #y, #a will be interpreted as mm/s
		DistanceWalkMode //!< #x, #y, #a will be interpreted as millimeters
	};

public:
	//!constructor
	WalkNode()
	  : StateNode("WalkNode"), walkid(MotionManager::invalid_MC_ID), walkidIsMine(true), x(0), y(0), a(0), n(-1), walkMode()
	{}

	//!constructor, positive @a yvel is counter-clockwise from above (to match coordinate system), assumes velocity
	WalkNode(float xvel, float yvel, float avel)
	  : StateNode("WalkNode"), walkid(MotionManager::invalid_MC_ID), walkidIsMine(true), x(xvel), y(yvel), a(avel), n(-1), walkMode(VelocityWalkMode)
	{}

	//!constructor, positive @a yvel is counter-clockwise from above (to match coordinate system), assumes distance
	WalkNode(float xvel, float yvel, float avel, int steps)
	  : StateNode("WalkNode"), walkid(MotionManager::invalid_MC_ID), walkidIsMine(true), x(xvel), y(yvel), a(avel), n(steps), walkMode(DistanceWalkMode)
	{}

	//!constructor, positive @a yvel is counter-clockwise from above (to match coordinate system), assumes velocity
	WalkNode(const std::string& name, float xvel, float yvel, float avel)
	  : StateNode("WalkNode",name), walkid(MotionManager::invalid_MC_ID), walkidIsMine(true), x(xvel), y(yvel), a(avel), n(-1), walkMode(VelocityWalkMode)
	{}

	//!constructor, positive @a yvel is counter-clockwise from above (to match coordinate system), assumes distance
	WalkNode(const std::string& name, float xvel, float yvel, float avel, int steps)
	  : StateNode("WalkNode",name), walkid(MotionManager::invalid_MC_ID), walkidIsMine(true), x(xvel), y(yvel), a(avel), n(steps), walkMode(DistanceWalkMode)
	{}

	//!destructor, check if we need to call our teardown
	~WalkNode() {
	  if(issetup)
	    teardown();
	}

	//! sets the velocity of the walk
	void setDisplacement(float xd, float yd, float ad, int np = -1) {
	  //cout << "SET_DISPLACEMENT" << endl;
	  //updateWalk(xd,yd,ad,np,false);
	  storeValues(xd, yd, ad, np, DistanceWalkMode);
	}

	//! sets the velocity of the walk
	void setVelocity(float xvel, float yvel, float avel, int np = -1) {
	  //updateWalk(xvel,yvel,avel,np);
	  storeValues(xvel, yvel, avel, np, VelocityWalkMode);
	}
	
	//! sets the velocity in x direction (positive is forward)
	void setXVelocity(float xvel) { x=xvel; storeValues(xvel, y, a, n, VelocityWalkMode); }

	//! returns the velocity in x direction (positive is forward)
	float getXVelocity() { return x; }

	//! sets the velocity in y direction (positive is forward)
	void setYVelocity(float yvel) { y=yvel; storeValues(x, yvel, a, n, VelocityWalkMode); }

	//! returns the velocity in y direction (positive is forward)
	float getYVelocity() { return y; }

	//! sets the velocity of the turn, positive is counter-clockwise from above (to match coordinate system)
	void setAVelocity(float avel) { a=avel; storeValues(x, y, avel, n, VelocityWalkMode); }

	//! returns the velocity of the turn, positive is counter-clockwise from above (to match coordinate system)
	float getAVelocity() { return a; }

	virtual void DoStart() {
		StateNode::DoStart();
		if (walkid != MotionManager::invalid_MC_ID) {
		  erouter->addListener(this, EventBase::locomotionEGID, walkid, EventBase::statusETID);
		}

	  updateWMC();
	}

	virtual void DoStop() {
	  //added the 0
	  //cout << "-----------WALK NODE: DoStop()" << endl;
	  erouter->removeListener(this);
	  if(walkid!=MotionManager::invalid_MC_ID) {
	    MMAccessor<WalkMC> walk(walkid);
	    walk->setTargetVelocity(0,0,0,0);
	  }
	  //updateWalk(0,0,0,0);
	  StateNode::DoStop();
	}

        //! receive locomotionEGID status event and throw stateMachineEGID status event, ie a completion event
        virtual void processEvent(const EventBase& e) {
	  //cout << "PROCESSING WALK EVENTS" << endl;
	  if (e.getGeneratorID() == EventBase::locomotionEGID) {
	    const LocomotionEvent le = *reinterpret_cast<const LocomotionEvent*>(&e);
	    //cout << "LE description: " << le.getDescription() << endl;
	    //cout << "LE X: " << le.x << " = 0? " << (le.x == 0) << endl;
	    //cout << "LE Y: " << le.y << " = 0? " << (le.y == 0) << endl;
	    //cout << "LE A: " << le.a << " = 0? " << (le.a == 0) << endl;
	    if (le.x == 0 && le.y == 0 && le.a == 0) {
	      //cout << "Posting Completion Event for Walk." << endl;
	      postCompletionEvent();
	    }
	  }
        }


	//! removes #walkid if #walkidIsMine
	virtual void teardown() {
		if(walkidIsMine) {
			motman->removeMotion(walkid);
			walkid=MotionManager::invalid_MC_ID;
		}
		StateNode::teardown();
	}

	//! use this to force the WalkNode to use a shared WalkMC - set to MotionManager::invalid_MC_ID to reset to internally generated walk
	virtual void setWalkID(MotionManager::MC_ID id) {
		if(walkidIsMine) {
			motman->removeMotion(walkid);
			walkid=MotionManager::invalid_MC_ID;
		}
		erouter->removeListener(this, EventBase::locomotionEGID);
		walkid=id;
		walkidIsMine=(id==MotionManager::invalid_MC_ID);
		erouter->addListener(this, EventBase::locomotionEGID, walkid, EventBase::statusETID);
	}

	//! use this to access the WalkMC that the WalkNode is using
	virtual MotionManager::MC_ID getWalkID() { return walkid; }

	//! returns true if #walkid was created (and will be destroyed) by this WalkNode - false if assigned by setWalkID()
	virtual bool ownsWalkID() { return walkidIsMine; }

protected:
	//! stores the values and if active, calls updateWMC()
  void storeValues(float xp, float yp, float ap, int np, WalkMode_t wmode) {
    x = xp;
    y = yp;
    a = ap;
    n = np;
    walkMode = wmode;

    if (isActive()) {
      updateWMC();
    }
  }

	//! makes the appropriate calls on the WalkMC
  void updateWMC() {
    if(walkid==MotionManager::invalid_MC_ID) {
      SharedObject<WalkMC> walk;
      MotionManager::MC_ID id = motman->addPersistentMotion(walk);
      setWalkID(id);
      walkidIsMine=true;
    }
    MMAccessor<WalkMC> walk(walkid);
    switch(walkMode) {
    case VelocityWalkMode:
      walk->setTargetVelocity(x,y,a,n);
      break;
    case DistanceWalkMode:
      walk->setTargetDisplacement(x,y,a,n); // WalkMC will calculate velocities.
      break;
    default:
      std::cout << "Unknown Walk Mode" << std::endl;
      break;
    }
  }

  /*
  void updateWMC() {
    if(walkid==MotionManager::invalid_MC_ID) {
      SharedObject<WalkMC> walk;
      switch(walkMode) {
      case VelocityWalkMode:
	walk->setTargetVelocity(x,y,a,n);
	break;
      case DistanceWalkMode:
	walk->setTargetDisplacement(x,y,a,n); // WalkMC will calculate velocities.
	break;
      default:
	cout << "Unknown Walk Mode" << endl;
	break;
      }
      MotionManager::MC_ID id = motman->addPersistentMotion(walk);
      setWalkID(id);
      walkidIsMine=true;
    } else {
      MMAccessor<WalkMC> walk(walkid);
      switch(walkMode) {
      case VelocityWalkMode:
	walk->setTargetVelocity(x,y,a,n);
	break;
      case DistanceWalkMode:
	walk->setTargetDisplacement(x,y,a,n); // WalkMC will calculate velocities.
	break;
      default:
	cout << "Unknown Walk Mode" << endl;
	break;
      }
    }
  }
  */

  /*
	//!if the walk is invalid, create; then set xya
	void updateWalk(float xp, float yp, float ap, int np=-1, bool isVelocity=true) {
	  cout << "NNNNNNNNNPPPPPPPPP: " << np << endl;
	  vector3d velocities;
	  if(walkid==MotionManager::invalid_MC_ID) {
	    SharedObject<WalkMC> walk;
	    if (isVelocity)
	      walk->setTargetVelocity(xp,yp,ap,np);
	    else
	      walk->setTargetDisplacement(xp,yp,ap,np); // WalkMC will calculate velocities.
	    velocities = walk->getCurVelocity();
	    MotionManager::MC_ID id = motman->addPersistentMotion(walk);
	    setWalkID(id);
	    walkidIsMine=true;
	  } else {
	    MMAccessor<WalkMC> walk(walkid);
	    if (isVelocity)
	      walk->setTargetVelocity(xp,yp,ap,np);
	    else
	      walk->setTargetDisplacement(xp,yp,ap,np); // WalkMC will calculate velocities.

	    velocities = walk->getCurVelocity();  // extract the velocities WalkMC calculated
	  }
	  
	  x = velocities.x;
	  y = velocities.y;
	  a = velocities.z;
	  n = np;
	}
  */

	MotionManager::MC_ID walkid; //!< the current WalkMC
	bool walkidIsMine; //!< true if the walk was created in updateWalk (instead of assigned externally)
	float x; //!< velocity in x direction (positive is forward), or distance if #walkMode is DistanceWalkMode
	float y; //!< velocity in y direction (positive is dog's left), or distance if #walkMode is DistanceWalkMode
	float a; //!< velocity of the turn, positive is counter-clockwise from above (to match coordinate system), or distance if #walkMode is DistanceWalkMode
	int n; //!< number of steps (-1 means walk forever)

  WalkMode_t walkMode; //!< the current interpretation of #x, #y, and #a
};

/*! @file
 * @brief Describes WalkNode, a  StateNode for walking in a direction
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_4_1 $
 * $Revision: 1.15 $
 * $State: Exp $
 * $Date: 2005/08/07 04:11:03 $
 */

#endif
