#include "SpiderMachineBehavior.h"
#include "Events/EventRouter.h"
#include "Behaviors/StateNode.h"
#include "Wireless/Wireless.h"

SpiderMachineBehavior * SpiderMachineBehavior::theOne=NULL;
unsigned int SpiderMachineBehavior::port=10081;

void SpiderMachineBehavior::DoStart() {
	BehaviorBase::DoStart(); // do this first
	theOne=this;
	// Turn on wireless
	cmdsock=wireless->socket(SocketNS::SOCK_STREAM, 1024, 1024*10);
	wireless->setDaemon(cmdsock,true);
	wireless->setReceiver(cmdsock->sock, callback);
	wireless->listen(cmdsock->sock, port);
	erouter->addListener(this,EventBase::stateMachineEGID);
	erouter->addListener(this,EventBase::stateTransitionEGID);
}

void SpiderMachineBehavior::DoStop() {
	erouter->removeListener(this);
	expected.clear();
	while(!queuedEvents.empty())
		queuedEvents.pop();
	// Close socket; turn wireless off
	wireless->setDaemon(cmdsock,false);
	wireless->close(cmdsock);
	if(theOne==this)
		theOne=NULL;
	BehaviorBase::DoStop(); // do this last
}

void SpiderMachineBehavior::processEvent(const EventBase& e) {
	if(!wireless->isConnected(cmdsock->sock) || listen.size()==0)
		return;

	if(e.getGeneratorID()==EventBase::stateTransitionEGID) {
		bool care=false;
		const Transition * trans = reinterpret_cast<Transition*>(e.getSourceID());
		const std::vector<StateNode*>& incoming=trans->getSources();
		const std::vector<StateNode*>& outgoing=trans->getDestinations();
		for(std::vector<StateNode*>::const_iterator it=incoming.begin(); it!=incoming.end() && !care; it++)
			care=isListening(*it);
		for(std::vector<StateNode*>::const_iterator it=outgoing.begin(); it!=outgoing.end() && !care; it++)
			care=isListening(*it);
		if(!care)
			return;

		if(expected.size()!=0) {
			queuedEvents.push(e);
		} else {
			cmdsock->printf("<event>\n");
			indent(1);
			cmdsock->printf("<fire id=\"%s\" time=\"%d\" />\n",trans->getName().c_str(),e.getTimeStamp());
			expected.insert(incoming.begin(),incoming.end());
			expected.insert(outgoing.begin(),outgoing.end());
			while(queuedEvents.size()>0) {
				EventBase qe=queuedEvents.front();
				queuedEvents.pop();
				processEvent(qe);
			}
		}

	} else if(e.getGeneratorID()==EventBase::stateMachineEGID) {
		if(e.getTypeID()==EventBase::statusETID)
			return;
		const StateNode * beh=reinterpret_cast<StateNode*>(e.getSourceID());
		expected_t::iterator it=expected.find(beh);
		char * format;
		if(isListening(beh)) {
			if(it==expected.end()) { //if not found
				if(queuedEvents.size()==0)
					format="<event><state%s id=\"%s\" time=\"%d\" /></event>\n"; // unexpected
				else {
					queuedEvents.push(e);
					return;
				}
			} else
				format="  <state%s id=\"%s\" time=\"%d\" />\n"; // expected as part of transition
			if(e.getTypeID()==EventBase::activateETID)
				cmdsock->printf(format,"start",beh->getName().c_str(),e.getTimeStamp());
			else if(e.getTypeID()==EventBase::deactivateETID)
				cmdsock->printf(format,"stop",beh->getName().c_str(),e.getTimeStamp());
			else
				serr->printf("WARNING: Unrecognized TypeID %d\n",e.getTypeID());
		}
		if(it!=expected.end()) { //was found
			expected.erase(it);
			if(expected.size()==0) {
				cmdsock->printf("</event>\n");
				while(queuedEvents.size()>0) {
					EventBase qe=queuedEvents.front();
					queuedEvents.pop();
					processEvent(qe);
				}
			}
		}

	} else {
		serr->printf("WARNING: Unknown event %s (%s)\n",e.getName().c_str(),e.getDescription().c_str());
	}
}

void SpiderMachineBehavior::spider(const StateNode* n, unsigned int depth/*=0*/) {
	if(n==NULL)
		return;

	const std::vector<StateNode*>& subnodes=n->getNodes();
	if(subnodes.size()==0) {
		// it's a leaf node, no subnodes or transitions between them
		indent(depth);
		cmdsock->printf("<state class=\"%s\" id=\"%s\" />\n", n->getClassName().c_str(), n->getName().c_str());
	} else {

		// first output current node's info
		indent(depth);
		cmdsock->printf("<state class=\"%s\" id=\"%s\">\n", n->getClassName().c_str(), n->getName().c_str());

		std::set<const Transition*> transitions;
		// now recurse on sub-nodes, extracting all of the subnodes transitions
		for(unsigned int i=0; i<subnodes.size(); i++) {
			spider(subnodes[i],depth+1);
			const std::vector<Transition*>& curt=subnodes[i]->getTransitions();
			transitions.insert(curt.begin(),curt.end());
		}

		// now output transitions between subnodes we collected in previous step
		for(std::set<const Transition*>::const_iterator it=transitions.begin(); it!=transitions.end(); it++) {
			indent(depth+1);
			cmdsock->printf("<transition class=\"%s\" id=\"%s\">\n", (*it)->getClassName().c_str(), (*it)->getName().c_str());
			const std::vector<StateNode*>& incoming=(*it)->getSources();
			for(unsigned int i=0; i<incoming.size(); i++) {
				indent(depth+2);
				cmdsock->printf("<source>%s</source>\n",incoming[i]->getName().c_str());
			}
			const std::vector<StateNode*>& outgoing=(*it)->getDestinations();
			for(unsigned int i=0; i<outgoing.size(); i++) {
				indent(depth+2);
				cmdsock->printf("<destination>%s</destination>\n",outgoing[i]->getName().c_str());
			}
			indent(depth+1);
			cmdsock->printf("</transition>\n");
		}

		indent(depth);
		cmdsock->printf("</state>\n");
	}
}
	
bool SpiderMachineBehavior::isListening(const StateNode* n) {
	while(n!=NULL) {
		if(listen.find(n->getName())!=listen.end())
			return true;
		n=n->getParent();
	}
	return false;
}

void SpiderMachineBehavior::indent(unsigned int level) {
	for(unsigned int i=0; i<level; i++)
		cmdsock->printf("  ");
}

const StateNode * SpiderMachineBehavior::find(const std::string& name) {
	for(registry_t::const_iterator it=registry.begin(); it!=registry.end(); it++) {
		const StateNode * cur=dynamic_cast<const StateNode*>(*it);
		if(cur!=NULL && cur->getName()==name)
			return cur;
	}
	//serr->printf("WARNING: SpiderMachineBehavior Could not find StateNode named `%s'\n",name.c_str());
	return NULL;
}

void SpiderMachineBehavior::runCommand(const std::string& s) {
	if(s==std::string("list")) {
		unsigned int numstate=0;
		for(registry_t::const_iterator it=registry.begin(); it!=registry.end(); it++) {
			const StateNode * cur=dynamic_cast<const StateNode*>(*it);
			if(cur!=NULL)
				numstate++;
		}
		cmdsock->printf("%d\n",numstate);
		for(registry_t::const_iterator it=registry.begin(); it!=registry.end(); it++) {
			const StateNode * cur=dynamic_cast<const StateNode*>(*it);
			if(cur!=NULL)
				cmdsock->printf("%s\n",cur->getName().c_str());
		}

	} else if(s.find("spider ")==0) {
		const StateNode * n=find(s.substr(7));
		if(n==NULL) {
			serr->printf("WARNING: SpiderMachineBehavior could not find \"%s\" for spidering\n",s.substr(7).c_str());
			cmdsock->printf("<model></model>\n");
		} else {
			cmdsock->printf("<model>\n");
			spider(n);
			cmdsock->printf("</model>\n");
		}

	} else if(s.find("listen ")==0) {
		listen.insert(s.substr(7));

	} else if(s.find("ignore ")==0) {
		listen.erase(s.substr(7));

	} else if(s=="clear") {
		listen.clear();

	} else {
		serr->printf("SpiderMachineBehavior::runCommand() - bad message: '%s'\n",s.c_str());
	}
}

// The command packet reassembly mechanism
int SpiderMachineBehavior::callback(char *buf, int bytes) {
	if(SpiderMachineBehavior::theOne==NULL)
		return 0;
	static std::string cmd;
	for(int i=0; i<bytes; i++) {
		if(buf[i]=='\n') {
			SpiderMachineBehavior::theOne->runCommand(cmd);
			cmd.clear();
		} else if(buf[i]!='\r')
			cmd+=buf[i];
	}
  return 0;
}

/*! @file
 * @brief Implements SpiderMachineBehavior, which when active, active and connected over network socket, outputs structure of requested state machine(s)
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_4_1 $
 * $Revision: 1.4 $
 * $State: Exp $
 * $Date: 2005/06/01 05:47:45 $
 */
