#include "RegionCamBehavior.h"
#include "Wireless/Wireless.h"
#include "Events/EventRouter.h"
#include "Events/FilterBankEvent.h"
#include "Behaviors/Controller.h"
#include "Shared/ProjectInterface.h"
#include "Vision/SegmentedColorGenerator.h"
#include "Vision/RegionGenerator.h"

RegionCamBehavior::RegionCamBehavior()
	: BehaviorBase("RegionCamBehavior"), visRegion(NULL), packet(NULL), cur(NULL), avail(0), max_buf(0)
{
}

void
RegionCamBehavior::DoStart() {
	BehaviorBase::DoStart();
	
	std::vector<std::string> args;
	args.push_back("reg"); 
	char port[50];
	snprintf(port,50,"%d",config->vision.region_port);
	args.push_back(port);
	if(config->vision.region_transport==0) {
		max_buf=UDP_WIRELESS_BUFFER_SIZE;
		visRegion=wireless->socket(SocketNS::SOCK_DGRAM, 1024, max_buf);
		args.push_back("udp");
	} else if(config->vision.region_transport==1) {
		max_buf=TCP_WIRELESS_BUFFER_SIZE;
		visRegion=wireless->socket(SocketNS::SOCK_STREAM, 1024, max_buf);
		wireless->setDaemon(visRegion,true);
		args.push_back("tcp");
	} else {
		serr->printf("ERROR: Invalid Config::vision.region_transport: %d\n",config->vision.region_transport);
		return;
	}
	wireless->listen(visRegion,config->vision.region_port);
	Controller::loadGUI("org.tekkotsu.mon.VisionGUI","RegionVisionGUI",config->vision.region_port,args);

	erouter->addListener(this,EventBase::visRegionEGID,ProjectInterface::visRegionSID);
}

void
RegionCamBehavior::DoStop() {
	erouter->removeListener(this);
	Controller::closeGUI("RegionVisionGUI");

	// this could be considered a bug in our wireless - if we don't setDaemon(...,false)
	// it will try to listen again even though we explicitly closed the server socket...
	wireless->setDaemon(visRegion,false);
	wireless->close(visRegion->sock);
	BehaviorBase::DoStop();
}

void
RegionCamBehavior::processEvent(const EventBase& e) {
	if(!wireless->isConnected(visRegion->sock))
		return;
	const FilterBankEvent* fbke=dynamic_cast<const FilterBankEvent*>(&e);
	ASSERTRET(fbke!=NULL,"unexpected event");
		bool succ=writeRegions(*fbke);
		ASSERTRET(succ,"serialization failed");
}

bool
RegionCamBehavior::openPacket(FilterBankGenerator& fbkgen, unsigned int time, unsigned int layer) {
	if(packet!=NULL)
		return false;

	avail=max_buf-1; //not sure why -1, but Alok had it, so i will too
	ASSERT(cur==NULL,"cur non-NULL");
	cur=NULL;
	char * buf=packet=(char*)visRegion->getWriteBuffer(avail);
	ASSERTRETVAL(packet!=NULL,"could not get buffer",false);
	
	unsigned int used;
	used=LoadSave::encode("TekkotsuImage",buf,avail);
	ASSERTRETVAL(used!=0,"ran out of space",false);
	avail-=used; buf+=used;
	used=LoadSave::encode(Config::vision_config::ENCODE_SINGLE_CHANNEL,buf,avail);
	ASSERTRETVAL(used!=0,"ran out of space",false);
	avail-=used; buf+=used;
	used=LoadSave::encode(Config::vision_config::COMPRESS_RLE,buf,avail);
	ASSERTRETVAL(used!=0,"ran out of space",false);
	avail-=used; buf+=used;

	used=LoadSave::encode(fbkgen.getWidth(layer),buf,avail);
	ASSERTRETVAL(used!=0,"ran out of space",false);
	avail-=used; buf+=used;
	used=LoadSave::encode(fbkgen.getHeight(layer),buf,avail);
	ASSERTRETVAL(used!=0,"ran out of space",false);
	avail-=used; buf+=used;
	used=LoadSave::encode(time,buf,avail);
	ASSERTRETVAL(used!=0,"ran out of space",false);
	avail-=used; buf+=used;
	used=LoadSave::encode(fbkgen.getFrameNumber(),buf,avail);
	ASSERTRETVAL(used!=0,"ran out of space",false);
	avail-=used; buf+=used;

	cur=buf;
	return true;
}

bool
RegionCamBehavior::writeRegions(const FilterBankEvent& e) {
	FilterBankGenerator& fbkgen=*e.getSource();

	unsigned int layer=fbkgen.getNumLayers()-1-config->vision.regioncam_skip;
	unsigned int used=0;
	openPacket(fbkgen,e.getTimeStamp(),layer);
	ASSERTRETVAL(cur!=NULL,"header failed",false);
	
	RegionGenerator * regGen = dynamic_cast<RegionGenerator*>(&fbkgen);
	ASSERTRETVAL(regGen!=NULL,"fbkgen isn't an RegionGenerator",false);

	regGen->selectSaveImage(layer,config->vision.rlecam_channel);
	used=regGen->SaveBuffer(cur,avail);
	ASSERTRETVAL(used!=0,"save region image failed",false);
	avail-=used; cur+=used;
	
	const SegmentedColorGenerator * seg = dynamic_cast<const SegmentedColorGenerator*>((regGen->getSourceGenerator())->getSourceGenerator()); //Get the source of his source (the SegmentedColorGenerator)
	ASSERTRETVAL(seg!=NULL,"The source of RegionGenerator's source is not a SegmentedColorGenerator - how do i know what the colors are?",false);
	if(0==(used=seg->encodeColors(cur,avail))) return false;
	avail-=used; cur+=used;

	closePacket();

	return true;
}


void
RegionCamBehavior::closePacket() {
	if(packet==NULL)
		return;
	visRegion->write(cur-packet);
	packet=cur=NULL;
	avail=0;
}


/*! @file
 * @brief Implements RegionCamBehavior, which forwards the regions from RegionGenerator over wireless
 * @author harm & niels (Creators)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_4_1 $
 * $Revision: 1.1 $
 * $State: Exp $
 * $Date: 2005/08/05 19:44:22 $
 */

