#include "RawCamBehavior.h"
#include "Wireless/Wireless.h"
#include "Events/EventRouter.h"
#include "Vision/RawCameraGenerator.h"
#include "Vision/JPEGGenerator.h"
#include "Events/FilterBankEvent.h"
#include "Behaviors/Controller.h"
#include "Shared/ProjectInterface.h"

RawCamBehavior::RawCamBehavior()
	: BehaviorBase("RawCamBehavior"), visRaw(NULL), packet(NULL), cur(NULL), avail(0), max_buf(0), lastProcessedTime(0)
{}

void
RawCamBehavior::DoStart() {
	BehaviorBase::DoStart();
	std::vector<std::string> args;
	args.push_back("raw");
	char port[50];
	snprintf(port,50,"%d",config->vision.rawcam_port);
	args.push_back(port);
	if(config->vision.rawcam_transport==0) {
		max_buf=UDP_WIRELESS_BUFFER_SIZE;
		visRaw=wireless->socket(SocketNS::SOCK_DGRAM, 1024, max_buf);
		args.push_back("udp");
	} else if(config->vision.rawcam_transport==1) {
		max_buf=TCP_WIRELESS_BUFFER_SIZE;
		visRaw=wireless->socket(SocketNS::SOCK_STREAM, 1024, max_buf);
		wireless->setDaemon(visRaw,true);
		args.push_back("tcp");
	} else {
		serr->printf("ERROR: Invalid Config::vision.rawcam_transport: %d\n",config->vision.rawcam_transport);
		return;
	}
	wireless->listen(visRaw,config->vision.rawcam_port);
	
	Controller::loadGUI("org.tekkotsu.mon.VisionGUI","RawVisionGUI",config->vision.rawcam_port,args);

	erouter->addListener(this,EventBase::visRawCameraEGID,ProjectInterface::visRawCameraSID,EventBase::deactivateETID);
	erouter->addListener(this,EventBase::visJPEGEGID,ProjectInterface::visColorJPEGSID,EventBase::deactivateETID);
	erouter->addListener(this,EventBase::visJPEGEGID,ProjectInterface::visGrayscaleJPEGSID,EventBase::deactivateETID);
}

void
RawCamBehavior::DoStop() {
	erouter->removeListener(this);
	if(wireless->isConnected(visRaw->sock))
		sendCloseConnectionPacket();
	Controller::closeGUI("RawVisionGUI");

	// this could be considered a bug in our wireless - if we don't setDaemon(...,false)
	// it will try to listen again even though we explicitly closed the server socket...
	wireless->setDaemon(visRaw,false);
	wireless->close(visRaw->sock);
	BehaviorBase::DoStop();
}

void
RawCamBehavior::processEvent(const EventBase& e) {
	if(!wireless->isConnected(visRaw->sock))
		return;
	const FilterBankEvent* fbke=dynamic_cast<const FilterBankEvent*>(&e);
	ASSERTRET(fbke!=NULL,"unexpected event");
	if ((get_time() - lastProcessedTime) < config->vision.rawcam_interval) {// not enough time has gone by
	  return;
	}
	/* // turning these off enables individual channel compression
		if(config->vision.rawcam_compression==Config::vision_config::COMPRESS_NONE && e.getGeneratorID()!=EventBase::visRawCameraEGID)
		return;
		if(config->vision.rawcam_compression==Config::vision_config::COMPRESS_JPEG && e.getGeneratorID()!=EventBase::visJPEGEGID)
		return; */
	if(config->vision.rawcam_encoding==Config::vision_config::ENCODE_COLOR) {
		bool succ=writeColor(*fbke);
		ASSERTRET(succ,"serialization failed");
	} else if(config->vision.rawcam_encoding==Config::vision_config::ENCODE_SINGLE_CHANNEL) {
		bool succ=writeSingleChannel(*fbke);
		ASSERTRET(succ,"serialization failed");
	} else {
		serr->printf("%s: Bad rawcam_encoding setting\n",getName().c_str());
	}
}

unsigned int RawCamBehavior::getSourceLayer(unsigned int chan, unsigned int numLayers) {
	if(config->vision.rawcam_encoding==Config::vision_config::ENCODE_SINGLE_CHANNEL) {
		if(config->vision.rawcam_channel!=(int)chan)
			return -1U;
		return numLayers-1-config->vision.rawcam_y_skip;
	}
	// must be full-color
	switch(chan) {
	case RawCameraGenerator::CHAN_Y:
		if(config->vision.rawcam_compression==Config::vision_config::COMPRESS_JPEG) {
			if(config->vision.rawcam_y_skip-config->vision.rawcam_uv_skip == 1)
				return numLayers-1-config->vision.rawcam_uv_skip;
		}
		return numLayers-1-config->vision.rawcam_y_skip;
	case RawCameraGenerator::CHAN_U:
	case RawCameraGenerator::CHAN_V:
		if(config->vision.rawcam_compression==Config::vision_config::COMPRESS_JPEG) {
			if(config->vision.rawcam_uv_skip-config->vision.rawcam_y_skip == 1)
				return numLayers-1-config->vision.rawcam_y_skip;
		}
		return numLayers-1-config->vision.rawcam_uv_skip;
	default: // other channels, i.e. Y-derivatives
		return -1U;
	}
}
unsigned int RawCamBehavior::getSourceYLayer(unsigned int numLayers) {
	return getSourceLayer(RawCameraGenerator::CHAN_Y,numLayers);
}
unsigned int RawCamBehavior::getSourceULayer(unsigned int numLayers) {
	return getSourceLayer(RawCameraGenerator::CHAN_U,numLayers);
}
unsigned int RawCamBehavior::getSourceVLayer(unsigned int numLayers) {
	return getSourceLayer(RawCameraGenerator::CHAN_V,numLayers);
}

bool
RawCamBehavior::openPacket(FilterBankGenerator& fbkgen, unsigned int time, unsigned int layer) {
	if(packet!=NULL)
		return false;

	avail=max_buf-1; //not sure why -1, but Alok had it, so i will too
	ASSERT(cur==NULL,"cur non-NULL");
	cur=NULL;
	char * buf=packet=(char*)visRaw->getWriteBuffer(avail);
	ASSERTRETVAL(packet!=NULL,"could not get buffer",false);
	
	unsigned int used;
	used=LoadSave::encode("TekkotsuImage",buf,avail);
	ASSERTRETVAL(used!=0,"ran out of space",false);
	avail-=used; buf+=used;
	used=LoadSave::encode(config->vision.rawcam_encoding,buf,avail);
	ASSERTRETVAL(used!=0,"ran out of space",false);
	avail-=used; buf+=used;
	used=LoadSave::encode(config->vision.rawcam_compression,buf,avail);
	ASSERTRETVAL(used!=0,"ran out of space",false);
	avail-=used; buf+=used;

	used=LoadSave::encode(fbkgen.getWidth(layer),buf,avail);
	ASSERTRETVAL(used!=0,"ran out of space",false);
	avail-=used; buf+=used;
	used=LoadSave::encode(fbkgen.getHeight(layer),buf,avail);
	ASSERTRETVAL(used!=0,"ran out of space",false);
	avail-=used; buf+=used;
	used=LoadSave::encode(time,buf,avail);
	ASSERTRETVAL(used!=0,"ran out of space",false);
	avail-=used; buf+=used;
	used=LoadSave::encode(fbkgen.getFrameNumber(),buf,avail);
	ASSERTRETVAL(used!=0,"ran out of space",false);
	avail-=used; buf+=used;

	cur=buf;
	return true;
}

bool
RawCamBehavior::writeColor(const FilterBankEvent& e) {
	FilterBankGenerator& fbkgen=*e.getSource();

	unsigned int y_layer=fbkgen.getNumLayers()-1-config->vision.rawcam_y_skip;
	unsigned int uv_layer=fbkgen.getNumLayers()-1-config->vision.rawcam_uv_skip;

	if(config->vision.rawcam_channel==-1) {
		if(e.getGeneratorID()==EventBase::visRawCameraEGID && config->vision.rawcam_compression==Config::vision_config::COMPRESS_NONE
			 || e.getGeneratorID()==EventBase::visJPEGEGID && config->vision.rawcam_compression==Config::vision_config::COMPRESS_JPEG) {
			if(const JPEGGenerator* jgen=dynamic_cast<const JPEGGenerator*>(&fbkgen))
				if(jgen->getCurrentSourceFormat()==JPEGGenerator::SRC_COLOR)
					return true;
			unsigned int used=0;
			openPacket(fbkgen,e.getTimeStamp(),uv_layer);
			ASSERTRETVAL(cur!=NULL,"header failed",false);
			
			used=LoadSave::encode("FbkImage",cur,avail);
			ASSERTRETVAL(used!=0,"save blank image failed",false);
			avail-=used; cur+=used;
			used=LoadSave::encode(0,cur,avail);
			ASSERTRETVAL(used!=0,"save blank image failed",false);
			avail-=used; cur+=used;
			used=LoadSave::encode(0,cur,avail);
			ASSERTRETVAL(used!=0,"save blank image failed",false);
			avail-=used; cur+=used;
			used=LoadSave::encode(-1,cur,avail);
			ASSERTRETVAL(used!=0,"save blank image failed",false);
			avail-=used; cur+=used;
			used=LoadSave::encode(RawCameraGenerator::CHAN_Y,cur,avail);
			ASSERTRETVAL(used!=0,"save blank image failed",false);
			avail-=used; cur+=used;
			used=LoadSave::encode("blank",cur,avail);
			ASSERTRETVAL(used!=0,"save blank image failed",false);
			avail-=used; cur+=used;

			fbkgen.selectSaveImage(uv_layer,RawCameraGenerator::CHAN_U);
			used=fbkgen.SaveBuffer(cur,avail);
			ASSERTRETVAL(used!=0,"save image failed",false);
			avail-=used; cur+=used;
			
			fbkgen.selectSaveImage(uv_layer,RawCameraGenerator::CHAN_V);
			used=fbkgen.SaveBuffer(cur,avail);
			ASSERTRETVAL(used!=0,"save image failed",false);
			avail-=used; cur+=used;

			closePacket();
		}
		return true;
	}

	unsigned int big_layer=y_layer;
	unsigned int small_layer=uv_layer;
	if(y_layer<uv_layer) { 
		big_layer=uv_layer;
		small_layer=y_layer;
	}
	if(const JPEGGenerator* jgen=dynamic_cast<const JPEGGenerator*>(&fbkgen)) {
		if(config->vision.rawcam_compression!=Config::vision_config::COMPRESS_JPEG)
			return true;
		if(jgen->getCurrentSourceFormat()==JPEGGenerator::SRC_COLOR && big_layer-small_layer<2) {
			unsigned int used=0;
			openPacket(fbkgen,e.getTimeStamp(),big_layer);
			ASSERTRETVAL(cur!=NULL,"header failed",false);
			
			fbkgen.selectSaveImage(big_layer,0);
			used=fbkgen.SaveBuffer(cur,avail);
			ASSERTRETVAL(used!=0,"save image failed",false);
			avail-=used; cur+=used;
			
			closePacket();
		} else if(jgen->getCurrentSourceFormat()==JPEGGenerator::SRC_GRAYSCALE && big_layer-small_layer>=2) {
			unsigned int used=0;
			bool opened=openPacket(fbkgen,e.getTimeStamp(),big_layer);
			ASSERTRETVAL(cur!=NULL,"header failed",false);
			
			if(big_layer==y_layer) {
				fbkgen.selectSaveImage(y_layer,RawCameraGenerator::CHAN_Y);
				used=fbkgen.SaveBuffer(cur,avail);
				ASSERTRETVAL(used!=0,"save image failed",false);
				avail-=used; cur+=used;
			} else {
				fbkgen.selectSaveImage(uv_layer,RawCameraGenerator::CHAN_U);
				used=fbkgen.SaveBuffer(cur,avail);
				ASSERTRETVAL(used!=0,"save image failed",false);
				avail-=used; cur+=used;
			
				fbkgen.selectSaveImage(uv_layer,RawCameraGenerator::CHAN_V);
				used=fbkgen.SaveBuffer(cur,avail);
				ASSERTRETVAL(used!=0,"save image failed",false);
				avail-=used; cur+=used;
			}
			
			if(!opened)
				closePacket();
		}
	} else {
		unsigned int used=0;
		bool opened=false;
		
		if(config->vision.rawcam_compression==Config::vision_config::COMPRESS_NONE || big_layer-small_layer>=2 && big_layer==uv_layer) {
			opened=openPacket(fbkgen,e.getTimeStamp(),big_layer);
			ASSERTRETVAL(cur!=NULL,"header failed",false);
			
			fbkgen.selectSaveImage(y_layer,RawCameraGenerator::CHAN_Y);
			used=fbkgen.SaveBuffer(cur,avail);
			ASSERTRETVAL(used!=0,"save image failed",false);
			avail-=used; cur+=used;
		}
		
		if(config->vision.rawcam_compression==Config::vision_config::COMPRESS_NONE || big_layer-small_layer>=2 && big_layer==y_layer) {
			opened=openPacket(fbkgen,e.getTimeStamp(),big_layer);
			ASSERTRETVAL(cur!=NULL,"header failed",false);
			
			fbkgen.selectSaveImage(uv_layer,RawCameraGenerator::CHAN_U);
			used=fbkgen.SaveBuffer(cur,avail);
			ASSERTRETVAL(used!=0,"save image failed",false);
			avail-=used; cur+=used;
			
			fbkgen.selectSaveImage(uv_layer,RawCameraGenerator::CHAN_V);
			used=fbkgen.SaveBuffer(cur,avail);
			ASSERTRETVAL(used!=0,"save image failed",false);
			avail-=used; cur+=used;
		}
		
		if(config->vision.rawcam_compression==Config::vision_config::COMPRESS_NONE || !opened)
			closePacket();
	}
	
	return true;
}

bool
RawCamBehavior::writeSingleChannel(const FilterBankEvent& e) {
	FilterBankGenerator& fbkgen=*e.getSource();
	if( config->vision.rawcam_compression==Config::vision_config::COMPRESS_NONE && e.getGeneratorID()==EventBase::visRawCameraEGID
			|| config->vision.rawcam_compression==Config::vision_config::COMPRESS_JPEG && e.getGeneratorID()==EventBase::visJPEGEGID )
		{
			if(const JPEGGenerator * jgen=dynamic_cast<const JPEGGenerator*>(&fbkgen))
				if(jgen->getCurrentSourceFormat()!=JPEGGenerator::SRC_GRAYSCALE)
					return true;
			unsigned int layer=fbkgen.getNumLayers()-1-config->vision.rawcam_y_skip;
			
			unsigned int used=0;
			openPacket(fbkgen,e.getTimeStamp(),layer);
			ASSERTRETVAL(cur!=NULL,"header failed",false);
			
			fbkgen.selectSaveImage(layer,config->vision.rawcam_channel);
			used=fbkgen.SaveBuffer(cur,avail);
			ASSERTRETVAL(used!=0,"save image failed",false);
			avail-=used; cur+=used;
			
			closePacket();
		}
	return true;
}

//#include "Shared/WorldState.h"

void
RawCamBehavior::closePacket() {
	if(packet==NULL)
		return;
	visRaw->write(cur-packet);
	/*	cout << "used=" << (cur-packet) << "\tavail==" << avail;
	if(state->robotDesign & WorldState::ERS7Mask)
		cout << "\tmax bandwidth=" << (cur-packet)/1024.f*30 << "KB/sec" << endl;
	else
		cout << "\tmax bandwidth=" << (cur-packet)/1024.f*24 << "KB/sec" << endl;
	*/
	packet=cur=NULL;
	avail=0;
	lastProcessedTime = get_time();
}

bool
RawCamBehavior::sendCloseConnectionPacket() {
	char msg[]="CloseConnection";
	unsigned int len=strlen(msg)+LoadSave::stringpad;
	char * buf = (char*)visRaw->getWriteBuffer(len);
	ASSERTRETVAL(buf!=NULL,"Could not get buffer for closing packet",false);
	unsigned int used=LoadSave::encode(msg,buf,len);
	ASSERTRETVAL(used!=0,"Could not write close packet",false);
	visRaw->write(used);
	return true;
}


/*! @file
 * @brief Implements RawCamBehavior, which forwards images from camera over wireless
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_4_1 $
 * $Revision: 1.21 $
 * $State: Exp $
 * $Date: 2005/07/29 18:35:05 $
 */

