#include "EventLogger.h"
#include "Events/EventRouter.h"
#include "Motion/MMAccessor.h"
#include "Motion/LedMC.h"
#include "ValueEditControl.h"
#include "StringInputControl.h"
#include "NullControl.h"
#include <sstream>
#include "Sound/SoundManager.h"
#include "Vision/FilterBankGenerator.h"
#include "Shared/Base64.h"

#include <libxml/xmlmemory.h>
#include <libxml/parser.h>

Socket* EventLogger::logSocket=NULL;
unsigned int EventLogger::logSocketRefCount=0;
int EventLogger::port=10080;

EventLogger::EventLogger() : ControlBase("Event Logger","Allows you to see/log all of the un-trapped events as they are generated"), logfilePath(), logfile(), verbosity(0) {
	for(unsigned int i=0; i<EventBase::numEGIDs; i++) {
		std::string tmp=EventBase::EventGeneratorNames[i];
		pushSlot(new NullControl(("[ ] "+tmp).c_str(),"Show/hide events from "+tmp));
	}
	pushSlot(NULL);
	pushSlot(new ValueEditControl<unsigned int>("Verbosity","Controls verbosity level: 0=(gen,source,type); 1=0+gen_id,source_id,type_id; 2=1+duration,timestamp; 3=2+magnitude; additional columns may be added for subclass info","Please enter a new verbosity level: 0=(gen,source,type); 1=0+gen_id,source_id,type_id; 2=1+duration,timestamp; 3=2+magnitude; additional columns may be added for subclass info",&verbosity));
	pushSlot(new ControlBase("[X] Console Output","If selected, outputs events to the console"));
	pushSlot(new StringInputControl("[ ] File Output","Please enter the filename to log to (in /ms/...)"));
	if(logSocket==NULL) {
		ASSERT(logSocketRefCount==0,"logSocket is NULL, ref count is non-zero");
		logSocket=wireless->socket(SocketNS::SOCK_STREAM,1024,1<<15);
		wireless->setDaemon(logSocket);
		wireless->listen(logSocket,port);
	}
	logSocketRefCount++;
}

EventLogger::~EventLogger() {
	clearSlots();
	if(--logSocketRefCount==0) {
		wireless->setDaemon(logSocket,false);
		wireless->close(logSocket);
		logSocket=NULL;
	}
}

ControlBase* EventLogger::doSelect() {
	ControlBase* ans=this;
	for(unsigned int i=0; i<hilights.size(); i++) {
		unsigned int cur=hilights[i];
		if(cur<EventBase::numEGIDs) {
			if(options[cur]->getName()[1]!=' ') {
				erouter->removeListener(this,(EventBase::EventGeneratorID_t)(cur));
				setStatus(cur,' ');
			} else {
				erouter->addListener(this,(EventBase::EventGeneratorID_t)(cur));
				setStatus(cur,'X');
			}
		} else if(cur==EventBase::numEGIDs+1) {
			ans=options[cur];
		} else if(cur==EventBase::numEGIDs+2) {
			if(options[cur]->getName()[1]!=' ') {
				setStatus(cur,' ');
			} else {
				setStatus(cur,'X');
			}
		} else if(cur==EventBase::numEGIDs+3) {
			if(options[cur]->getName()[1]!=' ') {
				logfile.close();
				options[cur]->setName("[ ] File Output");
			} else {
				ans=options[cur];
			}
		}
		sndman->PlayFile(config->controller.select_snd);
	}
	if(ans==this)
		refresh();
	return ans;
}

void EventLogger::refresh() {
	checkLogFile();
	ControlBase::refresh();
}

//!sends all events received to stdout and/or logfile
void EventLogger::processEvent(const EventBase& event) {
	std::string logdata = event.getDescription(true,verbosity);
	if(options[EventBase::numEGIDs+2]->getName()[1]=='X')
		sout->printf("EVENT: %s\n",logdata.c_str());
	if(logSocket!=NULL && wireless->isConnected(logSocket->sock)) {
		xmlDoc * doc = xmlNewDoc((const xmlChar*)"1.0");
		xmlNode * cur = xmlNewNode(NULL,(const xmlChar*)"");
		xmlSetProp(cur,(const xmlChar*)"type",(const xmlChar*)"log");
		xmlNode * desc = xmlNewNode(NULL,(const xmlChar*)"param");
		event.SaveXML(cur);
		xmlAddChild(cur,desc);
		xmlSetProp(desc,(const xmlChar*)"name",(const xmlChar*)"description");
		xmlSetProp(desc,(const xmlChar*)"value",(const xmlChar*)event.getDescription(true,3).c_str());
		xmlBuffer* buf=xmlBufferCreate();
		int n=xmlNodeDump(buf,doc,cur,0,1);
		xmlFreeDoc(doc);
		byte * nbuf = logSocket->getWriteBuffer(n+1);
		if(nbuf!=NULL) {
			memcpy(nbuf,xmlBufferContent(buf),n);
			nbuf[n]='\n';
			logSocket->write(n+1);
		}
		xmlBufferFree(buf);
	}
	checkLogFile();
	if(logfile)
		logfile << logdata << endl;
}

void EventLogger::logImage(FilterBankGenerator& fbg, unsigned int layer, unsigned int channel, const BehaviorBase* source/*=NULL*/) {
	if(logSocket!=NULL && wireless->isConnected(logSocket->sock)) {
		fbg.selectSaveImage(layer,channel);
		unsigned int len=fbg.getBinSize();
		char * binbuf=new char[len];
		fbg.SaveBuffer(binbuf,len);
		string b64buf=base64::encode(binbuf,len);
		
		xmlDoc * doc = xmlNewDoc((const xmlChar*)"1.0");
		xmlNode * cur = xmlNewNode(NULL,(const xmlChar*)"event");
		xmlSetProp(cur,(const xmlChar*)"type",(const xmlChar*)"image");
		if(source!=NULL)
			xmlSetProp(cur,(const xmlChar*)"sid",(const xmlChar*)source->getName().c_str());
		snprintf(binbuf,len,"%d",get_time());
		xmlSetProp(cur,(const xmlChar*)"time",(const xmlChar*)binbuf);
		delete [] binbuf;
		xmlNodeSetContent(cur,(const xmlChar*)b64buf.c_str());
		xmlBuffer* buf=xmlBufferCreate();
		int n=xmlNodeDump(buf,doc,cur,0,1);
		xmlFreeDoc(doc);
		byte * nbuf = logSocket->getWriteBuffer(n+1);
		if(nbuf!=NULL) {
			memcpy(nbuf,xmlBufferContent(buf),n);
			nbuf[n]='\n';
			logSocket->write(n+1);
		}
		xmlBufferFree(buf);
	}		
}

void EventLogger::logMessage(std::string msg, const BehaviorBase* source/*=NULL*/, const char* icon/*=NULL*/, unsigned int placement/*=0*/) {
	if(logSocket!=NULL && wireless->isConnected(logSocket->sock)) {
		xmlDoc * doc = xmlNewDoc((const xmlChar*)"1.0");
		xmlNode * cur = xmlNewNode(NULL,(const xmlChar*)"event");
		xmlSetProp(cur,(const xmlChar*)"type",(const xmlChar*)"userlog");
		if(source!=NULL)
			xmlSetProp(cur,(const xmlChar*)"sid",(const xmlChar*)source->getName().c_str());
		if(icon!=NULL)
			xmlSetProp(cur,(const xmlChar*)"icon",(const xmlChar*)icon);
		const unsigned int len=20;
		char sbuf[len];
		snprintf(sbuf,len,"%d",placement);
		xmlSetProp(cur,(const xmlChar*)"voff",(const xmlChar*)sbuf);
		snprintf(sbuf,len,"%d",get_time());
		xmlSetProp(cur,(const xmlChar*)"time",(const xmlChar*)sbuf);
		xmlNodeSetContent(cur,(const xmlChar*)msg.c_str());
		xmlBuffer* buf=xmlBufferCreate();
		int n=xmlNodeDump(buf,doc,cur,0,1);
		xmlFreeDoc(doc);
		byte * nbuf = logSocket->getWriteBuffer(n+1);
		if(nbuf!=NULL) {
			memcpy(nbuf,xmlBufferContent(buf),n);
			nbuf[n]='\n';
			logSocket->write(n+1);
		}
		xmlBufferFree(buf);
	}		
}

void EventLogger::logWebcam(const BehaviorBase* source/*=NULL*/) {
	if(logSocket!=NULL && wireless->isConnected(logSocket->sock)) {
		xmlDoc * doc = xmlNewDoc((const xmlChar*)"1.0");
		xmlNode * cur = xmlNewNode(NULL,(const xmlChar*)"event");
		xmlSetProp(cur,(const xmlChar*)"type",(const xmlChar*)"webcam");
		if(source!=NULL)
			xmlSetProp(cur,(const xmlChar*)"sid",(const xmlChar*)source->getName().c_str());
		const unsigned int len=20;
		char sbuf[len];
		snprintf(sbuf,len,"%d",get_time());
		xmlSetProp(cur,(const xmlChar*)"time",(const xmlChar*)sbuf);
		xmlNodeSetContent(cur,(const xmlChar*)" ");
		xmlBuffer* buf=xmlBufferCreate();
		int n=xmlNodeDump(buf,doc,cur,0,1);
		xmlFreeDoc(doc);
		byte * nbuf = logSocket->getWriteBuffer(n+1);
		if(nbuf!=NULL) {
			memcpy(nbuf,xmlBufferContent(buf),n);
			nbuf[n]='\n';
			logSocket->write(n+1);
		}
		xmlBufferFree(buf);
	}		
}

void EventLogger::clearSlots() {
	erouter->removeListener(this);
	ControlBase::clearSlots();
}

void EventLogger::setStatus(unsigned int i, char c) {
	std::string tmp=options[i]->getName();
	tmp[1]=c;
	options[i]->setName(tmp);
}

void EventLogger::checkLogFile() {
	unsigned int cur=EventBase::numEGIDs+3;
	StringInputControl * strin=dynamic_cast<StringInputControl*>(options[cur]);
	ASSERTRET(strin!=NULL,"The StringInputControl is misplaced");
	if(strin->getLastInput()!=logfilePath) {
		logfile.close();
		logfilePath=strin->getLastInput();
		logfile.clear();
		if(logfilePath.size()!=0) {
			sout->printf("Opening `%s'\n",(config->portPath(logfilePath)).c_str());
			logfile.open((config->portPath(logfilePath)).c_str());
			if(!logfile.fail()) {
				setStatus(cur,'X');
				strin->setName(strin->getName()+": "+logfilePath);
			} else {
				serr->printf("Opening `%s' failed\n",(config->portPath(logfilePath)).c_str());
			}
		}
	}
}

/*! @file
 * @brief Describes EventLogger, which allows logging of events to the console or a file
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_4_1 $
 * $Revision: 1.18 $
 * $State: Exp $
 * $Date: 2005/08/07 04:11:03 $
 */
