//-*-c++-*-
#ifndef INCLUDED_MotionManager_h
#define INCLUDED_MotionManager_h

#include "MotionCommand.h"
#include "OutputCmd.h"
#include "OutputPID.h"
#include "Shared/RobotInfo.h"
#include "Shared/ListMemBuf.h"
#include "Shared/MutexLock.h"

#ifdef PLATFORM_APERIOS
#include "Shared/SharedObject.h"
#include "MotionManagerMsg.h"
#endif

#ifdef PLATFORM_APERIOS
#include <OPENR/OPENR.h>
#include <OPENR/OPENRAPI.h>
#include <OPENR/OSubject.h>
#include <OPENR/ObjcommEvent.h>
#include <OPENR/OObject.h>
#endif

//! The purpose of this class is to provide mutually exclusive access to the MotionCommands and simplify their sharing between memory spaces
/*! Since MotionObject and MainObject run as separate processes, they
 *  could potentially try to access the same motion command at the
 *  same time, leading to unpredictable behavior.  The MotionManager
 *  enforces a set of locks to solve this.
 *
 *  The other problem is that we are sharing between processes.
 *  MotionManager will do what's necessary to distribute new
 *  MotionCommand's to all the processes (currently just MainObj and
 *  MotoObj)\n You should be able to create and add a new motion in
 *  one line:
 *
 *  @code
 *  motman->addMotion( SharedObject<YourMC>([arg1,...]) , autoprune [, priority ]);
 *  @endcode
 *  
 *  But if you want to do some more initializations not handled by the
 *  constructor (the @p arg1, @p arg2, ...  params) then you would
 *  want to do something like the following:
 *  
 *  @code
 *  SharedObject<YourMC> tmpvar([arg1,[arg2,...]]);
 *  tmpvar->cmd1();
 *  tmpvar->cmd2();
 *  //...
 *  motman->addPrunableMotion(tmpvar [, ...]); //or addPersistentMotion(...)
 *  @endcode
 *
 *  Notice that tmpvar is of type SharedObject, but you're calling @c
 *  YourMC functions on it...  SharedObject is a "smart pointer" which
 *  will pass your function calls on to the underlying templated type.
 *  Isn't C++ great? :)
 *
 *  @warning Once the MotionCommand has been added you must check it
 *  out to modify it or risk concurrent access problems.
 *
 *  @see MotionCommand for information on creating new motion primitives.
 *
 *  @see MMAccessor for information on accessing motions after you've
 *  added them to MotionManager.
 */
class MotionManager {
public:
	//! This is the number of processes which will be accessing the MotionManager
	/*! Probably just MainObject and MotionObject... This isn't really a
	 *  hard maximum, but should be actual expected, need to know when
	 *  they're all connected */
	static const unsigned int MAX_ACCESS=2;

	static const unsigned int MAX_MOTIONS=64;   //!< This is the maximum number of Motions which can be managed, can probably be increased reasonably without trouble

	typedef MotionManagerMsg::MC_ID MC_ID;      //!< use this type when referring to the ID numbers that MotionManager hands out
	static const MC_ID invalid_MC_ID=MotionManagerMsg::invalid_MC_ID; //!< for errors and undefined stuff

	//!Just to give you some guidelines for what values to use for different priority levels, but you can pick any value you like (that's why they are floats)
	//!@name Priority Level Constants
	static const float kIgnoredPriority;    //!< won't be expressed, handy if you want to temporarily pause something
	static const float kBackgroundPriority; //!< will only be expressed if *nothing* else is using that joint
	static const float kLowPriority;        //!< for stuff that's not background but lower than standard
	static const float kStdPriority;        //!< for every-day commands
	static const float kHighPriority;       //!< for stuff that should override standard stuff
	static const float kEmergencyPriority;  //!< for really important stuff, such as the emergency stop
	//@}

	MotionManager();                            //!< Constructor, sets all the outputs to 0

#ifdef PLATFORM_APERIOS
	void InitAccess(OSubject* subj);            //!< @b LOCKS @b MotionManager Everyone who is planning to use the MotionManager needs to call this before they access it or suffer a horrible fate
	void receivedMsg(const ONotifyEvent& event); //!< @b LOCKS @b MotionManager This gets called by an OObject when it receives a message from one of the other OObject's MotionManagerComm Subject
#endif

	//!@name MotionCommand Safe
	void setOutput(const MotionCommand* caller, unsigned int output, const OutputCmd& cmd); //!< @b LOCKS @b MotionManager Requests a value be set for the specified output, copies cmd across frames
	void setOutput(const MotionCommand* caller, unsigned int output, const OutputCmd& cmd, unsigned int frame); //!< @b LOCKS @b MotionManager Requests a value be set for the specified output in the specified frame
	void setOutput(const MotionCommand* caller, unsigned int output, const OutputCmd cmd[NumFrames]); //!< @b LOCKS @b MotionManager Requests a value be set for the specified output across frames
	void setOutput(const MotionCommand* caller, unsigned int output, const OutputPID& pid); //!< @b LOCKS @b MotionManager Requests a PID be set for the specified output, notice that this might be overruled by a higher priority motion
	void setOutput(const MotionCommand* caller, unsigned int output, const OutputCmd& cmd, const OutputPID& pid); //!< @b LOCKS @b MotionManager Requests a value and PID be set for the specified output
	void setOutput(const MotionCommand* caller, unsigned int output, const OutputCmd cmd[NumFrames], const OutputPID& pid); //!< @b LOCKS @b MotionManager Requests a value and PID be set for the specified output
	const OutputCmd& getOutputCmd(unsigned int output) const { return cmds[output]; } //!< Returns the value of the output last sent to the OS.  Note that this will differ from the sensed value in state, even when staying still.  There is no corresponding getOutputPID because this value *will* duplicate the value in state.
	void setPriority(MC_ID mcid, float p) { cmdlist[mcid].priority=p; }//!< sets the priority level of a MotionCommand
	float getPriority(MC_ID mcid) const { return cmdlist[mcid].priority; } //!< returns priority level of a MotionCommand
	//@}

	//@{
	inline MC_ID begin() const         { return skip_ahead(cmdlist.begin()); }   //!< returns the MC_ID of the first MotionCommand
	inline MC_ID next(MC_ID cur) const { return skip_ahead(cmdlist.next(cur)); } //!< returns the MC_ID of MotionCommand following the one that is passed
	inline MC_ID end() const           { return cmdlist.end();      } //!< returns the MC_ID of the one-past-the-end MotionCommand (like the STL)
	inline unsigned int size() const   { return cmdlist.size();     } //!< returns the number of MotionCommands being managed
	//@}

	//!You can have one MC check out and modify another, but make sure the other MC doesn't call setOutput()
	//!@name MotionCommand "Risky"
	MotionCommand * checkoutMotion(MC_ID mcid,bool block=true); //!< locks the command and possibly performs RTTI conversion; supports recursive calls
	void checkinMotion(MC_ID mcid); //!< marks a MotionCommand as unused
	MotionCommand * peekMotion(MC_ID mcid) { return mcid==invalid_MC_ID?NULL:cmdlist[mcid].baseaddrs[_MMaccID]; } //!< allows access to a MotionCommand without checking it out; warning @b never call a function based on this, only access member fields through it
	unsigned int checkoutLevel(MC_ID mcid) { return mcid==invalid_MC_ID?0:cmdlist[mcid].lock.get_lock_level(); } //!< returns the number of times @a mcid has been checked out minus the times it's been checked in
	bool isOwner(MC_ID mcid) { return mcid==invalid_MC_ID?false:(cmdlist[mcid].lock.owner()==_MMaccID); }
	//@}

	//!@name MotionCommand Unsafe
	//@{
#ifdef PLATFORM_APERIOS
	//! @b LOCKS @b MotionManager adds a new motion (wrapped in a SharedObject) and marks that it should be automatically deleted when the MotionCommand::isAlive() returns false.
	MC_ID addPrunableMotion(const SharedObjectBase& sm, float priority=kStdPriority) { return doAddMotion(sm,true,priority); }
	//! @b LOCKS @b MotionManager adds a new motion (wrapped in a SharedObject) and marks that it should @e not be deleted, until removeMotion(MC_ID mcid) is called.
	MC_ID addPersistentMotion(const SharedObjectBase& sm, float priority=kStdPriority) { return doAddMotion(sm,false,priority); }
#endif //PLATFORM_APERIOS
	void removeMotion(MC_ID mcid); //!< @b LOCKS @b MotionManager removes the specified MotionCommand
	//@}

	//!@name Deprecated
#ifdef PLATFORM_APERIOS
	MC_ID addMotion(const SharedObjectBase& sm) __attribute__((deprecated)); //!< deprecated, we're recommending users call either addPrunableMotion() or addPersistentMotion() so there are no surprises
	MC_ID addMotion(const SharedObjectBase& sm, bool autoprune) __attribute__((deprecated)); //!< deprecated, we're recommending users call either addPrunableMotion() or addPersistentMotion() so there are no surprises
	MC_ID addMotion(const SharedObjectBase& sm, float priority) __attribute__((deprecated)); //!< deprecated, we're recommending users call either addPrunableMotion() or addPersistentMotion() so there are no surprises
	MC_ID addMotion(const SharedObjectBase& sm, float priority, bool autoprune) __attribute__((deprecated)); //!< deprecated, we're recommending users call either addPrunableMotion() or addPersistentMotion() so there are no surprises
#endif //PLATFORM_APERIOS
	//@}

	//@{
	void lock()    { MMlock.lock(_MMaccID); } //!< gets an exclusive lock on MotionManager - functions marked @b LOCKS @b MotionManager will cause (and require) this to happen automatically
	bool trylock() { return MMlock.try_lock(_MMaccID); } //!< tries to get a lock without blocking
	void release() { MMlock.release(); } //!< releases a lock on the motion manager
	//@}

	//@{
	void getOutputs(float outputs[NumFrames][NumOutputs]);  //!< @b LOCKS @b MotionManager called by MotionObject to fill in the output values for the next ::NumFrames frames (only MotoObj should call this...)
	void updateWorldState();                                //!< call this when you want MotionManager to set the WorldState to reflect what things should be for unsensed outputs (LEDs, ears) (only MotoObj should be calling this...)
#ifdef PLATFORM_APERIOS
	bool updatePIDs(OPrimitiveID primIDs[NumOutputs]);      //!< call this when you want MotionManager to update modified PID values, returns true if changes made (only MotoObj should be calling this...), see PIDMC for general PID documentation
#endif
	//@}

	//! holds the full requested value of an output
	class OutputState {
	public:
		//!@name Constructors
		//!Constructor
		OutputState();
		OutputState(unsigned int out, float pri, MC_ID mc, const OutputCmd cmds[NumFrames]);
		OutputState(unsigned int out, float pri, MC_ID mc, const OutputCmd& cmd);
		OutputState(unsigned int out, float pri, MC_ID mc, const OutputCmd& cmd, unsigned int frame);
		OutputState(unsigned int out, float pri, MC_ID mc, const OutputPID& p);
		OutputState(unsigned int out, float pri, MC_ID mc, const OutputCmd cmds[NumFrames], const OutputPID& p);
		OutputState(unsigned int out, float pri, MC_ID mc, const OutputCmd& cmd, const OutputPID& p);
		//@}
		float priority;             //!< priority level
		MC_ID mcid;                 //!< MC_ID of requester
		OutputCmd frames[NumFrames]; //!< values of output planned ahead
		OutputPID pid;               //!< pid of output
	};

protected:
#ifdef PLATFORM_APERIOS
	//!does the actual work of adding a motion
	MC_ID doAddMotion(const SharedObjectBase& sm, bool autoprune, float priority);
#endif
	
	//! used to request pids for a given joint
	struct PIDUpdate {
		//!constructor
	 	PIDUpdate() : joint((unsigned int)-1) {}
		//!constructor
		PIDUpdate(unsigned int j, const float p[3]) : joint(j) {
			for(unsigned int i=0; i<3; i++)
				pids[i]=p[i];
		}
		unsigned int joint; //!< the joint ID (see RobotInfo.h for offset values)
		float pids[3]; //!< the PID values to use (see ::Pid )
	};
	ListMemBuf<PIDUpdate,NumPIDJoints> pidchanges;  //!< stores PID updates, up to one per joint (if same is set more than once, it's just overwrites previous update)
	void setPID(unsigned int j, const float p[3]); //!< @b LOCKS @b MotionManager, called internally to do the work of setting the PID... you probably want to call setOutput with an OutputPID argument, not this...

	typedef unsigned short accID_t; //!< type to use to refer to accessors of MotionManager (or its locks)

	void func_begin() { MMlock.lock(_MMaccID); } //!< called at the begining of many functions to lock MotionManager
	void func_end() { MMlock.release(); } //!< called at the end of a function which called func_begin() to release it
	template<class T> T func_end(T val) { func_end(); return val; } //!< same as func_end(), except passes return value through

	MC_ID skip_ahead(MC_ID mcid) const; //!< during iteration, skips over motioncommands which are still in transit from on OObject to another
		
	//!All the information we need to maintain about a MotionCommand
	struct CommandEntry {
		//! Constructor, sets everything to basics
		CommandEntry() : lastAccessor((unsigned short)-1),lock(),priority(MotionManager::kStdPriority) {
			for(unsigned int i=0; i<MAX_ACCESS; i++) {
				baseaddrs[i]=NULL;
#ifdef PLATFORM_APERIOS
				rcr[i]=NULL;
#endif
			}
		}
		MotionCommand * baseaddrs[MAX_ACCESS]; //!< for each accessor, the base address of the motion command
#ifdef PLATFORM_APERIOS
		RCRegion * rcr[MAX_ACCESS];            //!< for each accessor the shared memory region that holds the motion command
#endif
		accID_t lastAccessor;                  //!< the ID of the last accessor to touch the command (which implies if it wants to touch this again, we don't have to convert again)
		MutexLock<MAX_ACCESS> lock;            //!< a lock to maintain mutual exclusion
		float priority;                        //!< MotionCommand's priority level
	private:
		CommandEntry(const CommandEntry&); //!< this shouldn't be called...
		CommandEntry& operator=(const CommandEntry&); //!< this shouldn't be called...
	};
	ListMemBuf<CommandEntry,MAX_MOTIONS,MC_ID> cmdlist;     //!< the list where MotionCommands are stored, remember, we're in a shared memory region with different base addresses - no pointers!
	MC_ID cur_cmd; //!< MC_ID of the MotionCommand currently being updated by getOutputs(), or NULL if not in getOutputs.  This is used by the setOutput()'s to tell which MotionCommand is calling


	inline MC_ID pop_free() { return cmdlist.new_front(); } //!<pulls an entry from cmdlist's free section and returns its index
	inline void push_free(MC_ID a) { cmdlist.erase(a); }    //!<puts an entry back into cmdlist's free section

	MutexLock<MAX_ACCESS> MMlock;          //!< The main lock for the class

	typedef ListMemBuf<OutputState,MAX_MOTIONS> cmdstatelist_t; //!< shorthand for a list of OutputState's
	cmdstatelist_t cmdstates[NumOutputs];  //!< requested positions by each of the MC's for each of the outputs
	float cmdSums[NumOutputs];             //!<Holds the final values for the outputs of the last frame generated
	OutputCmd cmds[NumOutputs];            //!<Holds the weighted values and total weight for the outputs of the last frame

#ifdef PLATFORM_APERIOS
	accID_t numAcc;                        //!<The number of accessors who have registered with InitAccess()
	OSubject* subjs[MAX_ACCESS];           //!<The OSubject for each process (accessor) on which it should be broadcast when a command is added
#endif

	static int _MMaccID;          //!<Stores the accessor for the current process

private:
	MotionManager(const MotionManager&); //!< this shouldn't be called...
	MotionManager& operator=(const MotionManager&); //!< this shouldn't be called...
};

//!anyone who #includes MotionManager.h will be wanting to use the global motman... don't want multiple of these! created by MotoObj
extern MotionManager * motman;

/*! @file
 * @brief Describes MotionManager, simplifies sharing of MotionCommand's and provides mutual exclusion to their access
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_3 $
 * $Revision: 1.23 $
 * $State: Exp $
 * $Date: 2004/10/18 23:10:26 $
 */

#endif
