//-*-c++-*-
#ifndef INCLUDED_VisionObjectEvent_h
#define INCLUDED_VisionObjectEvent_h

#include "EventBase.h"

//! Extends EventBase to also include location in the visual field and distance (though distance is not implimented yet)
class VisionObjectEvent : public EventBase {
public:
	//! Constructor
  VisionObjectEvent()
		: EventBase(EventBase::visObjEGID,(unsigned int)-1,EventBase::statusETID,0),_x1(0),_x2(0),_y1(0),_y2(0),
			_clipLeft(false), _clipRight(false), _clipTop(false), _clipBottom(false)
	{}
	
	//! Constructor, pass a type id and source id
  VisionObjectEvent(EventTypeID_t tid, unsigned int sid)
		: EventBase(EventBase::visObjEGID,sid,tid,0),_x1(0),_x2(0),_y1(0),_y2(0),
			_clipLeft(false), _clipRight(false), _clipTop(false), _clipBottom(false)
	{}
		
	//! Constructor, pass the type id, source id, left, right, top, bottom, x range, and y range
  VisionObjectEvent(EventTypeID_t tid, unsigned int sid, float x1, float x2, float y1, float y2, float rx, float ry)
		: EventBase(EventBase::visObjEGID,sid,tid,0),_x1(x1),_x2(x2),_y1(y1),_y2(y2),
			_clipLeft(_x1<=-rx), _clipRight(_x2>=rx), _clipTop(_y1<=-ry), _clipBottom(_y2>=ry)
	{}
	
	//! destructor
	virtual ~VisionObjectEvent() {}
  
	//!@name Attribute Accessors
  float getLeft() const { return _x1;} //!< returns the initial x coordinate of the Bounding Box
  VisionObjectEvent& setLeft(float x1) { _x1=x1; return *this;} //!< sets the initial x coordinate of the Bounding Box
  
  float getRight() const { return _x2;} //!< returns the final x coordinate of the Bounding Box
  VisionObjectEvent& setRight(float x2) { _x2=x2; return *this;} //!< sets the final x coordinate of the Bounding Box
  
	float getTop() const { return _y1;} //!< returns the initial y coordinate of the Bounding Box
  VisionObjectEvent& setTop(float y1) { _y1=y1; return *this;} //!< sets the initial y coordinate of the Bounding Box
  
  float getBottom() const { return _y2;} //!< returns the final y coordinate of the Bounding Box
  VisionObjectEvent& setBottom(float y2) { _y2=y2; return *this;} //!< sets the final y coordinate of the Bounding Box
	//@}
 
	//!@name Calculated Attributes
  float getCenterX() const { return (_x1+_x2)/2; } //!< returns the center along x
  float getCenterY() const { return (_y1+_y2)/2; } //!< returns the center along y
	float getWidth() const { return _x2-_x1; } //!< return width along x
	float getHeight() const { return _x2-_x1; } //!< return height along y
  float getArea() const { return (_x2-_x1)*(_y2-_y1); } //!< returns the final x coordinate of the Bounding Box
	//@}

  //!@name Object out of bounds Detection Functions
  bool isClippedLeft() const { return _clipLeft; } //!< returns #_clipLeft
  bool isClippedRight() const { return _clipRight; }  //!< returns #_clipRight
  bool isClippedTop() const { return _clipTop; } //!< returns #_clipTop
  bool isClippedBottom() const {return _clipBottom; } //!< returns #_clipBottom
	void setClipping(bool left, bool right, bool top, bool bottom) { _clipLeft=left; _clipRight=right; _clipTop=top; _clipBottom=bottom; } //!< sets clipping boundaries
	//@}
      
	virtual std::string getDescription(bool showTypeSpecific=true, unsigned int verbosity=0) const;
	
	virtual unsigned int getBinSize() const;
	virtual unsigned int LoadBuffer(const char buf[], unsigned int len);
	virtual unsigned int SaveBuffer(char buf[], unsigned int len) const;

protected:
	float _x1;  //!< a value representing location in visual field - from -1 if on the left edge to 1 if it's on the right edge
	float _x2;  //!< a value representing location in visual field - from -1 if on the bottom edge to 1 if it's on the top edge
	float _y1;  //!< top boundary, in approximate range (+/- 160/208) for ERS-7; actual values vary depending on aspect ratio to keep square coordinates
	float _y2;  //!< top boundary, in approximate range (+/- 160/208) for ERS-7; actual values vary depending on aspect ratio to keep square coordinates
	bool _clipLeft;   //!< flag to indicate left boundary is on or beyond the camera image's boundary
	bool _clipRight;  //!< flag to indicate right boundary is on or beyond the camera image's boundary
	bool _clipTop;    //!< flag to indicate top boundary is on or beyond the camera image's boundary
	bool _clipBottom; //!< flag to indicate bottom boundary is on or beyond the camera image's boundary
};

/*! @file
 * @brief Describes VisionObjectEvent, which provides information about objects recognized in the camera image
 * @author alokl (Creator)
 * @author Ignacio Herrero Reder <nhr at dte uma es> (VisionObjectInfo Boundary Box - bug 74)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_3 $
 * $Revision: 1.5 $
 * $State: Exp $
 * $Date: 2005/01/12 21:48:17 $
 */

#endif
