//-*-c++-*-
#ifndef INCLUDED_EventGeneratorBase_h_
#define INCLUDED_EventGeneratorBase_h_

#include "Behaviors/BehaviorBase.h"

//! A simple convenience class for event generators
/*! Note that you don't need to inherit from this class to be able to
 *  send events!  Any code can send any event any time, just by calling 
 *  one of the EventRouter::postEvent() functions.
 *  
 *  Uses a BehaviorBase base class so that you can start and stop it.
 *
 *  Allows variable settings of the generator id and source id for
 *  outgoing events as well as automatic handling of
 *  listening/unlistening for a single optional event source.  If you
 *  want something more fancy that that though, you'll have to
 *  override DoStart/DoStop yourself. (or extend/modify this class...)
 *
 */
class EventGeneratorBase : public BehaviorBase {
public:
	// Constructors are all protected - doesn't make sense to
	// instantiate this class directly, you want to use a subclass

	//! destructor - does nothing
	virtual ~EventGeneratorBase() {}
	
	virtual void DoStart();
	
	virtual void DoStop();
	
	//! if autolistening, will receive EventRouter events concerning our own listeners
	/*! This will automatically reduce overhead by eliminating chains of events thrown
	 *  that don't have any end listeners.  However, this might mean your subclass's
	 *  processEvent will be receiving the events from erouterEGID, and will need
	 *  to call EventGeneratorBase::processEvent() in order to allow them to be used */
	virtual void processEvent(const EventBase& event);
	
	//! return the generator ID that will be broadcast from
	virtual EventBase::EventGeneratorID_t getGeneratorID() { return myGenID; }
	//! set the generator ID that will be broadcast from (typically it's a bad idea to call this...)
	virtual void setGeneratorID(EventBase::EventGeneratorID_t gid) { myGenID=gid; }

	//! return the source ID that will be broadcast on
	virtual unsigned int getSourceID() { return mySourceID; }
	//! set the source ID that will be broadcast on
	virtual void setSourceID(unsigned int sid) { mySourceID=sid; }

	//! returns the source ID that will be listened for (not the source of the FilterBankEvent to be created - that depends on the subclass)
	virtual unsigned int getListenSourceID() const { return srcSourceID; }
	//! returns the generator ID that will be listened for (not the generator of the FilterBankEvent to be created - that depends on the subclass)
	virtual EventBase::EventGeneratorID_t getListenGeneratorID() const { return srcGenID; }
	//! turns on auto listening to make it easier to set up dependancies between vision filters
	virtual void setAutoListen(EventBase::EventGeneratorID_t gid, unsigned int sid);

	//! turns off auto listening
	virtual void unsetAutoListen();

protected:
	//!@name Constructors
	//!
	EventGeneratorBase(const std::string& name, EventBase::EventGeneratorID_t mgid, unsigned int msid)
		: BehaviorBase(name), myGenID(mgid), mySourceID(msid), autoListen(false), isListening(false), srcGenID(EventBase::unknownEGID), srcSourceID(0)
	{}
	EventGeneratorBase(const std::string& name, EventBase::EventGeneratorID_t mgid, unsigned int msid,EventBase::EventGeneratorID_t srcgid, unsigned int srcsid)
		: BehaviorBase(name), myGenID(mgid), mySourceID(msid), autoListen(true), isListening(false), srcGenID(srcgid), srcSourceID(srcsid)
	{}
	EventGeneratorBase(const std::string& classname, const std::string& instancename, EventBase::EventGeneratorID_t mgid, unsigned int msid)
		: BehaviorBase(classname,instancename), myGenID(mgid), mySourceID(msid), autoListen(false), isListening(false), srcGenID(EventBase::unknownEGID), srcSourceID(0)
	{}
	EventGeneratorBase(const std::string& classname, const std::string& instancename, EventBase::EventGeneratorID_t mgid, unsigned int msid,EventBase::EventGeneratorID_t srcgid, unsigned int srcsid)
		: BehaviorBase(classname,instancename), myGenID(mgid), mySourceID(msid), autoListen(true), isListening(false), srcGenID(srcgid), srcSourceID(srcsid)
	{}
	//@}

	EventBase::EventGeneratorID_t myGenID; //!< the generator ID to broadcast on
	unsigned int mySourceID;     //!< the source ID to broadcast on
	bool autoListen;          //!< if true, will automatically start listening for EventBase(genID,sourceID) events
	bool isListening;         //!< true if listening triggered by autoListen
	EventBase::EventGeneratorID_t srcGenID; //!< the generator ID to listen for (typically the source that this filter works on)
	unsigned int srcSourceID;    //!< the source ID to listen for
};

/*! @file
 * @brief 
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_3 $
 * $Revision: 1.8 $
 * $State: Exp $
 * $Date: 2004/12/20 21:47:59 $
 */

#endif
