#include "SegCamBehavior.h"
#include "Wireless/Wireless.h"
#include "Events/EventRouter.h"
#include "Events/FilterBankEvent.h"
#include "Behaviors/Controller.h"
#include "Shared/ProjectInterface.h"
#include "Vision/SegmentedColorGenerator.h"
#include "Vision/RLEGenerator.h"

SegCamBehavior::SegCamBehavior()
	: BehaviorBase("SegCamBehavior"), visRLE(NULL), packet(NULL), cur(NULL), avail(0), max_buf(0)
{
}

void
SegCamBehavior::DoStart() {
	BehaviorBase::DoStart();
	
	std::vector<std::string> args;
	args.push_back("rle");
	char port[50];
	snprintf(port,50,"%d",config->vision.rle_port);
	args.push_back(port);
	if(config->vision.rle_transport==0) {
		max_buf=UDP_WIRELESS_BUFFER_SIZE;
		visRLE=wireless->socket(SocketNS::SOCK_DGRAM, 1024, max_buf);
		args.push_back("udp");
	} else if(config->vision.rle_transport==1) {
		max_buf=TCP_WIRELESS_BUFFER_SIZE;
		visRLE=wireless->socket(SocketNS::SOCK_STREAM, 1024, max_buf);
		wireless->setDaemon(visRLE,true);
		args.push_back("tcp");
	} else {
		serr->printf("ERROR: Invalid Config::vision.rle_transport: %d\n",config->vision.rle_transport);
		return;
	}
	wireless->listen(visRLE,config->vision.rle_port);

	Controller::loadGUI("org.tekkotsu.mon.VisionGUI","SegVisionGUI",config->vision.rle_port,args);

	erouter->addListener(this,EventBase::visSegmentEGID,ProjectInterface::visSegmentSID);
	erouter->addListener(this,EventBase::visRLEEGID,ProjectInterface::visRLESID);
}

void
SegCamBehavior::DoStop() {
	erouter->removeListener(this);
	Controller::closeGUI("SegVisionGUI");

	// this could be considered a bug in our wireless - if we don't setDaemon(...,false)
	// it will try to listen again even though we explicitly closed the server socket...
	wireless->setDaemon(visRLE,false);
	wireless->close(visRLE->sock);
	BehaviorBase::DoStop();
}

void
SegCamBehavior::processEvent(const EventBase& e) {
	if(!wireless->isConnected(visRLE->sock))
		return;
	const FilterBankEvent* fbke=dynamic_cast<const FilterBankEvent*>(&e);
	ASSERTRET(fbke!=NULL,"unexpected event");
	if(config->vision.rlecam_compression==Config::vision_config::COMPRESS_NONE && e.getGeneratorID()==EventBase::visSegmentEGID) {
		bool succ=writeSeg(*fbke);
		ASSERTRET(succ,"serialization failed");
	}
	if(config->vision.rlecam_compression==Config::vision_config::COMPRESS_RLE && e.getGeneratorID()==EventBase::visRLEEGID) {
		bool succ=writeRLE(*fbke);
		ASSERTRET(succ,"serialization failed");
	}
}

bool
SegCamBehavior::openPacket(FilterBankGenerator& fbkgen, unsigned int time, unsigned int layer) {
	if(packet!=NULL)
		return false;

	avail=max_buf-1; //not sure why -1, but Alok had it, so i will too
	ASSERT(cur==NULL,"cur non-NULL");
	cur=NULL;
	char * buf=packet=(char*)visRLE->getWriteBuffer(avail);
	ASSERTRETVAL(packet!=NULL,"could not get buffer",false);
	
	unsigned int used;
	used=LoadSave::encode("TekkotsuImage",buf,avail);
	ASSERTRETVAL(used!=0,"ran out of space",false);
	avail-=used; buf+=used;
	used=LoadSave::encode(Config::vision_config::ENCODE_SINGLE_CHANNEL,buf,avail);
	ASSERTRETVAL(used!=0,"ran out of space",false);
	avail-=used; buf+=used;
	used=LoadSave::encode(Config::vision_config::COMPRESS_RLE,buf,avail);
	ASSERTRETVAL(used!=0,"ran out of space",false);
	avail-=used; buf+=used;

	used=LoadSave::encode(fbkgen.getWidth(layer),buf,avail);
	ASSERTRETVAL(used!=0,"ran out of space",false);
	avail-=used; buf+=used;
	used=LoadSave::encode(fbkgen.getHeight(layer),buf,avail);
	ASSERTRETVAL(used!=0,"ran out of space",false);
	avail-=used; buf+=used;
	used=LoadSave::encode(time,buf,avail);
	ASSERTRETVAL(used!=0,"ran out of space",false);
	avail-=used; buf+=used;
	used=LoadSave::encode(fbkgen.getFrameNumber(),buf,avail);
	ASSERTRETVAL(used!=0,"ran out of space",false);
	avail-=used; buf+=used;

	cur=buf;
	return true;
}

bool
SegCamBehavior::writeRLE(const FilterBankEvent& e) {
	FilterBankGenerator& fbkgen=*e.getSource();

	unsigned int layer=fbkgen.getNumLayers()-1-config->vision.rlecam_skip;
	unsigned int used=0;
	openPacket(fbkgen,e.getTimeStamp(),layer);
	ASSERTRETVAL(cur!=NULL,"header failed",false);
	
	RLEGenerator * rle = dynamic_cast<RLEGenerator*>(&fbkgen);
	ASSERTRETVAL(rle!=NULL,"fbkgen isn't an RLEGenerator",false);

	rle->selectSaveImage(layer,config->vision.rlecam_channel);
	used=rle->SaveBuffer(cur,avail);
	ASSERTRETVAL(used!=0,"save rle image failed",false);
	avail-=used; cur+=used;
	
	// send out the color map ourselves (since RLE compression doesn't have a concept of color)
	const SegmentedColorGenerator * seg = dynamic_cast<const SegmentedColorGenerator*>(rle->getSourceGenerator());
	ASSERTRETVAL(seg!=NULL,"RLE's source is not a SegmentedColorGenerator - how do i know what the colors are?",false);
	if(0==(used=seg->encodeColors(cur,avail))) return false;
	avail-=used; cur+=used;

	closePacket();

	return true;
}

bool
SegCamBehavior::writeSeg(const FilterBankEvent& e) {
	FilterBankGenerator& fbkgen=*e.getSource();

	unsigned int layer=fbkgen.getNumLayers()-1-config->vision.rlecam_skip;
	unsigned int used=0;
	openPacket(fbkgen,e.getTimeStamp(),layer);
	ASSERTRETVAL(cur!=NULL,"header failed",false);
	
	fbkgen.selectSaveImage(layer,config->vision.rlecam_channel);
	used=fbkgen.SaveBuffer(cur,avail);
	ASSERTRETVAL(used!=0,"save seg image failed",false);
	avail-=used; cur+=used;
	
	closePacket();

	return true;
}

void
SegCamBehavior::closePacket() {
	if(packet==NULL)
		return;
	visRLE->write(cur-packet);
	packet=cur=NULL;
	avail=0;
}


/*! @file
 * @brief Implements SegCamBehavior, which forwards segmented images from camera over wireless
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_3 $
 * $Revision: 1.11 $
 * $State: Exp $
 * $Date: 2004/11/11 01:45:36 $
 */

