//-*-c++-*-
#ifndef INCLUDED_RunSequenceControl_h_
#define INCLUDED_RunSequenceControl_h_

#include "FileBrowserControl.h"
#include "Motion/MotionSequenceMC.h"
#include "Motion/EmergencyStopMC.h"
#include "Motion/LedMC.h"
#include "SoundPlay/SoundManager.h"
#include "Motion/MMAccessor.h"
#include "Shared/TimeET.h"
#include "Shared/Config.h"
#include <string>

//! Upon activation, loads a position from a file name read from cin (stored in ms/data/motion...)
/*! The template parameter is passed to MotionSequenceMC's template
 *  parameter in order to specify the number of keyframes to reserve -
 *  larger values use more memory, but will allow you to load more
 *  complicated sequences.
 *
 *  The motion sequence doesn't actually start playing until the
 *  emergency stop is deactivated.  This avoids either cutting off the
 *  beginning of the sequence while still in estop, or having to
 *  override the estop, which may be unexpected.
 */
template<unsigned int SequenceSize>
class RunSequenceControl : public FileBrowserControl, public EventListener {
public:
	//! Constructor, sets filter to *.mot
	RunSequenceControl(const std::string& n, MotionManager::MC_ID estop_id)
		: FileBrowserControl(n,"Runs a motion sequence from a user-specified file",config->motion.root),
			estopid(estop_id), ledid(MotionManager::invalid_MC_ID), waitingFile()
	{
		setFilter("*.mot");
	}

	virtual ~RunSequenceControl() {
		erouter->removeListener(this);
		motman->removeMotion(ledid);
	}

	//! only called when e-stop has been turned off and we're waiting to load a file
	virtual void processEvent(const EventBase& /*event*/) {
		erouter->removeListener(this);
		runFile();
		motman->removeMotion(ledid);
	}
	
protected:
	//! loads the motion sequence and runs it
	void runFile() {
		//TimeET timer;
		SharedObject< MotionSequenceMC<SequenceSize> > s(waitingFile);
		//cout << "Load Time: " << timer.Age() << endl;
		motman->addPrunableMotion(s);
		waitingFile="";
	}

	//!does the actual loading of the MotionSequence
	virtual ControlBase* selectedFile(const std::string& f) {
		waitingFile=f;
		if(!MMAccessor<EmergencyStopMC>(estopid)->getStopped()) {
			runFile();
		} else {
			//we have to wait for the estop to be turned off
			sndman->PlayFile("donkey.wav");
			SharedObject<LedMC> led;
			led->cset(FaceLEDMask,0);
			led->cycle(BotLLEDMask,1000,3,0,0);
			led->cycle(BotRLEDMask,1000,3,0,500);
			ledid=motman->addPersistentMotion(led);
			erouter->addListener(this,EventBase::estopEGID,estopid,EventBase::deactivateETID);
		}
		return this;
	}

	MotionManager::MC_ID estopid; //!< MC_ID of the emergency stop (so we can tell if it's active and wait until it's turned off)
	MotionManager::MC_ID ledid; //!< MC_ID of the led we use to signal there's a MotionSequence lined up
	std::string waitingFile; //!< filename of the motion sequence waiting to load
};

/*! @file
 * @brief Defines RunSequenceControl, which when activated, loads and runs a motion sequence from a file name read from cin (stored in ms/data/motion)
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_3 $
 * $Revision: 1.15 $
 * $State: Exp $
 * $Date: 2005/01/07 19:34:51 $
 */

#endif
