//-*-c++-*-
#ifndef INCLUDED_BehaviorBase_h_
#define INCLUDED_BehaviorBase_h_

#include "Events/EventListener.h"
#include "Shared/ReferenceCounter.h"
#include <string>
#include <set>

//! The basis from which all other Behaviors should inherit
/*! Makes use of ReferenceCounter so that behaviors can automatically delete themselves if
 *  wanted.
 *
 *  Make sure your own DoStart and DoStop call BehaviorBase::DoStart (or Stop) to allow
 *  the auto-deletion from reference counting... otherwise you'll get memory leaks if you
 *  rely on the reference counting.
 *
 *  For an empty behavior boilerplate file to help you get started quickly, try
 *  <a href="http://cvs.tekkotsu.org/cgi-bin/viewcvs.cgi/Tekkotsu/docs/behavior_header.h?rev=HEAD&content-type=text/vnd.viewcvs-markup"><i>Tekkotsu</i><tt>/docs/behavior_header.h</tt></a>:
 * 
 *  But it would probably still be a good idea to go through the "<a
 *  href="../FirstBehavior.html">First Behavior</a>" tutorial to get a better idea of
 *  what's going on.
 */
class BehaviorBase : public ReferenceCounter, public EventListener {
public:
	//! destructor - if is active when deleted, will call DoStop() first
	virtual ~BehaviorBase();
	
	//! By default, merely adds to the reference counter (through AddReference()); Note you should still call this from your overriding methods
	virtual void DoStart();

	//! By default, subtracts from the reference counter (RemoveReference()), and thus may deletex if zero;  Don't forget to still call this when you override this; <b>Warning:</b> call this at the <i>end</i> of your DoStop(), not beginning (it might @c delete @c this )
	virtual void DoStop();
	
	//! By defining here, allows you to get away with not supplying a processEvent() function for the EventListener interface.  By default, does nothing.
	virtual void processEvent(const EventBase& /*event*/) {}

	//! Identifies the behavior in menus and such
	virtual std::string getName() const { return instanceName; }

	//! Allows dynamic renaming of behaviors
	virtual void setName(const std::string& name) { instanceName=name; }

	//! Gives a short description of what this particular instantiation does (in case a more specific description is needed on an individual basis)
	/*! By default simply returns getName(), because any calls from a
	 *  BehaviorBase function to getClassDescription() are going to call
	 *  BehaviorBase::getClassDescription(), not
	 *  ~YourSubClass~::getClassDescription(), because static functions
	 *  can't be virtual in C++ (doh!)
	 *
	 *  This means that getDescription called on a pointer to a
	 *  BehaviorBase of unknown subtype would always return an empty
	 *  string, which is pretty useless.  So instead we return the name
	 *  in this situation.  If you want getDescription to return
	 *  getClassDescription, you'll have to override it in your subclass
	 *  to do so. */
	virtual std::string getDescription() const {
		std::string d=getClassDescription();
		return (d.size()==0)?getName():d;
	}

	//! Returns the name of the class of this behavior (aka its type)
	/*! Note that this isn't static to avoid the problems we found with
	 *  getDescription/getClassDescription.  So instead we wind up
	 *  wasting some memory in each instance of the class to store the
	 *  className, but at least it will work the way you expect. */
	virtual std::string getClassName() const { return className; }
	
	//! Gives a short description of what this class of behaviors does... you should override this (but don't have to)
	/*! If you do override this, also consider overriding getDescription() to return it */
	static std::string getClassDescription() { return ""; }

	//! Returns true if the behavior is currently running
	virtual bool isActive() const { return started; }
	
	//! Allows read-only access to the set of currently instantiated behaviors
	/*! Not all of these behaviors are necessarily active, this is everything that has been allocated and not yet deallocated */
	static const std::set<BehaviorBase*>& getRegistry() { return registry; }

	// Just some debugging stuff in stasis
	/*	virtual void AddReference() {
			std::cout << getName() << " AddReference()==" << GetReferences() << ' ' << this << std::endl;
			ReferenceCounter::AddReference();
			}
			
			virtual void RemoveReference() {
			std::cout << getName() << " RemoveReference()==" << GetReferences() << ' ' << this << std::endl;
			ReferenceCounter::RemoveReference();
			}
	*/
	
protected:
	//! deprecated, behavior constructors should take a name argument (which by default should be the name of the type of the class)
	BehaviorBase() __attribute__((deprecated));
	//! constructor, @a name is used as both instance name and class name
	explicit BehaviorBase(const std::string& name);
	//! constructor, allows different initial values for class name and instance name
	BehaviorBase(const std::string& classname, const std::string& instancename);
	//! copy constructor; assumes subclass handles copying approriately - i.e. if @a b is active, the copy will be as well, even though DoStart was never called..
	BehaviorBase(const BehaviorBase& b);
	//! assignment operator; assumes subclass handles assignment appropriately - i.e. if @a b is active, the copy will be as well, even though DoStart was never called..
	BehaviorBase& operator=(const BehaviorBase& b);

	bool started; //!< true when the behavior is active
	std::string instanceName; //!< holds the name of this instance of behavior
	const std::string className; //!< holds the type of the subclass of this behavior as a string
	static std::set<BehaviorBase*> registry; //!< allows us to keep track of all the current behaviors
};

/*! @file
 * @brief Describes BehaviorBase from which all Behaviors should inherit
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_3 $
 * $Revision: 1.18 $
 * $State: Exp $
 * $Date: 2005/01/04 19:51:37 $
 */

#endif
