//-*-c++-*-
#ifndef INCLUDED_JPEGGenerator_h_
#define INCLUDED_JPEGGenerator_h_

#include "Vision/FilterBankGenerator.h"

extern "C" {
#include <jpeglib.h>
}

//! Generates FilterBankEvents containing JPEG compressed images
/*! There's only one channel per resolution layer, which holds the
 *  compressed data.  This is mostly intended for being sent over
 *  wireless, but if you know how JPEG works, you may be able to
 *  interpret the compressed data directly and glean useful
 *  information from it.  After all, compression is all about throwing
 *  away unimportant details and storing the salient information.
 *  
 *  The generated events use 0 for their event source IDs.  The row
 *  skip and row stride are 0. (they don't really apply here)
 *
 *  This can either compress a greyscale image or an interleaved YUV
 *  image.  If the source generator's type is not
 *  InterleavedYUVGenerator, it will assume greyscale.  Call
 *  setSource() to override this.
 *
 *  The InterleavedYUVGenerator is separated from this because it
 *  wouldn't really make things much faster to combine the algorithms,
 *  and others might find the interleaved format handy for passing to
 *  other libraries which expect that format, such as what happened
 *  with libjpeg.
 *
 *  @todo possible speedup by using jpeg_write_raw_data
 */
class JPEGGenerator : public FilterBankGenerator {
public:
	static const unsigned int JPEG_HEADER_PAD=500; //!< add a bit to the expected size in getBinSize just to leave a little extra room for small images
	enum src_mode_t {
		SRC_AUTO,       //!< if src is not a InterleavedYUVGenerator, SRC_GREYSCALE, otherwise SRC_COLOR
		SRC_GRAYSCALE,
		SRC_COLOR
	};

	//! constructor
	JPEGGenerator(EventBase::EventGeneratorID_t gid, unsigned int sid, unsigned int mysid);
	//! constructor
	JPEGGenerator(EventBase::EventGeneratorID_t gid, unsigned int sid, unsigned int mysid, src_mode_t sMode);

	//! destructor
	virtual ~JPEGGenerator();

	//! set #srcMode and #curMode as well if @a mode==SRC_AUTO
	virtual void setSourceMode(src_mode_t mode) { srcMode=mode; if(mode!=SRC_AUTO) curMode=mode;}
	//! returns #srcMode
	virtual src_mode_t getSourceMode() const { return srcMode; }
	//! returns #curMode
	virtual src_mode_t getCurrentSourceFormat() const { return curMode; }

	static std::string getClassDescription() { return "Converts a compresses its source FilterBankGenerator's data into JPEG format"; }

	//! should receive FilterBankEvents from a RawCameraGenerator (or a subclass thereof)
	virtual void processEvent(const EventBase& event);
	
	//! if we don't already know bytesUsed, let's assume the size will be smaller than the original uncompressed. If we fail this assumption, probably better to fail anyway.
	virtual unsigned int getBinSize() const;

	virtual unsigned int LoadBuffer(const char buf[], unsigned int len);

	virtual unsigned int SaveBuffer(char buf[], unsigned int len) const;

	//! returns #quality
	virtual unsigned int getQuality() const { return quality; }
	//! sets #quality; this will invalidate the cache if @a q does not equal current #quality
	virtual void setQuality(unsigned int q) {
		if(quality!=q) {
			quality=q; 
			invalidateCaches();
		}
	}

	//! returns the number of bytes used for the image returned by getImage() - will return 0 if the image hasn't been calculated yet (so call it @e after getImage())
	virtual unsigned int getImageSize(unsigned int layer, unsigned int chan) const { return bytesUsed[layer][chan]; }

protected:
	virtual void setNumImages(unsigned int nLayers, unsigned int nChannels);
	virtual unsigned char * createImageCache(unsigned int layer, unsigned int chan) const;
	virtual void calcImage(unsigned int layer, unsigned int chan) const;
	virtual void destruct();

	src_mode_t srcMode;   //!< how to interpret source channel of next filter bank event
	src_mode_t curMode;   //!< how to interpret getImage's current image

	unsigned int ** bytesUsed; //!< number of bytes used per image to actually store data;

	mutable struct jpeg_compress_struct cinfo; //!< used to interface with libjpeg - holds compression parameters and state
	mutable struct jpeg_error_mgr jerr;        //!< used to interface with libjpeg - gives us access to error information

	unsigned int quality; //!< quality level to pass to libjpeg; -1U causes Config::vision_config::rawcam_compress_quality to be used
	
private:
	JPEGGenerator(const JPEGGenerator& fbk); //!< don't call
	const JPEGGenerator& operator=(const JPEGGenerator& fbk); //!< don't call
};

/*! @file 
 * @brief Describes JPEGGenerator, which generates FilterBankEvents containing JPEG compressed images
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_2 $
 * $Revision: 1.7 $
 * $State: Exp $
 * $Date: 2004/02/18 21:13:32 $
 */

#endif
