/*
Copyright (C) 2003-2004  Etienne Lachance

This library is free software; you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as
published by the Free Software Foundation; either version 2.1 of the
License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


Report problems and direct all questions to:

email: etienne.lachance@polytml.ca or richard.gourdeau@polymtl.ca

-------------------------------------------------------------------------------
Revision_history:

2004/06/10: Etienne Lachance
    -Added doxygen documentation.

2004/07/01: Ethan Tira-Thompson
    -Added support for newmat's use_namespace #define, using ROBOOP namespace

2004/07/13: Ethan Tira-Thompson
    -Added a select_real and add_real function for type indepence of Real
    -Added functions to test for sections and parameters existance

2004/07/23: Ethan Tira-Thompson
    -Fixed potentially uninitialized variables and some other warnings

2004/09/01: Ethan Tira-Thompson
    -Added optional parameter to constructor so you can automatically read_conf
    -select_* functions are now const
-------------------------------------------------------------------------------
*/

/*! 
  @file config.cpp
  @brief Configuration class functions.
*/

//! @brief RCS/CVS version.
static const char rcsid[] = "$Id: config.cpp,v 1.9 2004/09/01 19:51:04 ejt Exp $";


#include "config.h"

#ifdef use_namespace
namespace ROBOOP {
  using namespace NEWMAT;
#endif

Config::Config(const string & filename_, bool doRead/*=false*/)
//! @brief Constructor.
	: conf(), filename(filename_)
{
   if(doRead)
      read_conf();
}

Config::Config(const Config & x)
//! @brief Constructor.
	: conf(x.conf), filename(x.filename)
{
}

Config & Config::operator=(const Config & x)
//! @brief Overload = operator.
{
   conf = x.conf;
   filename = x.filename;
   return *this;
}

short Config::read_conf()
/*!
  @brief Read a configuration file.

  This function reads the configuration file specified in 
  the constructor parameter. The information is stored in the
  variable conf.

  A configuration file contains section(s) (between [ ]), and the 
  section(s) contains parameter(s) with there respective value(s). 
  The section and the parameter are always access via a string. Below 
  is an exemple: one section named PUMA560_mDH, and two parameters.

  [PUMA560_mDH]
  DH:         0
  dof:        6
*/
{
   const char *ptr_filename = filename.c_str(); //transform string to *char

   std::ifstream inconffile(ptr_filename, std::ios::in);

   if(inconffile)
   {
      string temp;
      int tmpPos;
      Data data;
#ifdef __WATCOMC__
      char tempo[256], tempo2[256];
      string section, parameter, value;
      inconffile.getline(tempo,sizeof(tempo));
      temp = tempo;
#else
      getline(inconffile, temp);
#endif

      while( !inconffile.eof() )
      {
	 // Is-it comment line?
         if(temp.substr(0,1) != "#")
         {
	    // Is-it a section name?
            if(temp.substr(0,1) == "[") // [section]
            {
	       // Search for the end of the section name and ignore the rest of the line.
 	       tmpPos = temp.find("]");
	       if (tmpPos != (int)string::npos)
		 {
		   data.section = temp.substr(1, tmpPos-1); // remove []
		   // Read the next line, it's should be a parameter.
#ifdef __WATCOMC__
		   inconffile.getline(tempo,sizeof(tempo));
		   temp = tempo;
#else
		   getline(inconffile, temp);
#endif
		   // Be sure that is not another section name.
		   while( (temp.substr(0,1) != "[") &&
			  (!inconffile.eof()) )
		     {
		       if(temp.substr(0,1) != "#") // ignore comments
			 {
			   if(temp.find(":") != string::npos)
			     {
#ifdef __WATCOMC__
			       istrstream inputString(tempo);
			       inputString >> tempo2;
			       data.parameter = tempo2;
			       inputString >> tempo2;
			       data.value = tempo2;
#else
			       istringstream inputString(temp);
			       inputString >> data.parameter >> data.value;
#endif
			       // Find ":" in parameter.
			       tmpPos = data.parameter.find(":");
			       if (tmpPos != (int)string::npos)
				 // remove ":" a the end of parameter
				 data.parameter = data.parameter.substr(0, tmpPos);
			       else
				 {
#ifdef __WATCOMC__
				   inputString >> tempo2;
				   data.value = tempo2;
#else
				   inputString >> data.value;
#endif
				 }

			       // Add data to the config vector
			       conf.push_back(data);
			     }

			   // Read the next line.
#ifdef __WATCOMC__
			   inconffile.getline(tempo,sizeof(tempo));
			   temp = tempo;
#else
			   getline(inconffile, temp);
#endif
			 }
		       else
			 {
			   // Ignore comments and read the next line.
#ifdef __WATCOMC__
			   inconffile.getline(tempo,sizeof(tempo));
			   temp = tempo;
#else
			   getline(inconffile, temp);
#endif
			 }
		     }
		 }
            }
         }
         if(temp.substr(0,1) != "[") {
#ifdef __WATCOMC__
            inconffile.getline(tempo,sizeof(tempo));
            temp = tempo;
#else
            getline(inconffile, temp);
#endif
         }
      }
   }
   else
   {
      cerr << "Config::read_conf: can not open file " << filename.c_str() << endl;
      return CAN_NOT_OPEN_FILE;
   }
   return 0;
}

void Config::print()
//! @brief Print the configuration data.
{
  string tmpSection;
  for(Conf_data::iterator iter = conf.begin(); iter != conf.end(); ++iter)
    {
      if (tmpSection != iter->section)
	{
	  //Beginning of a section
	  tmpSection = iter->section;
	  cout << "\n[" << tmpSection << "]" << endl;
	}
      else
	  cout << setw(15-iter->parameter.size()) << iter->parameter+":" << " " 
	       << iter->value << endl;
    }
}

bool Config::section_exists(const string& section) const
/*!
  @brief Test to see if a section exists
  @return true if @a section is found
*/
{
	for(Conf_data::const_iterator iter = conf.begin(); iter != conf.end(); ++iter)
		if(iter->section == section)
			return true;
	return false;
}

bool Config::parameter_exists(const string& section, const string& parameter) const
/*!
  @brief Test to see if a parameter exists within a section
  @return true if @a parameter is found within @a section
*/
{
	for(Conf_data::const_iterator iter = conf.begin(); iter != conf.end(); ++iter)
		if( (iter->section == section) && (iter->parameter == parameter) )
			return true;
	return false;
}

short Config::select_string(const string section, const string parameter, string & value) const
/*!
  @brief Get a parameter data, of a certain section, into the string value.
  @return 0 or SECTION_OR_PARAMETER_DOES_NOT_EXIST if the data can not be found.
*/
{
   for(Conf_data::const_iterator iter = conf.begin(); iter != conf.end(); ++iter)
   {
      if( (iter->section == section) && (iter->parameter == parameter) )
      {
         value = iter->value;
         return 0;
      }
   }
   value = "";
   cerr << "Config::select_string: section " << section.c_str() << " or parameter "
   << parameter.c_str() << " does not exist." << endl;
   return SECTION_OR_PARAMETER_DOES_NOT_EXIST;
}

short Config::select_bool(const string section, const string parameter, bool & value) const
/*!
  @brief Get a parameter data, of a certain section, into the bool value.
  @return 0 or SECTION_OR_PARAMETER_DOES_NOT_EXIST if the data can not be found.
*/
{
   string value_string;
   if(!select_string(section, parameter, value_string))
   {
#ifdef __WATCOMC__
      char temp[256];
      strcpy(temp,value_string.c_str());
      istrstream istr(temp);
      value = istr;
#else
      istringstream istr(value_string);
      istr >> value;
#endif
      return 0;
   }
   cerr << "Config::select_bool: section " << section.c_str() << " or parameter "
   << parameter.c_str() << " does not exist" << endl;
   value = false;
   return SECTION_OR_PARAMETER_DOES_NOT_EXIST;
}

short Config::select_int(const string section, const string parameter, int & value) const
/*!
  @brief Get a parameter data, of a certain section, into the int value.
  @return 0 or SECTION_OR_PARAMETER_DOES_NOT_EXIST if the data can not be found.
*/
{
   string value_string;
   if(!select_string(section, parameter, value_string))
   {
#ifdef __WATCOMC__
      char temp[256];
      strcpy(temp,value_string.c_str());
      istrstream istr(temp);
      istr >> value;
#else
      istringstream istr(value_string);
      istr >> value;
#endif
      return 0;
   }
   cerr << "Config::select_int: section " << section.c_str() << " or parameter "
   << parameter.c_str() << " does not exist" << endl;
   value = 0;
   return SECTION_OR_PARAMETER_DOES_NOT_EXIST;
}

short Config::select_short(const string section, const string parameter, short & value) const
/*!
  @brief Get a parameter data, of a certain section, into the short value.
  @return 0 or SECTION_OR_PARAMETER_DOES_NOT_EXIST if the data can not be found.
*/
{
   string value_string;
   if(!select_string(section, parameter, value_string))
   {
#ifdef __WATCOMC__
      char temp[256];
      strcpy(temp,value_string.c_str());
      istrstream istr(temp);
      istr >> value;
#else
      istringstream istr(value_string);
      istr >> value;
#endif
      return 0;
   }
   cerr << "Config::select_short: section " << section.c_str() << " or parameter "
   << parameter.c_str() << " does not exist" << endl;
   value = 0;
   return SECTION_OR_PARAMETER_DOES_NOT_EXIST;
}

short Config::select_float(const string section, const string parameter, float & value) const
/*!
  @brief Get a parameter data, of a certain section, into the float value.
  @return 0 or SECTION_OR_PARAMETER_DOES_NOT_EXIST if the data can not be found.
*/
{
   string value_string;
   if(!select_string(section, parameter, value_string))
   {
#ifdef __WATCOMC__
      char temp[256];
      strcpy(temp,value_string.c_str());
      istrstream istr(temp);
      istr >> value;
#else
      istringstream istr(value_string);
      istr >> value;
#endif
      return 0;
   }
   cerr << "Config::select_float: section " << section.c_str() << " or parameter "
   << parameter.c_str() << " does not exist" << endl;
   value = 0.0;
   return SECTION_OR_PARAMETER_DOES_NOT_EXIST;
}

short Config::select_double(const string section, const string parameter, double & value) const
/*!
  @brief Get a parameter data, of a certain section, into the double value.
  @return 0 or SECTION_OR_PARAMETER_DOES_NOT_EXIST if the data can not be found.
*/
{
   string value_string;
   if(!select_string(section, parameter, value_string))
   {
#ifdef __WATCOMC__
      char temp[256];
      strcpy(temp,value_string.c_str());
      istrstream istr(temp);
      istr >> value;
#else
      istringstream istr(value_string);
      istr >> value;
#endif
      return 0;
   }
   cerr << "Config::select_double: section " << section.c_str() << " or parameter "
   << parameter.c_str() << " does not exist" << endl;
   value = 0.0;
   return SECTION_OR_PARAMETER_DOES_NOT_EXIST;
}

short Config::select_real(const string section, const string parameter, Real & value) const
/*!
  @brief Get a parameter data, of a certain section, into the Real value.
  @return 0 or SECTION_OR_PARAMETER_DOES_NOT_EXIST if the data can not be found.
*/
{
   string value_string;
   if(!select_string(section, parameter, value_string))
   {
#ifdef __WATCOMC__
      char temp[256];
      strcpy(temp,value_string.c_str());
      istrstream istr(temp);
      istr >> value;
#else
      istringstream istr(value_string);
      istr >> value;
#endif
      return 0;
   }
   cerr << "Config::select_real: section " << section.c_str() << " or parameter "
   << parameter.c_str() << " does not exist" << endl;
   value = 0.0;
   return SECTION_OR_PARAMETER_DOES_NOT_EXIST;
}

short Config::write_conf(const string name, const string file_title,
                         const int space_between_column)
/*!
  @brief Write the configuration information, contained in conf, on disk.
  @param name: Configuration file name.
  @param file_title: Title in the configuration file header.
  @param space_between_column: Number of blanks between : (of a parameter) and it's value.
*/
{
   const char *ptr_filename = name.c_str(); //transform string to *char
   std::ofstream outconffile(ptr_filename, std::ios::out);

   if(outconffile)
   {                     // file header
      outconffile << "# ---------------------------------------------------" << endl;
      outconffile << "# " << file_title << endl;
      outconffile << "# ---------------------------------------------------" << endl;
      outconffile << endl;

      string section = "";

      for(Conf_data::iterator iterConf = conf.begin(); iterConf != conf.end(); ++iterConf)
      {
         if(section != iterConf->section)
         {
            section = iterConf->section;
            outconffile << "\n[" << section << "]\n" << endl;
         }
	 outconffile << setw(space_between_column-iterConf->parameter.size()) 
		     << iterConf->parameter + ":" << " " << iterConf->value << endl;
      }
      return 0;
   }
   else
   {
      cerr << "Config::write_conf: can not create file " << name.c_str() << endl;
      return CAN_NOT_CREATE_FILE;
   }
}

void Config::add_string(const string section, const string parameter, const string value)
/*!
  @brief Added the value(string) of the parameter in the section in the configuration data.

  The functions will added the parameter and the section if it does not already exist.
*/
{
   Data dataSet;
   dataSet.section = section;
   dataSet.parameter = parameter;
   dataSet.value = value;
   for(Conf_data::iterator iterConf = conf.begin(); iterConf != conf.end(); ++iterConf)
   {
      if(section == iterConf->section) // section already exist
      {
         if(parameter == iterConf->parameter) // parameter already exist
         {
            iterConf->value = value;
            return;
         }
         // parameter does not exist
         for(Conf_data::iterator iterConf2 = iterConf; iterConf2 != conf.end(); ++iterConf2)
         {
            if(section != iterConf2->section)
            {
               conf.insert(iterConf2, dataSet);
               return;
            }
         }
      }
   }
   // section and parameter does not exist.
   conf.push_back(dataSet);
}

void Config::add_bool(const string section, const string parameter,
                      const bool value)
/*!
  @brief Added the value (bool) of the parameter in the section in the configuration data.

  The functions use add_string by first converting the value (bool) in value (string) 
*/
{
   string value_tmp;
#ifdef __WATCOMC__
   ostrstream ostr;
#else
   ostringstream ostr;
#endif
   ostr << value;
   value_tmp = ostr.str();

   add_string(section, parameter, value_tmp);
}

void Config::add_int(const string section, const string parameter,
                     const int value)
/*!
  @brief Added the value (int) of the parameter in the section in the configuration data.

  The functions use add_string by first converting the value (int) in value (string) 
*/
{
   string value_tmp;
#ifdef __WATCOMC__
   ostrstream ostr;
#else
   ostringstream ostr;
#endif
   ostr << value;
   value_tmp = ostr.str();

   add_string(section, parameter, value_tmp);
}

void Config::add_float(const string section, const string parameter,
		       const float value)
/*!
  @brief Added the value (float) of the parameter in the section in the configuration data.

  The functions use add_string by first converting the value (float) in value (string) 
*/
{
   string value_tmp;
#ifdef __WATCOMC__
   ostrstream ostr;
#else
   ostringstream ostr;
#endif
#ifdef _MSC_VER  
   ostr << value; // need to be fixed !
#else
   ostr << setprecision(13) << value;
#endif
   value_tmp = ostr.str();

   add_string(section, parameter, value_tmp);
}

void Config::add_double(const string section, const string parameter,
                        const double value)
/*!
  @brief Added the value (double) of the parameter in the section in the configuration data.

  The functions use add_string by first converting the value (double) in value (string) 
*/
{
   string value_tmp;
#ifdef __WATCOMC__
   ostrstream ostr;
#else
   ostringstream ostr;
#endif
#ifdef _MSC_VER  
   ostr << value; // need to be fixed !
#else
   ostr << setprecision(13) << value;
#endif
   value_tmp = ostr.str();

   add_string(section, parameter, value_tmp);
}

void Config::add_real(const string section, const string parameter,
                        const Real value)
/*!
  @brief Added the value (Real) of the parameter in the section in the configuration data.

  The functions use add_string by first converting the value (Real) in value (string) 
*/
{
   string value_tmp;
#ifdef __WATCOMC__
   ostrstream ostr;
#else
   ostringstream ostr;
#endif
#ifdef _MSC_VER  
   ostr << value; // need to be fixed !
#else
   ostr << setprecision(13) << value;
#endif
   value_tmp = ostr.str();

   add_string(section, parameter, value_tmp);
}


#ifdef use_namespace
}
#endif










