//-*-c++-*-
#ifndef INCLUDED_PostureEngine_h
#define INCLUDED_PostureEngine_h

#include "Motion/OutputCmd.h"
#include "Motion/Kinematics.h"
#include "Shared/RobotInfo.h"
#include "Shared/LoadSave.h"

class WorldState;

//! A class for storing a set of positions and weights for all the outputs
/*! Handy for any class which wants to deal with setting joints and postures without writing a custom class
 *  @see PostureMC */
class PostureEngine : public LoadSave, public Kinematics {
public:
	//!constructor
	PostureEngine() : LoadSave(), Kinematics(*kine) {}
	//!constructor, loads a position from a file
	/*! @todo might want to make a library of common positions so they don't have to be loaded repeatedly from memstick */
	PostureEngine(const char * filename) : LoadSave(), Kinematics(*kine) { LoadFile(filename); }
	//!constructor, initializes joint positions to the current state of the outputs as defined by @a state
	PostureEngine(const WorldState* st) : LoadSave(), Kinematics(*kine) { takeSnapshot(st); }
	//! destructor
	virtual ~PostureEngine();

	//! sets the internal #cmds to the current state of the outputs
	virtual void takeSnapshot();

	//! sets the internal #cmds to the current state of the outputs as defined by @a state
	virtual void takeSnapshot(const WorldState* st);

	//! sets all joints to unused
	virtual void clear();

	//! sets joints of this to all joints of @a pe which are not equal to unused (layers @a pe over this) stores into this
	virtual PostureEngine& setOverlay(const PostureEngine& pe);
	//! sets joints of this to all joints of @a pe which are not equal to unused (layers @a pe over this) returns new PostureEngine
	virtual PostureEngine createOverlay(const PostureEngine& pe) const;

	//! sets joints of this which are equal to unused to @a pe, (layers this over @a pe) stores into this
	virtual PostureEngine& setUnderlay(const PostureEngine& pe);
	//! sets joints of this which are equal to unused to @a pe, (layers this over @a pe) returns new PostureEngine
	virtual PostureEngine createUnderlay(const PostureEngine& pe) const;

	//! computes a weighted average of this vs. @a pe, @a w being the weight towards @a pe (so @a w==1 just copies @a pe)
	virtual PostureEngine& setAverage(const PostureEngine& pe,float w=0.5);
	//! computes a weighted average of this vs. @a pe, @a w being the weight towards @a pe (so @a w==1 just copies @a pe)
	virtual PostureEngine createAverage(const PostureEngine& pe,float w=0.5) const;

	//! computes a weighted average of this vs. @a pe, using the weight values of the joints, storing the total weight in the result's weight value
	virtual PostureEngine& setCombine(const PostureEngine& pe);
	//! computes a weighted average of this vs. @a pe, using the weight values of the joints, storing the total weight in the result's weight value
	virtual PostureEngine createCombine(const PostureEngine& pe) const;

	//! returns the sum squared error between this and pe's output values, but only between outputs which are both not unused
	/*! @todo create a version which does weighted summing?  This treats weights as all or nothing */
	float diff(const PostureEngine& pe) const;
	
	//! returns the average sum squared error between this and pe's output values for outputs which are both not unused
	/*! @todo create a version which does weighted summing?  This treats weights as all or nothing */
	float avgdiff(const PostureEngine& pe) const;
	
	//! returns the max error between this and pe's output values for outputs which are both not unused
	/*! @todo create a version which does weighted summing?  This treats weights as all or nothing */
	float maxdiff(const PostureEngine& pe) const;
	
	//! NOT VIRTUAL! You should be able to call this to set outputs without checking out, just a peekMotion().  Theoretically.
	//!@name Output Accessors
	inline PostureEngine& setOutputCmd(unsigned int i, const OutputCmd& c) { cmds[i]=c; return *this; } //!<sets output @a i to OutputCmd @a c, returns @c *this so you can chain them
	inline OutputCmd& operator()(unsigned int i) { return cmds[i]; } //!< returns output @a i, returns a reference so you can also set "through" this call.
	inline const OutputCmd& operator()(unsigned int i) const { return cmds[i]; } //!< returns output @a i
	inline OutputCmd& getOutputCmd(unsigned int i) { return cmds[i]; } //!< returns output @a i, returns a reference so you can also set "through" this call.
	inline const OutputCmd& getOutputCmd(unsigned int i) const { return cmds[i]; } //!< returns output @a i
	//@}

	//!Uses LoadSave interface so you can load/save to files, uses a human-readable storage format
	//!@name LoadSave
	virtual unsigned int getBinSize() const;
	virtual unsigned int LoadBuffer(const char buf[], unsigned int len);
	virtual unsigned int SaveBuffer(char buf[], unsigned int len) const;
	virtual unsigned int LoadFile(const char filename[]);
	virtual unsigned int SaveFile(const char filename[]) const;
	//@}

	//! Performs inverse kinematics to solve for positioning @a Plink on link @a j to @a Pobj in base coordinates (expects homogenous form); if solution found, stores result in this posture and returns true
	bool solveLinkPosition(const NEWMAT::ColumnVector& Pobj, unsigned int j, const NEWMAT::ColumnVector& Plink);
	//! Performs inverse kinematics to solve for positioning Plink on link @a j to Pobj in base coordinates; if solution found, stores result in this posture and returns true
	bool solveLinkPosition(float Pobj_x, float Pobj_y, float Pobj_z, unsigned int j, float Plink_x, float Plink_y, float Plink_z)
	{ return solveLinkPosition(pack(Pobj_x,Pobj_y,Pobj_z),j,pack(Plink_x,Plink_y,Plink_z)); }

	//! Performs inverse kinematics to solve for positioning @a Plink on link @a j to @a Pobj in base coordinates (expects homogenous form); if solution found, stores result in this posture and returns true
	bool solveLinkVector(const NEWMAT::ColumnVector& Pobj, unsigned int j, const NEWMAT::ColumnVector& Plink);
	//! Performs inverse kinematics to solve for positioning Plink on link @a j to Pobj in base coordinates; if solution found, stores result in this posture and returns true
	bool solveLinkVector(float Pobj_x, float Pobj_y, float Pobj_z, unsigned int j, float Plink_x, float Plink_y, float Plink_z)
	{ return solveLinkVector(pack(Pobj_x,Pobj_y,Pobj_z),j,pack(Plink_x,Plink_y,Plink_z)); }

protected:
	//all updates come from this posture engine's own state, not WorldState
	virtual void update(unsigned int c, unsigned int l);

	//!the table of outputs' values and weights, can be accessed through setOutputCmd() and getOutputCmd()
	OutputCmd cmds[NumOutputs];
};

/*! @file
 * @brief Describes PostureEngine, a base class for managing the values and weights of all the outputs
 * @todo write a binary version of Load/Save commands for faster access
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_2 $
 * $Revision: 1.13 $
 * $State: Exp $
 * $Date: 2004/10/18 19:53:33 $
 */

#endif
