//-*-c++-*-
#ifndef INCLUDED_HeadPointerMC_h
#define INCLUDED_HeadPointerMC_h

#include "MotionCommand.h"
#include "OutputCmd.h"
#include "Shared/RobotInfo.h"
#include "roboop/robot.h"

//! This class gives some quick and easy functions to point the head at things
class HeadPointerMC : public MotionCommand {
public:
	//! Constructor, defaults to all joints to current value in ::state
	HeadPointerMC();
	
	//! Destructor
	virtual ~HeadPointerMC() {}
	
	//!@name Speed Control
	
	//! Sets #maxSpeed to 0 (no maximum)
	void noMaxSpeed() { for(unsigned int i=0; i<NumHeadJoints; i++) maxSpeed[i]=0; }
	
	//! Restores #maxSpeed to default settings from Config::Motion_Config
	void defaultMaxSpeed();
	
	//! Sets #maxSpeed in rad/sec
	/*! @param i joint offset relative to HeadOffset (i.e. one of TPROffset_t)
	 *  @param x maximum radians per second to move */
	void setMaxSpeed(unsigned int i, float x) { maxSpeed[i]=x*FrameTime/1000; }
	
	//! Returns #maxSpeed in rad/sec
	/*! @param i joint offset relative to HeadOffset (i.e. one of TPROffset_t)
	 *  @return the maximum speed of joint @a i in radians per second */
	float getMaxSpeed(unsigned int i) { return maxSpeed[i]*1000/FrameTime; }
	
	//@}
	
	//!@name Joint Accessors
	
	//! Sets the weight values for all the neck joints
	void setWeight(float w);
	
	//! Directly sets the neck values (all values in radians)
	/*! @param j1 value for first neck joint (tilt on all ERS models)
	 *  @param j2 value for second neck joint (pan on all ERS models)
	 *  @param j3 value for third neck joint (nod on ERS-7, roll on ERS-2xx) */
	void setJoints(float j1, float j2, float j3);
	
	//! Directly set a single neck joint value
	/*! @param i joint offset relative to HeadOffset (i.e. one of TPROffset_t)
	 *  @param value the value to be assigned to join @a i, in radians */
	void setJointValue(unsigned int i, float value) { headTargets[i]=clipAngularRange(HeadOffset+i,value); 	markDirty(); }
	
	//! Returns the target value of joint @a i.  Use this if you want to know the current @b commanded joint value; To get the current joint @b position, look in WorldState::outputs
	/*! @param i joint offset relative to HeadOffset (i.e. one of TPROffset_t) */
	float getJointValue(unsigned int i) const { return headTargets[i]; }
	
	//! Centers the camera on a point in space, attempting to keep the camera as far away from the point as possible
	/*! @param x location in millimeters
	 *  @param y location in millimeters
	 *  @param z location in millimeters */
	bool lookAtPoint(float x, float y, float z);
	
	//! Centers the camera on a point in space, attempting to move the camera @a d millimeters away from the point
	/*! @param x location in millimeters
	 *  @param y location in millimeters
	 *  @param z location in millimeters
	 *  @param d target distance from point in millimeters */
	bool lookAtPoint(float x, float y, float z, float d);
	
	//! Points the camera in a given direction
	/*! @param x component of the direction vector
	 *  @param y component of the direction vector
	 *  @param z component of the direction vector */
	bool lookInDirection(float x, float y, float z);
	
	//@}
	
	
	//!@name Inherited:
	virtual int updateOutputs(); //!< Updates where the head is looking
	virtual int isDirty() { return (dirty || !targetReached)?3:0; } //!< true if a change has been made since the last updateJointCmds() and we're active
	virtual int isAlive() { return true; } //!< always true, doesn't autoprune (yet)
	virtual void DoStart() { MotionCommand::DoStart(); markDirty(); } //!< marks this as dirty each time it is added
	//@}

 protected:
	//! puts x in the range (-pi,pi)
	static float normalizeAngle(float x) { return x-rint(x/(2*M_PI))*(2*M_PI); }
	
	//! if @a x is outside of the range of joint @a i, it is set to either the min or the max, whichever is closer
	static float clipAngularRange(unsigned int i, float x) {
		float min=outputRanges[i][MinRange];
		float max=outputRanges[i][MaxRange];
		if(x<min || x>max) {
			float mn_dist=fabs(normalizeAngle(min-x));
			float mx_dist=fabs(normalizeAngle(max-x));
			if(mn_dist<mx_dist)
				return min;
			else
				return max;
		} else
			return x;
	}
	//! if targetReached, reassigns headCmds from ::state, then sets dirty to true and targetReached to false
	/*! should be called each time a joint value gets modified in case
	 *  the head isn't where it's supposed to be, it won't jerk around */
	void markDirty();

	bool dirty;                          //!< true if a change has been made since last call to updateJointCmds()
	bool targetReached;                  //!< false if the head is still moving towards its target
  float headTargets[NumHeadJoints];    //!< stores the target value of each joint
	OutputCmd headCmds[NumHeadJoints];   //!< stores the last values we sent from updateOutputs
	float maxSpeed[NumHeadJoints];       //!< initialized from Config::motion_config, but can be overridden by setMaxSpeed(); rad per frame
	ROBOOP::Robot headkin;               //!< provides kinematics computations
};

/*! @file
 * @brief Describes HeadPointerMC, a class for various ways to control where the head is looking
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_2 $
 * $Revision: 1.14 $
 * $State: Exp $
 * $Date: 2004/10/14 23:02:53 $
 */

#endif

