//-*-c++-*-
#ifndef INCLUDED_BehaviorBase_h_
#define INCLUDED_BehaviorBase_h_

#include "Events/EventListener.h"
#include "Shared/ReferenceCounter.h"
#include <string>

//! The basis from which all other Behaviors should inherit
/*! Makes use of ReferenceCounter so that behaviors can automatically delete themselves if
 *  wanted.
 *
 *  Make sure your own DoStart and DoStop call BehaviorBase::DoStart (or Stop) to allow
 *  the auto-deletion from reference counting... otherwise you'll get memory leaks if you
 *  rely on the reference counting.
 *
 *  For an empty behavior boilerplate file to help you get started quickly, try
 *  <a href="http://cvs.tekkotsu.org/cgi-bin/viewcvs.cgi/Tekkotsu/docs/behavior_header.h?rev=HEAD&content-type=text/vnd.viewcvs-markup"><i>Tekkotsu</i><tt>/docs/behavior_header.h</tt></a>:
 * 
 *  But it would probably still be a good idea to go through the "<a
 *  href="../FirstBehavior.html">First Behavior</a>" tutorial to get a better idea of
 *  what's going on.
 */
class BehaviorBase : public ReferenceCounter, public EventListener {
 public:
	//! constructor
	BehaviorBase() : ReferenceCounter(), EventListener(), started(false) {}
	//! copy constructor; assumes subclass handles copying approriately - i.e. if @a b is active, the copy will be as well, even though DoStart was never called..
	BehaviorBase(const BehaviorBase& b) : ReferenceCounter(b), EventListener(b), started(b.started) {}
	//! assignment operator; assumes subclass handles assignment appropriately - i.e. if @a b is active, the copy will be as well, even though DoStart was never called..
	BehaviorBase& operator=(const BehaviorBase& b) { ReferenceCounter::operator=(b); EventListener::operator=(b); started=b.started; return *this; }

	//! destructor - if is active when deleted, will call DoStop() first
	virtual ~BehaviorBase() {
		SetAutoDelete(false);
		if(started)
			DoStop();
		//{ if(started) { references++; DoStop(); references--; } }
	}
	
	//! By default, merely adds to the reference counter (through AddReference()); Note you should still call this from your overriding methods
	virtual void DoStart() {
		//std::cout << getName() << " started " << this << std::endl;
		if(!started) {
			started=true;
			AddReference();
		}
	}

	//! By default, subtracts from the reference counter (RemoveReference()), and thus may deletex if zero;  Don't forget to still call this when you override this; <b>Warning:</b> call this at the <i>end</i> of your DoStop(), not beginning (it might @c delete @c this )
	virtual void DoStop() {
		//std::cout << getName() << " stopped " << this << std::endl;
		if(started) {
			started=false;
			RemoveReference();
		}
	}
	
	//! By defining here, allows you to get away with not supplying a processEvent() function for the EventListener interface.  By default, does nothing.
	virtual void processEvent(const EventBase& /*event*/) {};

	/*	virtual void AddReference() {
			std::cout << getName() << " AddReference()==" << GetReferences() << ' ' << this << std::endl;
			ReferenceCounter::AddReference();
			}
			
			virtual void RemoveReference() {
			std::cout << getName() << " RemoveReference()==" << GetReferences() << ' ' << this << std::endl;
			ReferenceCounter::RemoveReference();
			}
	*/
	
	//! Identifies the behavior in menus and such
	virtual std::string getName() const =0;

	//! Gives a short description of what this class of behaviors does... you should override this (but don't have to)
	static std::string getClassDescription() { return ""; }

	//! Gives a short description of what this particular instantiation does (in case a more specific description is needed on an individual basis)  By default simply returns getClassDescription()
	virtual std::string getDescription() const { return getClassDescription(); }

	//! Returns true if the behavior is currently running
	virtual bool isActive() const { return started; }

 protected:
	bool started; //!< true when the behavior is active
};

/*! @file
 * @brief Defines BehaviorBase from which all Behaviors should inherit
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_2 $
 * $Revision: 1.14 $
 * $State: Exp $
 * $Date: 2004/03/24 06:38:21 $
 */

#endif
