#include <OPENR/OSyslog.h>
#include <OPENR/OPENRAPI.h>
#include <ant.h>
#include <EndpointTypes.h>
#include <TCPEndpointMsg.h>
#include <UDPEndpointMsg.h>
#include <cstring>
#include "Wireless.h"
#include "Socket.h"
#include "MMCombo/entry.h"

using namespace SocketNS;

Wireless *wireless=NULL;

Wireless::Wireless ()
	: ipstackRef(), myOID(), freeSockets()
{
	ipstackRef = antStackRef("IPStack");
	WhoAmI(&myOID);

	sockets[0]=new DummySocket(0);
	for (int sock = 1; sock < WIRELESS_MAX_SOCKETS; sock++) {
		sockets[sock]=NULL;
		freeSockets.push_back(sock);
	}
}    

Wireless::~Wireless ()
{
	// TODO
}

Socket* Wireless::socket(TransportType_t ttype)
{
	return socket(ttype, WIRELESS_DEF_RECV_SIZE, WIRELESS_DEF_SEND_SIZE);
}

Socket* Wireless::socket(TransportType_t ttype, int recvsize, int sendsize)
{
	if (freeSockets.empty()
			|| (recvsize + sendsize) <= 256) return sockets[0];
	int sock_num=freeSockets.front();
	freeSockets.pop_front();

	sockets[sock_num]=new Socket(sock_num);
	sockets[sock_num]->trType=ttype;
	sockets[sock_num]->sendBufSize=sendsize;
	sockets[sock_num]->recvBufSize=recvsize;

	// setup send buffer
	antEnvCreateSharedBufferMsg sendBufferMsg(sendsize*2);
	sendBufferMsg.Call(ipstackRef, sizeof(sendBufferMsg));
	if (sendBufferMsg.error != ANT_SUCCESS) return sockets[0];

	sockets[sock_num]->sendBuffer = sendBufferMsg.buffer;
	sockets[sock_num]->sendBuffer.Map();
	sockets[sock_num]->sendData = ( byte * ) ( sockets[sock_num]->sendBuffer.GetAddress() );

	// setup receive buffer
	antEnvCreateSharedBufferMsg recvBufferMsg(recvsize*2);
	recvBufferMsg.Call(ipstackRef, sizeof(recvBufferMsg));
	if (recvBufferMsg.error != ANT_SUCCESS) return sockets[0];

	sockets[sock_num]->recvBuffer = recvBufferMsg.buffer;
	sockets[sock_num]->recvBuffer.Map();
	sockets[sock_num]->recvData = ( byte * ) ( sockets[sock_num]->recvBuffer.GetAddress() );

	sockets[sock_num]->readData = sockets[sock_num]->recvData + recvsize;
	sockets[sock_num]->writeData = sockets[sock_num]->sendData + sendsize;

	return sockets[sock_num]; 
}

int Wireless::listen(int sock, int port)
{
	if ( port <= 0 || port >= 65535 || sock <= 0 || sock >= WIRELESS_MAX_SOCKETS || sockets[sock] == NULL
			 || sockets[sock]->state != CONNECTION_CLOSED )return -1;

	sockets[sock]->server_port = port;
	sockets[sock]->init();

	if (sockets[sock]->trType==SOCK_STREAM) {
		// create endpoint
		antEnvCreateEndpointMsg tcpCreateMsg( EndpointType_TCP, ( sockets[sock]->recvBufSize + sockets[sock]->sendBufSize ) * 3 );
		tcpCreateMsg.Call( ipstackRef, sizeof( tcpCreateMsg ) );
		if ( tcpCreateMsg.error != ANT_SUCCESS ) return -1;
		sockets[sock]->endpoint = tcpCreateMsg.moduleRef;

		// listen
		TCPEndpointListenMsg listenMsg( sockets[sock]->endpoint, IP_ADDR_ANY, port );
		listenMsg.continuation = ( void * ) sock;

		listenMsg.Send( ipstackRef, myOID, Extra_Entry[entryListenCont], sizeof( listenMsg ) );

		sockets[sock]->state = CONNECTION_LISTENING;
		return 0;
	} else if (sockets[sock]->trType==SOCK_DGRAM) {
		// create endpoint
		antEnvCreateEndpointMsg udpCreateMsg( EndpointType_UDP, ( sockets[sock]->recvBufSize + sockets[sock]->sendBufSize ) * 3 );
		udpCreateMsg.Call( ipstackRef, sizeof( udpCreateMsg ) );
		if ( udpCreateMsg.error != ANT_SUCCESS ) return -1;

		// bind socket
		sockets[sock]->endpoint = udpCreateMsg.moduleRef;
		UDPEndpointBindMsg bindMsg( sockets[sock]->endpoint, IP_ADDR_ANY, port );
		bindMsg.Call( ipstackRef, sizeof( bindMsg ) );
		bindMsg.continuation = ( void * ) sock;

		sockets[sock]->state = CONNECTION_CONNECTING;

		receive( sock );

		return 0;

	}

	else
		return -1;
}

/** Tell the ipstack we want to recieve messages with this function. */

int Wireless::connect( int sock, const char * ipaddr, int port )
{
	if ( port <= 0 || port >= 65535 || sock <= 0 || sock >= WIRELESS_MAX_SOCKETS || sockets[sock] == NULL
			 || ( sockets[sock]->trType == SOCK_STREAM && sockets[sock]->state != CONNECTION_CLOSED ) ) return -1;

	sockets[sock]->init();
	if (sockets[sock]->trType==SOCK_STREAM) {
		// create endpoint
		antEnvCreateEndpointMsg tcpCreateMsg( EndpointType_TCP, ( sockets[sock]->recvBufSize + sockets[sock]->sendBufSize ) * 3 );
		tcpCreateMsg.Call( ipstackRef, sizeof( tcpCreateMsg ) );
		if ( tcpCreateMsg.error != ANT_SUCCESS ) return -1;
		sockets[sock]->endpoint = tcpCreateMsg.moduleRef;

		// connect
		TCPEndpointConnectMsg connectMsg( sockets[sock]->endpoint, IP_ADDR_ANY, IP_PORT_ANY, ipaddr, port );
		connectMsg.continuation = ( void * ) sock;

		connectMsg.Send( ipstackRef, myOID, Extra_Entry[entryConnectCont], sizeof( connectMsg ) );

		sockets[sock]->state = CONNECTION_CONNECTING;
		return 0;
	}

	else if ( sockets[sock]->trType == SOCK_DGRAM )
		{
			// connect
			UDPEndpointConnectMsg connectMsg( sockets[sock]->endpoint, ipaddr, port );

			connectMsg.continuation = ( void * ) sock;

			connectMsg.Send( ipstackRef, myOID, Extra_Entry[entryConnectCont], sizeof( connectMsg ) );

			sockets[sock]->state = CONNECTION_CONNECTED;
			//std::cout << "Sock " << sock << " connected via UDP to IP " << ipaddr << " port " << port << std::flush << std::endl;

			return 0;
		}

	else
		{
			return -1;
		}
}

void
Wireless::ListenCont(void* msg)
{
	antEnvMsg * Msg = ( antEnvMsg * ) msg;
	int sock = ( int )( Msg->continuation );

	if ( sockets[sock]->trType == SOCK_STREAM )
		{
			TCPEndpointListenMsg * listenMsg = ( TCPEndpointListenMsg * ) msg;

			if ( listenMsg->error != TCP_SUCCESS )
				{
					sockets[sock]->state = CONNECTION_ERROR;

					// no use recycling since its a resource issue
					return;
				}

			sockets[sock]->state = CONNECTION_CONNECTED;
			receive( sock );
		}
}

void
Wireless::ConnectCont(void *msg)
{
	antEnvMsg * Msg = ( antEnvMsg * ) msg;
	int sock = ( int )( Msg->continuation );

	if ( sockets[sock]->trType == SOCK_STREAM )
		{
			TCPEndpointConnectMsg * connectMsg = ( TCPEndpointConnectMsg * ) msg;
			if ( connectMsg->error != TCP_SUCCESS )
				{
					sockets[sock]->state = CONNECTION_ERROR;
					return;
				}

			sockets[sock]->state = CONNECTION_CONNECTED;
			receive( sock );
		}
}

void
Wireless::BindCont(void *msg)
{
	UDPEndpointBindMsg* bindMsg = (UDPEndpointBindMsg*)msg;
	int sock = (int)bindMsg->continuation;

	if (bindMsg->error != UDP_SUCCESS) {
		sockets[sock]->state = CONNECTION_ERROR;
		return;
	}

	sockets[sock]->state = CONNECTION_CONNECTED;
}

void
Wireless::send(int sock)
{
	if ( sock <= 0 || sock >= WIRELESS_MAX_SOCKETS || sockets[sock] == NULL || sockets[sock]->state != CONNECTION_CONNECTED
			 || sockets[sock]->sendSize <= 0 ) return;

	if ( sockets[sock]->trType == SOCK_STREAM )
		{
			TCPEndpointSendMsg sendMsg( sockets[sock]->endpoint, sockets[sock]->sendData, sockets[sock]->sendSize );
			sendMsg.continuation = ( void * ) sock;

			sockets[sock]->tx = true;
			sendMsg.Send( ipstackRef, myOID, Extra_Entry[entrySendCont], sizeof( TCPEndpointSendMsg ) );
			sockets[sock]->sendSize = 0;
		}

	else if ( sockets[sock]->trType == SOCK_DGRAM )
		{
			UDPEndpointSendMsg sendMsg( sockets[sock]->endpoint, sockets[sock]->sendData, sockets[sock]->sendSize );

			// this field is just hijacked to id the socket # this message is being sent across
			sendMsg.continuation = ( void * ) sock;

			sockets[sock]->tx = true;
			sendMsg.Send( ipstackRef, myOID, Extra_Entry[entrySendCont], sizeof( UDPEndpointSendMsg ) );
			sockets[sock]->sendSize = 0;
		}
}

void
Wireless::SendCont(void* msg)
{
	antEnvMsg * Msg = ( antEnvMsg * ) msg;
	int sock = ( int )( Msg->continuation );

	if ( sockets[sock]->trType == SOCK_STREAM )
		{
			TCPEndpointSendMsg * sendMsg = ( TCPEndpointSendMsg * ) msg;
			sockets[sock]->tx = false;
			if ( sendMsg->error != TCP_SUCCESS )
				{
					sockets[sock]->state = CONNECTION_ERROR;
					close( sock );
					return;
				}
		}

	else if ( sockets[sock]->trType == SOCK_DGRAM )
		{
			UDPEndpointSendMsg * sendMsg = ( UDPEndpointSendMsg * ) msg;
			sockets[sock]->tx = false;
			if ( sendMsg->error != UDP_SUCCESS )
				{
					sockets[sock]->state = CONNECTION_ERROR;
					close( sock );
					return;
				}
		}

	sockets[sock]->flush();
}

/*! @bug This doesn't actually seem to block until the message is
*	fully sent... a crash immediately after this will still cause a
*	line or two to be dropped.  This is still less dropped than
*	regular send, but doesn't do much good for debugging until we fix
*	this. (if we can...) */
void
Wireless::blockingSend(int sock)
{
	if ( sock <= 0 || sock >= WIRELESS_MAX_SOCKETS || sockets[sock] == NULL || sockets[sock]->state != CONNECTION_CONNECTED
			 || sockets[sock]->sendSize <= 0 ) return;

	if ( sockets[sock]->trType == SOCK_STREAM )
		{
			TCPEndpointSendMsg sendMsg( sockets[sock]->endpoint, sockets[sock]->sendData, sockets[sock]->sendSize );
			sendMsg.continuation = ( void * ) sock;

			sockets[sock]->tx=true;
			sockets[sock]->sendSize = 0;
			sendMsg.Call( ipstackRef, sizeof( TCPEndpointSendMsg ) );
			sockets[sock]->tx = false;
		}

	// no double buffering
}

void
Wireless::setReceiver(int sock, int (*rcvcbckfn) (char*, int) )
{
	if (sock<=0 || sock>=WIRELESS_MAX_SOCKETS || sockets[sock]==NULL) return;

	sockets[sock]->rcvcbckfn=rcvcbckfn;
}

void
Wireless::receive(int sock)
{
	if ( sock <= 0 || sock >= WIRELESS_MAX_SOCKETS || sockets[sock] == NULL
			 || ( sockets[sock]->trType == SOCK_STREAM && sockets[sock]->state != CONNECTION_CONNECTED ) )
		return;

	if ( sockets[sock]->trType == SOCK_STREAM )
		{
			TCPEndpointReceiveMsg receiveMsg( sockets[sock]->endpoint, sockets[sock]->recvData, 1, sockets[sock]->recvBufSize );
			receiveMsg.continuation = ( void * ) sock;
			receiveMsg.Send( ipstackRef, myOID, Extra_Entry[entryReceiveCont], sizeof( receiveMsg ) );
		}

	else if ( sockets[sock]->trType == SOCK_DGRAM )
		{
			UDPEndpointReceiveMsg receiveMsg( sockets[sock]->endpoint, sockets[sock]->recvData, sockets[sock]->recvBufSize );
			receiveMsg.continuation = ( void * ) sock;
			receiveMsg.Send( ipstackRef, myOID, Extra_Entry[entryReceiveCont], sizeof( receiveMsg ) );
		}

	sockets[sock]->rx = true;
}

void
Wireless::receive(int sock, int (*rcvcbckfn) (char*, int) )
{
	if (sock<=0 || sock>=WIRELESS_MAX_SOCKETS || sockets[sock]==NULL
			|| sockets[sock]->state != CONNECTION_CONNECTED) return;

	sockets[sock]->rcvcbckfn = rcvcbckfn;

	if ( sockets[sock]->trType == SOCK_STREAM )
		{
			TCPEndpointReceiveMsg receiveMsg( sockets[sock]->endpoint, sockets[sock]->recvData, 1, sockets[sock]->recvBufSize );
			receiveMsg.continuation = ( void * ) sock;
			receiveMsg.Send( ipstackRef, myOID, Extra_Entry[entryReceiveCont], sizeof( receiveMsg ) );
		}

	else if ( sockets[sock]->trType == SOCK_DGRAM )
		{
			UDPEndpointReceiveMsg receiveMsg( sockets[sock]->endpoint, sockets[sock]->recvData, sockets[sock]->recvBufSize );
			receiveMsg.continuation = ( void * ) sock;
			receiveMsg.Send( ipstackRef, myOID, Extra_Entry[entryReceiveCont], sizeof( receiveMsg ) );
		}

	sockets[sock]->rx = true;
}

void
Wireless::ReceiveCont(void* msg)
{
	// get the socket index before casting the message into UDP or TCP form
	antEnvMsg * Msg = ( antEnvMsg * ) msg;
	int sock = ( int )( Msg->continuation );

	if ( sock <= 0 || sock >= WIRELESS_MAX_SOCKETS || sockets[sock] == NULL
			 || ( sockets[sock]->state != CONNECTION_CONNECTED && sockets[sock]->state != CONNECTION_CONNECTING ) )
		return;

	if ( sockets[sock]->trType == SOCK_STREAM )
		{
			TCPEndpointReceiveMsg * receiveMsg = ( TCPEndpointReceiveMsg * ) msg;
			if ( receiveMsg->error != TCP_SUCCESS )
				{
					sockets[sock]->state = CONNECTION_ERROR;
					close( sock );
					return;
				}

			sockets[sock]->recvSize = receiveMsg->sizeMin;
			if ( sockets[sock]->rcvcbckfn != NULL )
				sockets[sock]->rcvcbckfn( ( char * ) sockets[sock]->recvData, sockets[sock]->recvSize );
			sockets[sock]->recvSize = 0;

		}

	else if ( sockets[sock]->trType == SOCK_DGRAM )
		{
			UDPEndpointReceiveMsg * receiveMsg;
			receiveMsg = ( UDPEndpointReceiveMsg * ) antEnvMsg::Receive( msg );
			sockets[sock]->recvSize = receiveMsg->size;

			if ( receiveMsg->error == UDP_SUCCESS )
				{
					// if this UDP connection is not connected yet, connect it
					// to the address & port of the computer that sent this message.
					// This allows us to send UDP messages to any address instead of
					// hard-coding a specific address beforehand

					if ( !strncmp( "connection request", ( char * ) sockets[sock]->recvData, 18 ) )
						{
							// clear this message from the receiving buffer
							sockets[sock]->recvData += sockets[sock]->recvSize;

							if ( sockets[sock]->state != CONNECTION_CONNECTED )
								{
									char caller[14];
									receiveMsg->address.GetAsString( caller );
									connect( sock, caller, receiveMsg->port );
								}
						}

					else if ( sockets[sock]->rcvcbckfn != NULL )
						sockets[sock]->rcvcbckfn( ( char * ) sockets[sock]->recvData, sockets[sock]->recvSize );

				}

			sockets[sock]->recvSize = 0;

		}

	receive( sock );
}

void
Wireless::close(int sock)
{
	if (sockets[sock]->state == CONNECTION_CLOSED ||
			sockets[sock]->state == CONNECTION_CLOSING) return;

	if (!(sockets[sock]->server_port>0 && sockets[sock]->daemon)) {
		sockets[sock]->recvBuffer.UnMap();
		antEnvDestroySharedBufferMsg receiveBufferMsg(sockets[sock]->recvBuffer);
		receiveBufferMsg.Call(ipstackRef, sizeof(antEnvDestroySharedBufferMsg));
		sockets[sock]->sendBuffer.UnMap();
		antEnvDestroySharedBufferMsg sendBufferMsg(sockets[sock]->sendBuffer);
		sendBufferMsg.Call(ipstackRef, sizeof(antEnvDestroySharedBufferMsg));
	}

	if ( sockets[sock]->trType == SOCK_STREAM )
		{
			TCPEndpointCloseMsg closeMsg( sockets[sock]->endpoint );
			closeMsg.continuation = ( void * ) sock;
			closeMsg.Send( ipstackRef, myOID, Extra_Entry[entryCloseCont], sizeof( closeMsg ) );
		}

	else if ( sockets[sock]->trType == SOCK_DGRAM )
		{
			UDPEndpointCloseMsg closeMsg( sockets[sock]->endpoint );
			closeMsg.continuation = ( void * ) sock;
			closeMsg.Send( ipstackRef, myOID, Extra_Entry[entryCloseCont], sizeof( closeMsg ) );
		}

	sockets[sock]->state = CONNECTION_CLOSING;
}

void
Wireless::CloseCont(void* msg)
{
	antEnvMsg * closeMsg = ( antEnvMsg * ) msg;
	int sock = ( int )( closeMsg->continuation );
	if ( sockets[sock] == NULL )
		return;

	sockets[sock]->state = CONNECTION_CLOSED;
	if ( sockets[sock]->server_port > 0 && sockets[sock]->daemon )
		{
			// recycle if server
			listen( sock, sockets[sock]->server_port );
		}

	else
		{
			delete( sockets[sock] );
			sockets[sock] = NULL;
			freeSockets.push_back( sock );
		}
}

/*! @file
 * @brief Interacts with the system to provide networking services
 * @author alokl (Creator)
 * @author Erik Berglund and Bryan Johnson (UDP support)
 * 
 * @verbinclude CMPack_license.txt
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_2_2 $
 * $Revision: 1.19 $
 * $State: Exp $
 * $Date: 2004/09/28 22:12:35 $
 */

