//-*-c++-*-
#ifndef INCLUDED_SharedObject_h
#define INCLUDED_SharedObject_h

#include <OPENR/RCRegion.h>

//! It's nice to have a parent class of SharedObject (which is what you probably want to be reading) so that you can pass around the data structure without worrying about what type is inside the shared memory region.
/*! See MotionManager for an example on how to use this. */
class SharedObjectBase {
public:
	void* data() const { return rcr->Base(); } //!< returns a pointer to the data region
	RCRegion * getRegion() const { return rcr; } //!< returns the OPEN-R memory region, should you need it

protected:
	SharedObjectBase() : rcr(NULL) {} //!< constructor, protected because you shouldn't need to create this directly, just a common interface to all templates of SharedObject

	//< destructor, automatically dereferences #rcr
	virtual ~SharedObjectBase() {
		//std::cout << "~SharedObjectBase(): rcr->NumberOfReference()==" << rcr->NumberOfReference() << std::endl;
		if(rcr && rcr->NumberOfReference()>0)
			rcr->RemoveReference();
		else
			std::cout << "WARNING: SharedObjectBase destructed without reference" << std::endl;
		//std::cout << "~SharedObjectBase()NOW: rcr->NumberOfReference()==" << rcr->NumberOfReference() << std::endl;
	} 
	RCRegion * rcr; //!< the pointer to the shared memory region this is in charge of
	
private:
	SharedObjectBase(const SharedObjectBase&); //!< this shouldn't be called...
	SharedObjectBase& operator=(const SharedObjectBase&); //!< this shouldn't be called...
};	

//! This templated class allows convenient creation of any type of class wrapped in a shared memory region
/*! @see MotionManager for an example on how to use this.*/
template<class MC>
class SharedObject : public SharedObjectBase {
public:
	//!if you really need more than 5 arguments for your class, well, you're one crazy puppy but if you really want to, just make more like shown... (yay templates!)
	//!@name templated contructors - allows you to pass constructor arguments on to the object being created

	//! Creates the class with the default constructor
	SharedObject() : SharedObjectBase() {
		rcr=createRCRegion();
		new (rcr->Base()) MC;
	}
	//! Creates the class, passing its constructor t1
	template<class T1> SharedObject(T1 t1) : SharedObjectBase() {
		rcr=createRCRegion();
		new (rcr->Base()) MC(t1);
	}
	//! Creates the class, passing its constructor t1 and t2
	template<class T1, class T2> SharedObject(T1 t1, T2 t2) : SharedObjectBase(){
		rcr=createRCRegion();
		new (rcr->Base()) MC(t1,t2);
	}
	//! Creates the class, passing its constructor t1, t2, and t3
	template<class T1, class T2, class T3> SharedObject(T1 t1, T2 t2, T3 t3) : SharedObjectBase(){
		rcr=createRCRegion();
		new (rcr->Base()) MC(t1,t2,t3);
	}
	//! Creates the class, passing its constructor t1, t2, t3 and t4
	template<class T1, class T2, class T3, class T4> SharedObject(T1 t1, T2 t2, T3 t3, T4 t4) : SharedObjectBase(){
		rcr=createRCRegion();
		new (rcr->Base()) MC(t1,t2,t3,t4);
	}
	//! Creates the class, passing its constructor t1, t2, t3, t4 and t5 - if you need more arguments, just add them
	template<class T1, class T2, class T3, class T4, class T5> SharedObject(T1 t1, T2 t2, T3 t3, T4 t4, T5 t5) : SharedObjectBase(){
		rcr=createRCRegion();
		new (rcr->Base()) MC(t1,t2,t3,t4,t5);
	}
	//@}

	MC* operator->() const { return dataCasted(); } //!< smart pointer to the underlying class
	MC& operator*() const { return *dataCasted(); } //!< smart pointer to the underlying class
	MC& operator[](int i) const { return dataCasted()[i]; } //!< smart pointer to the underlying class
protected:
	//! creates and returns RCRegion of correct size for current class.  Adds a reference (which is removed in the destructor)
	static RCRegion * createRCRegion() {
		RCRegion * r = new RCRegion(calcsize());
		//std::cout << "createRCRegion(): rcr->NumberOfReference()==" << r->NumberOfReference() << std::endl;
		r->AddReference();
		//std::cout << "createRCRegion()NOW: rcr->NumberOfReference()==" << r->NumberOfReference() << std::endl;
		return r;
	}
	MC* dataCasted() const { return static_cast<MC*>(data()); } //!< returns a correctly typed pointer to the object's memory

	//!Calculates the size of the memory region to be used, rounding up to the nearest page size
	/*! Not sure this is completely necessary, but may be nice.  Of course, this also means even
	 *  small regions are going to be at least 4K (current page size)  If memory gets tight or we
	 *  get a lot of little regions floating around, this might be worth checking into */
	static unsigned int calcsize() {
		size_t size = sizeof(MC);
		sError error;
		size_t page_size;
		error = GetPageSize(&page_size);
		if (error != sSUCCESS) {
			cout << "error: " << error << " getting page size in SharedMem" << endl;
			page_size = 4096;
		}
		
		int new_size,num_pages;
		num_pages = (size+page_size-1)/page_size;
		new_size = num_pages*page_size;
		//cout << "req" << size << "new" << new_size << "ps" << page_size << endl;
		/*		cout << "data size is " << sizeof(MC) << endl;
					cout << "msg size is " << MotionManagerMsg::SIZEOF_MSG << endl;
					cout << "SIZE is " << rcr->Size() << endl;
					cout << "PAGE is " << page_size << endl;
					cout << "BASE is " << (void*)rcr->Base() << endl; */
		return new_size;
	}
};

/*! @file
 * @brief Defines SharedObject, a wrapper for objects in order to facilitate sending them between processes
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_2_2 $
 * $Revision: 1.3 $
 * $State: Rel $
 * $Date: 2003/09/12 23:42:12 $
 */

#endif //INCLUDED_SharedObject_h
