//-*-c++-*-
#ifndef INCLUDED_PostureMC_h
#define INCLUDED_PostureMC_h

#include "MotionCommand.h"
#include "PostureEngine.h"
#include "MotionManager.h"

class WorldState;

//! a MotionCommand shell for PostureEngine
/*! Will autoprune by default once it reaches the target pose.
 * 
 *  If you want to keep it alive so your behavior can use the posture
 *  to move around, either call setAutoPrune(@c false), or pass
 *  @c false to the MotionManager::addMotion() function.
 */
class PostureMC : public MotionCommand, public PostureEngine {
public:
	//!constructor
	PostureMC() : MotionCommand(),PostureEngine(),dirty(true),tolerance(.01) { }
	//!constructor, loads from @a filename
	PostureMC(const char* filename) : MotionCommand(),PostureEngine(filename),dirty(true),tolerance(.01) { }
	//!constructor, initializes joint positions to the current state of the outputs as defined by @a state
	PostureMC(const WorldState* st) : MotionCommand(),PostureEngine(st),dirty(true),tolerance(.01) { }
	//!destructor
	virtual ~PostureMC() {}
	
	//!@name New Stuff
	/*!@brief call this if you call PostureEngine::setOutputCmd(), that doesn't know about dirty flags */
	/* This is the downside of making setOutputCmd() not virtual - if you pass this around as just a
	 * PostureEngine, calls made to that won't be able to update the dirty flag automatically.  However,
	 * if the object is typed as a PostureMC at the time the call is made, then you don't have to worry.\n
	 * These are also not virtual - otherwise you'd have to still check the motion out and it would make
	 * this all pointless! @return @c *this */
	PostureMC& setDirty(bool d=true) { dirty=d; return *this; } 
	virtual PostureMC& setTolerance(float t) { tolerance=t; return *this; } //!< sets #tolerance, returns @c *this
	virtual float getTolerance() { return tolerance; } //!< returns #tolerance
	//@}


	//!@name MotionCommand Stuff
	virtual int updateOutputs() {
		for(unsigned int i=0; i<NumOutputs; i++)
			if(cmds[i].weight>0)
				motman->setOutput(this,i,cmds[i]);
		bool r=dirty; dirty=false; return r;
	}
	virtual int isDirty() { return dirty; }

	//! returns non-zero (true) if PostureEngine::maxdiff() between this and the current position is over #tolerance
	/*! This is handy so you can set to have the robot go to a position and then automatically remove
	 *  the MotionCommand when it gets there - but beware fighting Postures which average out and neither
	 *  succeeds */
	virtual int isAlive() {
		if(dirty || getAutoPrune()==false) //doesn't bother computing if we're staying alive anyway
			return true;
		PostureEngine tmp; 
		tmp.takeSnapshot(); 
		return (maxdiff(tmp)>tolerance);
	}
	virtual void DoStart() { MotionCommand::DoStart(); dirty=true; } //!< marks this as dirty each time it is added
	//@}

	//!Had to override stuff to manage a dirty flag
	//!@name PostureEngine Stuff
	inline virtual void takeSnapshot() { dirty=true; PostureEngine::takeSnapshot(); }
	inline virtual void takeSnapshot(const WorldState* st) { dirty=true; PostureEngine::takeSnapshot(st); }
	inline virtual void clear() { dirty=true; PostureEngine::clear(); }
	inline virtual PostureEngine& setOverlay(const PostureEngine& pe) { dirty=true; PostureEngine::setOverlay(pe); return *this; }
	inline virtual PostureEngine& setUnderlay(const PostureEngine& pe) { dirty=true; PostureEngine::setUnderlay(pe); return *this; }
	inline virtual PostureEngine& setAverage(const PostureEngine& pe,float w=0.5) { dirty=true; PostureEngine::setAverage(pe,w); return *this; }
	inline virtual PostureEngine& setCombine(const PostureEngine& pe) { dirty=true; PostureEngine::setCombine(pe); return *this; }
	inline PostureEngine& setOutputCmd(unsigned int i, const OutputCmd& c) { dirty=true; PostureEngine::setOutputCmd(i,c); return *this; }
	inline virtual unsigned int LoadBuffer(const char buf[], unsigned int len) { dirty=true; return PostureEngine::LoadBuffer(buf,len); }
	//@}

protected:
	bool dirty; //!< true if changes have been made since last updateOutputs()
	float tolerance; //!< when autopruning, if the maxdiff() of this posture and the robot's current position is below this value, isAlive() will be false, defaults to 0.01 (5.7 degree error)

	//!defines Registerable stuff
	/*	REGDEFCUSTOMMEM(PostureMC) {
		MotionCommand::registerDataRegions(regions);
		registerRegion(regions,&cmds,&cmds,sizeof(cmds));//for PostureEngine, LoadSave doesn't have any (nonstatic) members
		registerRegion(regions,&dirty,&tolerance,sizeof(tolerance));
		}*/
};

/*! @file
 * @brief Defines PostureMC, a MotionCommand shell for PostureEngine
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_2_2 $
 * $Revision: 1.9 $
 * $State: Exp $
 * $Date: 2004/12/21 00:42:21 $
 */

#endif
