#include "PostureEngine.h"
#include "Shared/WorldState.h"
#include "Motion/roboop/robot.h"
#include "Shared/Config.h"
#include <stdio.h>

PostureEngine::~PostureEngine() {}

void PostureEngine::takeSnapshot() {
	for(unsigned int i=0; i<NumOutputs; i++)
		cmds[i].set(state->outputs[i],1);
}

void PostureEngine::takeSnapshot(const WorldState* st) {
	for(unsigned int i=0; i<NumOutputs; i++)
		cmds[i].set(st->outputs[i],1);
}

void PostureEngine::clear() {
	for(unsigned int i=0; i<NumOutputs; i++)
		cmds[i].unset();
}

PostureEngine& PostureEngine::setOverlay(const PostureEngine& pe) {
	for(unsigned int i=0; i<NumOutputs; i++)
		if(pe.cmds[i].weight>0)
			cmds[i]=pe.cmds[i];
	return *this;
}
PostureEngine PostureEngine::createOverlay(const PostureEngine& pe) const {
	PostureEngine tmp(*this);
	return tmp.setOverlay(pe);
}
PostureEngine& PostureEngine::setUnderlay(const PostureEngine& pe) {
	for(unsigned int i=0; i<NumOutputs; i++)
		if(cmds[i].weight<=0)
			cmds[i]=pe.cmds[i];
	return *this;
}
PostureEngine PostureEngine::createUnderlay(const PostureEngine& pe) const {
	PostureEngine tmp(*this);
	return tmp.setUnderlay(pe);
}
/*! joints being averaged with unused joints have their weights averaged, but not their values (so an output can crossfade properly)\n
 *  @param pe the other PostureEngine
 *  @param w amount to weight towards @a pe
 *  - if @a w < .001, nothing is done
 *  - if @a w > .999, a straight copy of @a pe occurs (sets joints to unused properly at end of fade)
 *  - .001 and .999 is used instead of 0 and 1 to allow for slight addition errors in a loop (if
 *    using repeated additions of a delta value instead of repeated divisions)
 *  @return @c *this, stores results into this */
PostureEngine& PostureEngine::setAverage(const PostureEngine& pe, float w) {
	if(w<0.001)
		return *this;
	if(w>0.999)
		return (*this=pe);
	float wp=1-w;
	for(unsigned int i=0; i<NumOutputs; i++)
		if(cmds[i].weight>0) {
			if(pe.cmds[i].weight>0)
				cmds[i].set(cmds[i].value*wp+pe.cmds[i].value*w,cmds[i].weight*wp+pe.cmds[i].weight*w);
			else
				cmds[i].weight*=wp;
		} else
			cmds[i].set(pe.cmds[i].value,pe.cmds[i].weight*w);
	return *this;
}
/*! joints being averaged with weight<=0 have their weights averaged, but not their values (so an output can crossfade properly)\n
 *  @param pe the other PostureEngine
 *  @param w amount to weight towards @a pe
 *  - if @a w < .001, nothing is done
 *  - if @a w > .999, a straight copy of @a pe occurs (sets joints to unused properly at end of fade)
 *  - .001 and .999 is used instead of 0 and 1 to allow for slight addition errors in a loop (if
 *    using repeated additions of a delta value instead of repeated divisions)
 *  @return a new posture containing the results */
PostureEngine PostureEngine::createAverage(const PostureEngine& pe, float w) const {
	PostureEngine tmp(*this);
	return tmp.setAverage(pe,w);
}
PostureEngine& PostureEngine::setCombine(const PostureEngine& pe) {
	for(unsigned int i=0; i<NumOutputs; i++) {
		float total=cmds[i].weight+pe.cmds[i].weight;
		cmds[i].set((cmds[i].value*cmds[i].weight+pe.cmds[i].value*pe.cmds[i].weight)/total,total);
	}
	return *this;
}
PostureEngine PostureEngine::createCombine(const PostureEngine& pe) const {
	PostureEngine tmp(*this);
	return tmp.setCombine(pe);
}

float PostureEngine::diff(const PostureEngine& pe) const {
	float ans=0;
	for(unsigned int i=0; i<NumOutputs; i++)
		if(cmds[i].weight>0 && pe.cmds[i].weight>0) {
			float dif=cmds[i].value-pe.cmds[i].value;
			ans+=dif*dif;
		}
	return ans;
}

float PostureEngine::avgdiff(const PostureEngine& pe) const {
	float ans=0;
	unsigned int cnt=0;
	for(unsigned int i=0; i<NumOutputs; i++)
		if(cmds[i].weight>0 && pe.cmds[i].weight>0) {
			float dif=cmds[i].value-pe.cmds[i].value;
			ans+=dif*dif;
			cnt++;
		}
	return ans/cnt;
}

float PostureEngine::maxdiff(const PostureEngine& pe) const {
	float max=0;
	for(unsigned int i=0; i<NumOutputs; i++)
		if(cmds[i].weight>0 && pe.cmds[i].weight>0) {
			float dif=cmds[i].value-pe.cmds[i].value;
			if(dif>max)
				max=dif;
		}
	return max;
}

unsigned int PostureEngine::getBinSize() const {
	unsigned int len=11;
	char buf[255];
	for(unsigned int i=0; i<NumOutputs; i++)
		if(cmds[i].weight>0)
			len+=snprintf(buf,len,"%s\t% .4f\t% .4f\n",outputNames[i],cmds[i].value,cmds[i].weight);
	return len;
}

unsigned int PostureEngine::LoadBuffer(const char buf[], unsigned int len) {
	unsigned int origlen=len;
	clear();
	if(strncmp("#POS",buf,4)!=0) {
		// we don't want to display an error here because we may be only testing file type,
		// so it's up to the caller to decide if it's necessarily an error if the file isn't
		// a posture
		//std::cout << "ERROR PostureEngine load corrupted - expected #POS header" << std::endl;
		return 0;
	}
	char formatstring[64];
	snprintf(formatstring,64,"%%%dc %%f %%f%%n",outputNameLen); //std::cout << "Format: " << formatstring << std::endl;
	unsigned int idx=0;
	unsigned int linenum=1;
	char jname[outputNameLen+1];
	jname[outputNameLen]='\0';
	while(len<=origlen && len>0) {
		float fval, fwht=1;
		int written;
		//		printf("%d %.9s\n",linenum+1,buf);
		if(buf[0]=='\r') {
			buf++; len--;
			if(buf[0]=='\n') {
				buf++; len--;
			}
			linenum++;
			continue;
		}
		if(buf[0]=='\n') {
			buf++; len--;
			linenum++;
			continue;
		}
		if(buf[0]=='#') {
			if(strncmp("#END\n",buf,5)==0 || strncmp("#END\r",buf,5)==0) {
				return origlen-len+5;
			} else if(strncmp("#END\r\n",buf,6)==0) {
				return origlen-len+6;
			} else {
				while(len>0 && *buf!='\n' && *buf!='\r') {len--;buf++;}
				if(*buf=='\n') { //in case of \r\n
					buf++;
					len--;
				}
				linenum++;
				continue;
			}
		}
		written=-1;
		jname[0]='\0';
		sscanf(buf,formatstring,jname,&fval,&fwht,&written);
		if(!ChkAdvance(written,&buf,&len,"*** ERROR PostureEngine load corrupted - line %d\n",linenum)) return 0;
		while(*buf!='\n' && *buf!='\r')
			buf++;
		if(buf[0]=='\r' && buf[1]=='\n')
			buf+=2;
		else
			buf++;
		linenum++;
		//std::cout << '"' << jname << "\"\t" << (float)fval << '\t' << (float)fwht << std::endl;
		// we continue the search in order from where we left off - these are often
		// going to go in order, so might as well save a little time
		unsigned int startidx=idx;
		for(;idx<NumOutputs;idx++)
			if(strcmp(jname,outputNames[idx])==0) {
				cmds[idx].set(fval,fwht);
				break;
			}
		if(idx==NumOutputs) { //not found following startidx, look from beginning
			for(idx=0;idx<startidx;idx++)
				if(strcmp(jname,outputNames[idx])==0) {
					cmds[idx].set(fval,fwht);
					break;
			}
			if(idx==startidx && strcmp(jname,outputNames[idx])!=0) //not found at all
				std::cout << "*** WARNING '" << jname << "' is not a valid joint on this model." << std::endl;
		}
	}
	std::cout << "*** WARNING PostureEngine load missing #END" << std::endl;
	return origlen-len;
}

//Why do i need a cast to const char**??? It doesn't work without, should be automatic... grrrr
unsigned int PostureEngine::SaveBuffer(char buf[], unsigned int len) const {
	unsigned int origlen=len;
	int written=snprintf(buf,len,"#POS\n");
	if(!ChkAdvance(written,(const char**)&buf,&len,"*** ERROR PostureEngine save failed on header\n")) return 0;
	if(len==0 || len>origlen) {
		std::cout << "*** ERROR PostureEngine save overflow on header" << std::endl;
		return 0;
	}
	for(unsigned int i=0; i<NumOutputs; i++)
		if(cmds[i].weight>0) {
			written=snprintf(buf,len,"%s\t% .4f\t% .4f\n",outputNames[i],cmds[i].value,cmds[i].weight);
			if(!ChkAdvance(written,(const char**)&buf,&len,"*** ERROR PostureEngine save failed\n")) return 0;
			if(len==0 || len>origlen) {
				std::cout << "*** ERROR PostureEngine save overflow" << std::endl;
				return 0;
			}
		}
	written=snprintf(buf,len,"#END\n");
	if(!ChkAdvance(written,(const char**)&buf,&len,"*** ERROR PostureEngine save failed on #END\n")) return 0;
	if(len==0 || len>origlen) {
		std::cout << "*** ERROR PostureEngine save overflow on #END" << std::endl;
		return 0;
	}
	return origlen-len;
}

unsigned int PostureEngine::LoadFile(const char filename[]) {
	return LoadSave::LoadFile(config->motion.makePath(filename).c_str());
}
unsigned int PostureEngine::SaveFile(const char filename[]) const {
	return LoadSave::SaveFile(config->motion.makePath(filename).c_str());
}

/*
NEWMAT::ReturnMatrix
Kinematics::getOrientation(unsigned int j) {
	unsigned int c=-1U,l=-1U;
	if(!lookup(j,c,l)) {
		NEWMAT::Matrix A(4,4);
		A<<ROBOOP::fourbyfourident;
		A.Release(); return A;
	}
	update(c,l);
	NEWMAT::Matrix R;
	NEWMAT::ColumnVector p;
	chains[c]->kine(R,p,j);
	R.Release(); return R;
}

NEWMAT::ReturnMatrix
Kinematics::getPosition(unsigned int j) {
	unsigned int c=-1U,l=-1U;
	if(!lookup(j,c,l)) {
		NEWMAT::Matrix A(4,4);
		A<<ROBOOP::fourbyfourident;
		A.Release(); return A;
	}
	update(c,l);
	NEWMAT::Matrix R;
	NEWMAT::ColumnVector p;
	chains[c]->kine(R,p,j);
	p.Release(); return p;
}
*/

bool
PostureEngine::solveLinkPosition(const NEWMAT::ColumnVector& Pobj, unsigned int j, const NEWMAT::ColumnVector& Plink) {
	unsigned int c=-1U,l=-1U;
	if(!lookup(j,c,l))
		return false;
	update(c,l);
	bool conv=false;
	NEWMAT::ColumnVector q=chains[c]->inv_kin_pos(Pobj,0,l,Plink,conv);
	for(unsigned int i=1; i<=l && i<=chainMaps[c].size(); i++)
		if(chainMaps[c][i]<NumOutputs)
			setOutputCmd(chainMaps[c][i],chains[c]->get_q(i));
	return conv;
}

bool
PostureEngine::solveLinkVector(const NEWMAT::ColumnVector& Pobj, unsigned int j, const NEWMAT::ColumnVector& Plink) {
	solveLinkPosition(Pobj,j,Plink);
	//This method is an approximation, not entirely precise or fast as it could be
	//Something to work on some more down the road! :)
	//(this method is shared with HeadPointerMC::lookAtPoint(x,y,z))
	NEWMAT::ColumnVector poE=baseToLink(j)*Pobj;
	if(poE.nrows()>3 && poE(4)!=0) {
		poE/=poE(4);
	}
	poE=poE.SubMatrix(1,3,1,1);
	NEWMAT::ColumnVector plE=Plink.SubMatrix(1,3,1,1);
	if(Plink.nrows()>3 && Plink(4)!=0)
		plE/=Plink(4);
	float plE2=plE.SumSquare();
	float plE_len=sqrt(plE2);
	float obj_comp_link=NEWMAT::DotProduct(plE,poE)/plE_len;
	if(obj_comp_link<plE_len)
		obj_comp_link=obj_comp_link*.975; //.975 is a bit of fudge - accounts for joints moving Plink when adjusting
	else
		obj_comp_link=obj_comp_link/.975; //.975 is a bit of fudge - accounts for joints moving Plink when adjusting
	NEWMAT::ColumnVector obj_proj_link(4);
	obj_proj_link.SubMatrix(1,3,1,1)=obj_comp_link*plE/plE_len;
	obj_proj_link(4)=1;
	return solveLinkPosition(Pobj,j,obj_proj_link);
}

void
PostureEngine::update(unsigned int c, unsigned int l) {
	for(unsigned int j=1; j<=l; j++) {
		unsigned int tkout=chainMaps[c][j];
		if(tkout<NumOutputs)
			chains[c]->set_q(getOutputCmd(tkout).value,j);
	}
}

/*! @file
 * @brief Implements PostureEngine, a base class for managing the values and weights of all the outputs
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_2_2 $
 * $Revision: 1.23 $
 * $State: Exp $
 * $Date: 2004/12/23 00:59:03 $
 */

