//-*-c++-*-
#ifndef INCLUDED_SpiderMachineBehavior_h_
#define INCLUDED_SpiderMachineBehavior_h_

#include "Behaviors/BehaviorBase.h"
#include <queue>

class StateNode;

//! When active and connected over network socket, outputs structure of requested state machine(s)
/*! The protocol is:
 *  - '<tt>list</tt>' - send list of all instantiated StateNodes
 *  - '<tt>spider </tt><i>name</i>' - spider the current structure of StateNode named <i>name</i>
 *  - '<tt>listen </tt><i>name</i>' - send updates regarding the activation status of <i>name</i> and its subnodes; you can specify a state which is not yet running
 *  - '<tt>ignore </tt><i>name</i>' - cancels a previous listen command
 *  - '<tt>clear</tt>' - cancels all previous listen commands; should be called at the beginning or end of each connection, preferably both
 *  
 *  Each of those commands should be terminated with a newline -
 *  i.e. one command per line
 *
 *  After a <tt>list</tt> command, the first line will be the number
 *  of StateNodes, followed by that number of lines, one StateNode
 *  name per line.
 *
 *  After a <tt>spider</tt> command, an XML description of the model
 *  will be sent.  If no matching StateNode is found, an warning will
 *  be displayed on #serr, and an empty model
 *  ("<model></model>") returned over the network
 *  connection.
 *
 *  All other commands give no direct response - listen can be
 *  executed before the specified StateNode is yet running, and ignore
 *  doesn't care whether or not the specified StateNode was actually
 *  being listened for.
 *
 *  The format of the model is:
 @verbatim
 <!DOCTYPE model [
 <!ELEMENT model (state*, transition*)>
 <!ELEMENT state (state*, transition*)>
 <!ELEMENT transition (source+, dest+)>
 <!ELEMENT source (#PCDATA)>
 <!ELEMENT dest (#PCDATA)>
 
 <!ATTLIST state id CDATA #REQUIRED>
 <!ATTLIST state class CDATA #REQUIRED>
 <!ATTLIST transition id CDATA #REQUIRED>
 <!ATTLIST transition class CDATA #REQUIRED>
 ]>@endverbatim
 *
 *  The format of status updates following a listen command is:
 @verbatim
 <!DOCTYPE event [
 <!ELEMENT event (fire*, statestart*, statestop*)>
 <!ELEMENT fire (EMPTY)>
 <!ELEMENT statestart (EMPTY)>
 <!ELEMENT statestop (EMPTY)>

 <!ATTLIST fire id CDATA #REQUIRED>
 <!ATTLIST fire time CDATA #REQUIRED>
 <!ATTLIST statestart id CDATA #REQUIRED>
 <!ATTLIST statestart time CDATA #REQUIRED>
 <!ATTLIST statestop id CDATA #REQUIRED>
 <!ATTLIST statestop time CDATA #REQUIRED>
 ]>@endverbatim
*/
class SpiderMachineBehavior : public BehaviorBase {
public:	
	//! Points to the one SpiderMachineBehavior object that the input command stream is talking to.
	/*! A kludge. Dunno how you're gonna make sure you're not using this uninitialized. */
	static SpiderMachineBehavior * theOne;
	static unsigned int port; //!< the port to listen on (10080 by default)
	static int callback(char *buf, int bytes); //!< called by wireless when there's new data

public:
	//! constructor
	SpiderMachineBehavior() : BehaviorBase("SpiderMachineBehavior"), cmdsock(NULL), expected(), listen(), queuedEvents() {}

	virtual void DoStart();
	virtual void DoStop();
	virtual void processEvent(const EventBase& e);

	//! dumps all of the transitions and subnodes of a given statenode
	void spider(const StateNode* n, unsigned int depth=0);

	//! returns true iff @a n or one of its parents is found in #listen
	bool isListening(const StateNode* n);

	static std::string getClassDescription() {
		char tmp[20];
		sprintf(tmp,"%d",port);
		return std::string("When active and connected over network socket on port ")+tmp
			+std::string(", outputs structure of requested state machine(s)");
	}
	virtual std::string getDescription() const { return getClassDescription(); }
	
	//! parses commands sent from callback()
	void runCommand(const std::string& s);

protected:
	//!just to prettify the data sent out - probably should make this a null-op to save bandwidth after debugging is done
	void indent(unsigned int level);

	//!searches currently instantiated StateNodes to find the one named @a name
	const StateNode * find(const std::string& name);

	class Socket *cmdsock; //!< the socket for communication

	typedef std::set<BehaviorBase*> registry_t; //!< the type of the behavior registry (BehaviorBase::registry)
	typedef std::multiset<const StateNode*> expected_t; //!< the type of #ignore
	typedef std::set<std::string> listen_t; //!< the type of #listen
	typedef std::queue<EventBase> queuedEvents_t; //!< the type of #queuedEvents

	expected_t expected; //!< a set of behaviors which are involved with an impending transition - their next stateMachineEGID event should be ignored

	listen_t listen; //!< a set of state machine names which should have their subnodes monitored

	queuedEvents_t queuedEvents; //!< used if a transition causes other transitions, those transitions need to be remembered

private:
	SpiderMachineBehavior(const SpiderMachineBehavior&); //!< don't call
	SpiderMachineBehavior operator=(const SpiderMachineBehavior&); //!< don't call
};

/*! @file
 * @brief Defines SpiderMachineBehavior, which active and connected over network socket, outputs structure of requested state machine(s)
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_2_2 $
 * $Revision: 1.7 $
 * $State: Exp $
 * $Date: 2004/12/23 01:47:07 $
 */

#endif
