//-*-c++-*-
#ifndef INCLUDED_StareAtPawBehavior2_h_
#define INCLUDED_StareAtPawBehavior2_h_

#include "Behaviors/BehaviorBase.h"
#include "Motion/PIDMC.h"
#include "Motion/HeadPointerMC.h"
#include "Motion/PostureMC.h"
#include "Motion/MotionManager.h"
#include "Shared/SharedObject.h"
#include "Motion/roboop/robot.h"
#include "Shared/Config.h"
#include "Motion/Kinematics.h"


//! Uses new-style ROBOOP kinematics to track the paw which last received a button press with the camera
class StareAtPawBehavior2 : public BehaviorBase {
public:
	//! constructor
	StareAtPawBehavior2()
		: BehaviorBase("StareAtPawBehavior2"), lastLeg(LFrLegOrder), pointID(MotionManager::invalid_MC_ID)
	{ }

	virtual void DoStart() {
		BehaviorBase::DoStart(); // do this first
		pointID=motman->addPersistentMotion(SharedObject<HeadPointerMC>());
		erouter->addListener(this,EventBase::sensorEGID);
		erouter->addListener(this,EventBase::buttonEGID);
		if(state->robotDesign == WorldState::ERS7Mask)
			sout->printf("NOTICE: The ERS-7 has a rather \"sticky\" nod joint\n"
									 "(the upper tilt joint).  This can cause it to hesitate\n"
									 "or altogether fail to precisely center the target position\n"
									 "vertically in the center of the image...\n");
	}

	virtual void DoStop() {
		motman->removeMotion(pointID);
		pointID=MotionManager::invalid_MC_ID;
		erouter->removeListener(this);
		BehaviorBase::DoStop(); // do this last
	}

	virtual void processEvent(const EventBase& e) {
		if(e.getGeneratorID()==EventBase::buttonEGID) {

			if(e.getSourceID()==LFrPawOffset) {
				lastLeg=LFrLegOrder;
			} else if(e.getSourceID()==RFrPawOffset) {
				lastLeg=RFrLegOrder;
			} else
				return;
			if(e.getTypeID()==EventBase::activateETID) {
				unsigned int lastlegoff=LegOffset+lastLeg*JointsPerLeg;
				SharedObject<PIDMC> relaxLeg(lastlegoff,lastlegoff+JointsPerLeg,0);
				motman->addPrunableMotion(relaxLeg);
			} else if(e.getTypeID()==EventBase::deactivateETID) {
				unsigned int lastlegoff=LegOffset+lastLeg*JointsPerLeg;
				SharedObject<PIDMC> tightLeg(lastlegoff,lastlegoff+JointsPerLeg,1);
				motman->addPrunableMotion(tightLeg);
			}

		} else if(e.getGeneratorID()==EventBase::sensorEGID) {

			//Which paw do we want?
			const char * ipname= (lastLeg==LFrLegOrder) ? "ToeLFrPaw" : "ToeRFrPaw";

			//Ask kinematics for current location of that paw (this is the "objective" aka target)
			NEWMAT::ColumnVector Pobj=kine->getJointInterestPoint(BaseFrameOffset,ipname);

			//Now point the head there
			MMAccessor<HeadPointerMC>(pointID)->lookAtPoint(Pobj(1),Pobj(2),Pobj(3)); //keep head as far away as possible
			//Alternative method:
			// MMAccessor<HeadPointerMC>(pointID)->lookAtPoint(Pobj(1),Pobj(2),Pobj(3),80); //keep head 80mm away

		} else {
			serr->printf("StareAtPawBehavior2: Unhandled event %s\n",e.getName().c_str());
		}
	}

	static std::string getClassDescription() { return "Uses kinematics to track the paw which last received a button press with the camera"; }
	virtual std::string getDescription() const { return getClassDescription(); }
	
protected:
	LegOrder_t lastLeg; //!< last leg to have it's button pressed, i.e. the one we are looking at
	MotionManager::MC_ID pointID; //!< the HeadPointerMC we are using to do the looking
};

/*! @file
 * @brief Defines StareAtPawBehavior2, which uses new-style ROBOOP kinematics to track the paw which last received a button press with the camera
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_2_2 $
 * $Revision: 1.18 $
 * $State: Exp $
 * $Date: 2004/12/23 01:47:07 $
 */

#endif
