//-*-c++-*-
#ifndef INCLUDED_BehaviorSwitchControl_h_
#define INCLUDED_BehaviorSwitchControl_h_

#include "ControlBase.h"
#include "Behaviors/BehaviorBase.h"
#include "Shared/ReferenceCounter.h"
#include "Shared/Factory.h"
#include "Shared/debuget.h"

//! Holds some utility classes and functions for BehaviorSwitchControl which shouldn't be stored in a templated class
class BehaviorSwitchControlBase : public ControlBase {
public:
	//! A simple utility class to allow the BehaviorSwitchControl's to be able to deactivate the current behavior when a new one becomes active
	/*! Most behaviors are either major actions which you'll only want one of active
	 *  at a time, or else their background monitors of some sort, that can run in different
	 *  combinations.  Think radio buttons vs. checkboxes.  This will help you implement the
	 *  "radio button" style... just assign all the behaviors to the same group, they will
	 *  automatically use it to turn the previous behavior off when a new one becomes active.\n
	 *  Pass NULL instead of one of these to get checkbox-style. */
	class BehaviorGroup : public ReferenceCounter {
	public:
		BehaviorGroup() : curBehavior(NULL) { } //!< contructor
		~BehaviorGroup() { if(curBehavior!=NULL) curBehavior->DoStop(); } //!< destructor, will stop the current behavior if it was a one-shot
		BehaviorBase * curBehavior; //!< pointer to current behavior
	private:
		BehaviorGroup(const BehaviorGroup&); //!< shouldn't be called
		BehaviorGroup operator=(const BehaviorGroup&); //!< shouldn't be called
	};
	
	//! constructor
	BehaviorSwitchControlBase(const std::string& n, BehaviorBase* beh, BehaviorGroup* bg=NULL)
		: ControlBase(n), behgrp(bg), mybeh(beh) {
		if(behgrp!=NULL)
			behgrp->AddReference();
		if(mybeh!=NULL)
			mybeh->AddReference();
	}
	//! constructor
	BehaviorSwitchControlBase(BehaviorBase* beh, BehaviorGroup* bg=NULL)
		: ControlBase(), behgrp(bg), mybeh(beh) {
		if(behgrp!=NULL)
			behgrp->AddReference();
		mybeh->AddReference();
	}

	//! destructor
	virtual ~BehaviorSwitchControlBase() {
		//cout << "~BehaviorSwitchControlBase(): " << getName() << endl;
		if(mybeh!=NULL)
			stop();
		if(behgrp!=NULL) {
			behgrp->RemoveReference();
			behgrp=NULL;
		}
		if(mybeh!=NULL)
			mybeh->RemoveReference();
	}

	//! activates the behavior, handy for making start-up behaviors that you can turn off again with the Controller
	/*! If you start twice without stopping (ie it's already running), shouldn't do anything */
	virtual BehaviorSwitchControlBase* start() { if(!isRunning()) { stopother(); startmine(); } return this; }

	//! stops the behavior
	virtual BehaviorSwitchControlBase* stop() { if(isRunning()) stopother(); return this; }

	//! toggles the behavior
	virtual BehaviorSwitchControlBase* toggle() { if(isRunning()) stopother(); else { stopother(); startmine(); } return this; }

	//! tells the current behavior (if there is one) to stop then loads its own
	/*! @return NULL unless there are submenus */
	virtual ControlBase * activate(MotionManager::MC_ID display, Socket * gui) {
		if(slotsSize()==0) {
			toggle();
			return NULL;
		} else
			return ControlBase::activate(display,gui);
	}

	//! adds a status to the name: - if in memory, # if running
	virtual std::string getName() const {
		if(mybeh==NULL)
			return ControlBase::getName();
		return (mybeh->isActive()?'#':'-')+mybeh->getName();
	}
	virtual std::string getDescription() const {
		if(mybeh==NULL)
			return ControlBase::getDescription();
		return "Class "+mybeh->getClassName()+": "+mybeh->getDescription();
	}
	
protected:
	//! Stops the "other" guy's behavior - if ::behgrp is NULL, stops ourselves
	virtual void stopother() {
		if(behgrp==NULL) {
			if(mybeh->isActive())
				mybeh->DoStop();
		} else if(behgrp->curBehavior!=NULL) {
			behgrp->curBehavior->DoStop();
			behgrp->curBehavior=NULL;
		}
	}

	//! Starts our behavior
	virtual void startmine() {
		if(behgrp!=NULL)
			behgrp->curBehavior=mybeh;
		mybeh->DoStart();
	}

	//! Returns true if the associated behavior is running
	virtual bool isRunning() const {
		if(mybeh==NULL) //not created or has been destroyed, definitely not running
			return false;
		// so, beh has been created (but may have been stopped by another in the group)
		if(behgrp==NULL) //no group
			return mybeh->isActive(); //just check active flag (is valid object, we would have set it to NULL if we stopped it ourselves)
		// so, we're in a group, someone else could have stopped us
		return (behgrp->curBehavior==mybeh); //all we can see is if the current behavior is ours.  If it is, it'll be active
	}

	BehaviorGroup * behgrp; //!< the behavior group this belongs to.  Uses this to track the "current" behavior
	BehaviorBase* mybeh; //!< used to store the behavior.  If retained and non-NULL, will be valid.  However, if not retained, only valid if equals behgrp->curBehavior

private:
	BehaviorSwitchControlBase(const BehaviorSwitchControlBase&); //!< shouldn't copy these
	BehaviorSwitchControlBase operator=(const BehaviorSwitchControlBase&); //!< shouldn't assign these
};




//! Allows proper switching between major behaviors, calling DoStart and DoStop
template < class B, class Al = Factory< B > >
class BehaviorSwitchControl : public BehaviorSwitchControlBase {
public:
	//! constructor, can use this to toggle a single behavior on and off
	BehaviorSwitchControl(const std::string& n, bool retain=false)
		: BehaviorSwitchControlBase(n,NULL,NULL), retained(retain)
	{}
	//! constructor, if you want to use an already constructed behavior
	BehaviorSwitchControl(B* beh, BehaviorGroup* bg=NULL)
		: BehaviorSwitchControlBase(beh,bg), retained(true)
	{}
	//! constructor, if you want to use an already constructed behavior, but unretain it if it's stopped (if not retaining, will start @a beh if it's not already started)
	BehaviorSwitchControl(const std::string& n, B* beh, BehaviorGroup* bg=NULL, bool retain=false)
		: BehaviorSwitchControlBase(n,beh,bg), retained(retain)
	{
		if(!retained) {
			// have to make sure behavior is started to maintain invariants
			if(!mybeh->isActive()) {
				//keep reference from superclass's constructor in case mybeh stops itself right away
				mybeh->DoStart();
				bool stopped=!mybeh->isActive();
				mybeh->RemoveReference(); //cancels reference from BehaviorSwitchControlBase's constructor
				if(stopped)
					mybeh=NULL;
			} else
				mybeh->RemoveReference(); //cancels reference from BehaviorSwitchControlBase's constructor
		}
		if(behgrp!=NULL) {
			if(mybeh->isActive()) {
				if(behgrp->curBehavior!=NULL)
					behgrp->curBehavior->DoStop();
				behgrp->curBehavior=mybeh;
			} else if(!retained) {
				if(behgrp->curBehavior!=NULL)
					behgrp->curBehavior->DoStop();
				behgrp->curBehavior=NULL;
			}
		}
	}
	//! constructor, needs to know what group its in and whether to retain its behavior
	BehaviorSwitchControl(const std::string& n, BehaviorGroup* bg, bool retain=false)
		: BehaviorSwitchControlBase(n,NULL,bg), retained(retain)
	{}
	
	//! destructor
	virtual ~BehaviorSwitchControl() {
		stop();
		if(behgrp!=NULL) {
			behgrp->RemoveReference();
			behgrp=NULL;
		}
		if(mybeh!=NULL && retained)
			mybeh->RemoveReference();
		mybeh=NULL;
	}

	virtual std::string getName() const {
		if(!isValid())
			return ControlBase::getName();
		else
			return BehaviorSwitchControlBase::getName();
	}
	virtual std::string getDescription() const {
		if(!isValid() || mybeh->getDescription().size()==0)
			return B::getClassDescription();
		else
			return BehaviorSwitchControlBase::getDescription();
	}
	

protected:

	virtual void stopother() {
		if(behgrp==NULL) {
			if(mybeh!=NULL) {
				if(mybeh->isActive()) {
					mybeh->DoStop();
					if(!retained)
						mybeh=NULL;
				} else
					ASSERT(retained,"null group, non-null not retained beh, not active, did you call inherited DoStart/DoStop in your Behavior?");
			}
		} else if(behgrp->curBehavior!=NULL) {
			behgrp->curBehavior->DoStop();
			if(behgrp->curBehavior==mybeh)
				mybeh=NULL;
			behgrp->curBehavior=NULL;
		}
	}

	virtual void startmine() {
		if(!retained) {
			mybeh=Al::construct();
			mybeh->setName(getName());
			if(behgrp!=NULL)
				behgrp->curBehavior=mybeh;
		} else {
			if(mybeh==NULL) {
				mybeh=Al::construct();
				mybeh->setName(getName());
				mybeh->AddReference();
			}
			if(behgrp!=NULL)
				behgrp->curBehavior=mybeh;
		}
		mybeh->AddReference(); //temporary reference in case mybeh stops itself right away
		mybeh->DoStart();
		bool stopped=(!mybeh->isActive() && !retained);
		mybeh->RemoveReference();
		if(stopped) {
			if(behgrp!=NULL && behgrp->curBehavior==mybeh)
				behgrp->curBehavior=NULL;
			mybeh=NULL;
		}
	}

	//! Returns true if mybeh is pointing to a valid object
	virtual bool isValid() const {
		if(isRunning())
			return true;
		return retained;
	}

private:
	bool retained; //!< true if the behavior should be generated once and retained after DoStop.  Otherwise, a new one is generated each time it is started
	BehaviorSwitchControl(const BehaviorSwitchControl&); //!< shouldn't call this
	BehaviorSwitchControl operator=(const BehaviorSwitchControl&); //!<shouldn't call this
};

/*! @file
 * @brief Defines BehaviorSwitchControl and the BehaviorSwitch namespace - a control for turning behaviors on and off
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_2_2 $
 * $Revision: 1.14 $
 * $State: Exp $
 * $Date: 2004/11/12 00:05:03 $
 */

#endif
