//-*-c++-*-
#ifndef INCLUDED_InterleavedYUVGenerator_h_
#define INCLUDED_InterleavedYUVGenerator_h_

#include "Vision/FilterBankGenerator.h"

//! Generates FilterBankEvents containing raw camera images with interleaved pixels (YVUYVUYVU... instead of YYY...UUU...VVV...)
/*! @note That's not a typo - the byte ordering is YVU, @e not YUV
 *
 *  Sorry about the misleading name... This @a takes filter banks
 *  containing YUV information, however, the main reason for this
 *  class is to interleave the image in order to pass it to the jpeg
 *  compression routines.  JPEG expect YCbCr, which corresponds to YVU
 *  order.
 *
 *  Also, I should point out that mathematically, V!=Cb, and U!=Cr,
 *  but they do carry the same information.  It's just a matter of
 *  scaling and offset.  These comments use the "Y", "U", and "V"
 *  labels loosely.
 *
 *  There's only one channel, which holds the interleaved data.  The
 *  increment is set to 3, but if you want to access each component in
 *  order, just use 1 instead (as you would expect hopefully, since
 *  that's the whole point of this class)
 *  
 *  The generated events use 0 for their event source IDs.  The row
 *  skip is always 0, and the row stride is always width*3.  But it
 *  would be better to use the proper accessor functions to be more
 *  general.
 *
 *	should receive FilterBankEvents from any standard format FilterBankGenerator (like RawCameraGenerator)
 *
 *  @see FilterBankGenerator for information on serialization format
 */
class InterleavedYUVGenerator : public FilterBankGenerator {
public:
	//! constructor
	InterleavedYUVGenerator(EventBase::EventGeneratorID_t gid, unsigned int sid, unsigned int mysid);
	//! constructor, you can pass which channels to interleave
	InterleavedYUVGenerator(EventBase::EventGeneratorID_t gid, unsigned int sid, unsigned int mysid, unsigned int syc, unsigned int suc, unsigned int svc);

	//! destructor
	virtual ~InterleavedYUVGenerator() {
		freeCaches();
		destruct();
	}

	static const unsigned int CHAN_YUV=0; //!< so you can refer to the YUV channel symbolically.

	static std::string getClassDescription() { return "Converts a FilterBankGenerator's data into interleaved format"; }

	//! should receive FilterBankEvents from any standard format FilterBankGenerator (like RawCameraGenerator)
	virtual void processEvent(const EventBase& event);
	
	virtual unsigned int getBinSize() const;

	virtual unsigned int LoadBuffer(const char buf[], unsigned int len);

	virtual unsigned int SaveBuffer(char buf[], unsigned int len) const;

protected:
	virtual void setNumImages(unsigned int nLayers, unsigned int nChannels);
	virtual void setDimensions(); //!< resets stride parameter (to correspond to width*3 from FilterBankGenerator::setDimensions())
	virtual unsigned char * createImageCache(unsigned int layer, unsigned int chan) const;
	virtual void calcImage(unsigned int layer, unsigned int chan) const;

	unsigned int srcYChan; //!< the channel of the source's Y channel
	unsigned int srcUChan; //!< the channel of the source's U channel
	unsigned int srcVChan; //!< the channel of the source's V channel

private:
	InterleavedYUVGenerator(const InterleavedYUVGenerator& fbk); //!< don't call
	const InterleavedYUVGenerator& operator=(const InterleavedYUVGenerator& fbk); //!< don't call
};

/*! @file 
 * @brief Describes InterleavedYUVGenerator, which generates FilterBankEvents containing raw camera images with interleaved pixels (YUVYUVYUV... instead of YYY...UUU...VVV...)
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_2_1 $
 * $Revision: 1.11 $
 * $State: Exp $
 * $Date: 2004/02/18 21:13:32 $
 */

#endif
