//
// Copyright 2002,2003 Sony Corporation 
//
// Permission to use, copy, modify, and redistribute this software for
// non-commercial use is hereby granted.
//
// This software is provided "as is" without warranty of any kind,
// either expressed or implied, including but not limited to the
// implied warranties of fitness for a particular purpose.
//

// Additional modifications for Tekkotsu framework

#include "SoundPlay/entry.h"
#include <OPENR/OPENRAPI.h>
#include <OPENR/OSyslog.h>
#include <OPENR/core_macro.h>
#include "SoundPlay.h"

#include <iostream>
#include "SoundPlay/SoundManager.h"
#include "Shared/RobotInfo.h"
#include "Shared/Config.h"
#include "Shared/debuget.h"
#include "Events/EventRouter.h"

SoundPlay::SoundPlay()
	: active(0), soundManagerMemRgn(NULL), eventTranslatorQueueMemRgn(NULL), etrans(), speakerID(oprimitiveID_UNDEF)
{
	for (unsigned int i = 0; i < SOUND_NUM_BUFFER; i++)
		region[i] = 0;
}

OStatus
SoundPlay::DoInit(const OSystemEvent&)
{
	//OSYSDEBUG(("SoundPlay::DoInit()\n"));

	NEW_ALL_SUBJECT_AND_OBSERVER;
	REGISTER_ALL_ENTRY;
	SET_ALL_READY_AND_NOTIFY_ENTRY;

	observer[obsSoundManagerComm]->SetBufCtrlParam(0,1,SoundManager::MAX_SND+1);
	//+1 to MAX_SND so that we can still get a delete message after we've filled up

	//Set process ID
	ProcessID::setID(ProcessID::SoundProcess);

	//Read config file
	config=new Config("/ms/config/tekkotsu.cfg");

	erouter = new EventRouter;

	//soundManagerMemRgn -> sndman setup
	soundManagerMemRgn = InitRegion(sizeof(SoundManager));
	sndman=new ((SoundManager*)soundManagerMemRgn->Base()) SoundManager;
	sndman->InitAccess(subject[sbjSoundManagerComm]);
	for(unsigned int i=0; i<config->sound.preload.size(); i++)
		sndman->LoadFile(config->sound.preload[i].c_str());

	OpenSpeaker();
	NewSoundVectorData();
	SetPowerAndVolume();

	return oSUCCESS;
}

OStatus
SoundPlay::DoStart(const OSystemEvent&)
{
	//OSYSDEBUG(("SoundPlay::DoStart()\n"));

	ENABLE_ALL_SUBJECT;
	ASSERT_READY_TO_ALL_OBSERVER;

	return oSUCCESS;
}

OStatus
SoundPlay::DoStop(const OSystemEvent&)
{
	//OSYSDEBUG(("SoundPlay::DoStop()\n"));

	sndman->StopPlay();

	DISABLE_ALL_SUBJECT;
	DEASSERT_READY_TO_ALL_OBSERVER;

	return oSUCCESS;
}

OStatus
SoundPlay::DoDestroy(const OSystemEvent&)
{
	for(unsigned int i=0; i<config->sound.preload.size(); i++)
		sndman->ReleaseFile(config->sound.preload[i].c_str());
	delete erouter;
	eventTranslatorQueueMemRgn->RemoveReference();
	DELETE_ALL_SUBJECT_AND_OBSERVER;
	return oSUCCESS;
}

void
SoundPlay::ReadySendSound(const OReadyEvent&)
{
	//	OSYSDEBUG(("SoundPlay::Ready()\n"));
	doSendSound();
}

void
SoundPlay::ReadyRegisterSoundManager(const OReadyEvent&) {
	static bool is_init=true;
	if(is_init) {
		is_init=false;
		cout << objectName << " Registering SoundManager" << endl;
		subject[sbjRegisterSoundManager]->SetData(soundManagerMemRgn);
		subject[sbjRegisterSoundManager]->NotifyObservers();
	}
}

void
SoundPlay::GotEventTranslatorQueue(const ONotifyEvent& event){
	std::cout << "SoundPlay-GOTEventTranslatorQueue..." << std::flush;
	ASSERT(event.NumOfData()==1,"Too many EventTranslatorQueue");
	eventTranslatorQueueMemRgn = event.RCData(0);
	eventTranslatorQueueMemRgn->AddReference();
	etrans.setQueue(reinterpret_cast<EventTranslator::Queue_t*>(eventTranslatorQueueMemRgn->Base()));
	erouter->addTrapper(&etrans);
	observer[obsReceiveEventTranslatorQueue]->AssertReady();
	cout << "done" << endl;
}

void
SoundPlay::GotSoundMsg(const ONotifyEvent& event) {
	//	OSYSDEBUG(("SoundPlay::GotSoundMsg()\n"));
	sndman->ReceivedMsg(event);
	unsigned int curact=sndman->GetNumPlaying();
	//	std::cout << "got-active=" << active << " cur=" << curact << std::endl;
	if(active==0 && curact>0) {
		active=curact;
		doSendSound();
	}
	observer[obsSoundManagerComm]->AssertReady();
}

void
SoundPlay::doSendSound() {
	//	std::cout << "do-active=" << active << std::endl;
	static unsigned int bufInUse=0;

	active=sndman->GetNumPlaying();
	if(active==0) { //if we don't have any active channels, don't send a buffer
		if(bufInUse>0)
			bufInUse--;
		return;
	}

	RCRegion* rgn = FindFreeRegion();
	if(rgn==NULL) //this can happen if a wakeup message comes in right after a sound stopped
		return;
	active=sndman->CopyTo(reinterpret_cast<OSoundVectorData*>(rgn->Base()));
	subject[sbjSpeaker]->SetData(rgn);

	if(bufInUse<SOUND_NUM_BUFFER-1) {
		bufInUse++;
		doSendSound();
	} else //recursive base case
		subject[sbjSpeaker]->NotifyObservers();
}

void
SoundPlay::OpenSpeaker()
{
	OStatus result = OPENR::OpenPrimitive(SpeakerLocator, &speakerID);
	if (result != oSUCCESS)
		OSYSLOG1((osyslogERROR, "%s : %s %d","SoundPlay::OpenSpeaker()","OPENR::OpenPrimitive() FAILED", result));
}

void
SoundPlay::NewSoundVectorData()
{
	OStatus result;
	MemoryRegionID    soundVecDataID;
	OSoundVectorData* soundVecData;

	/*soundUnitSize (bytes/buffer) = sample_rate (samples/sec)
	 *                               * sample_bits (bits/sample)
	 *                               * [1/8] (bytes/bit)
	 *                               * SoundBufferTime (ms/buffer)
	 *                               * [1/1000] (sec/ms)
	 */
	size_t soundUnitSize = config->sound.sample_rate*config->sound.sample_bits/8*SoundBufferTime/1000;

	for (unsigned int i = 0; i < SOUND_NUM_BUFFER; i++) {
		result = OPENR::NewSoundVectorData(1, soundUnitSize,&soundVecDataID, &soundVecData);
		if (result != oSUCCESS) {
			OSYSLOG1((osyslogERROR, "%s : %s %d","SoundPlay::NewSoundVectorData()","OPENR::NewSoundVectorData() FAILED", result));
			return;
		}

		soundVecData->SetNumData(1);
		OSoundInfo* sinfo = soundVecData->GetInfo(0);
		sinfo->Set(odataSOUND_VECTOR,speakerID,soundUnitSize);
		sinfo->dataSize      = soundUnitSize;
		sinfo->format        = osoundformatPCM;
		sinfo->channel       = osoundchannelMONO;
		sinfo->samplingRate  = config->sound.sample_rate;
		sinfo->bitsPerSample = config->sound.sample_bits;

		region[i] = new RCRegion(soundVecData->vectorInfo.memRegionID,soundVecData->vectorInfo.offset,(void*)soundVecData,soundVecData->vectorInfo.totalSize);
	}
}

void
SoundPlay::SetPowerAndVolume()
{
	OStatus result;

	result = OPENR::ControlPrimitive(speakerID,oprmreqSPEAKER_MUTE_OFF, 0, 0, 0, 0);
	if (result != oSUCCESS)
		OSYSLOG1((osyslogERROR, "%s : %s %d","SoundPlay::SetPowerAndVolume()","OPENR::ControlPrimitive(SPEAKER_MUTE_OFF) FAILED", result));

	OPrimitiveControl_SpeakerVolume volume(config->sound.volume);
	result = OPENR::ControlPrimitive(speakerID,oprmreqSPEAKER_SET_VOLUME,&volume, sizeof(volume), 0, 0);
	if (result != oSUCCESS)
		OSYSLOG1((osyslogERROR, "%s : %s %d","SoundPlay::SetPowerAndVolume()","OPENR::ControlPrimitive(SPEAKER_SET_VOLUME) FAILED",result));

	if (config->sound.sample_rate == 16000 && config->sound.sample_bits == 16) {
		OPrimitiveControl_SpeakerSoundType soundType(ospksndMONO16K16B);
		result = OPENR::ControlPrimitive(speakerID,oprmreqSPEAKER_SET_SOUND_TYPE,&soundType, sizeof(soundType), 0, 0);
		if (result != oSUCCESS)
			OSYSLOG1((osyslogERROR, "%s : %s %d","SoundPlay::SetPowerAndVolume()","OPENR::ControlPrimitive(SPEAKER_SET_SOUND_TYPE) FAILED",result));
	}
}

/*! Will round up size to the nearest page */
RCRegion*
SoundPlay::InitRegion(unsigned int size) {
	unsigned int pagesize=4096;
	sError err=GetPageSize(&pagesize);
	ASSERT(err==sSUCCESS,"Error "<<err<<" getting page size");
	unsigned int pages=(size+pagesize-1)/pagesize;
	return new RCRegion(pages*pagesize);
}

RCRegion*
SoundPlay::FindFreeRegion()
{
	for (unsigned int i = 0; i < SOUND_NUM_BUFFER; i++)
		if (region[i]->NumberOfReference() == 1)
			return region[i];
	return 0;
}

/*! @file
 * @brief Implements the SoundPlay process (a.k.a. OObject), which is responsible for sending sound buffers to the system to play
 * @author Sony (Creator)
 *
 * This is basically the SoundPlay example from the Sony code, with a few modifications.
 * Here's the license Sony provided with it:
 *
 * Copyright 2002,2003 Sony Corporation 
 *
 * Permission to use, copy, modify, and redistribute this software for
 * non-commercial use is hereby granted.
 *
 * This software is provided "as is" without warranty of any kind,
 * either expressed or implied, including but not limited to the
 * implied warranties of fitness for a particular purpose.
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_2_1 $
 * $Revision: 1.13 $
 * $State: Exp $
 * $Date: 2004/01/16 23:55:32 $
 */

