#include "Shared/Config.h"
#include "SoundManager.h"
#include "Shared/LockScope.h"
#include "WAV.h"
#include "Events/EventRouter.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fstream>
#include <OPENR/OSubject.h>
#include <OPENR/ObjcommEvent.h>


SoundManager * sndman=NULL;

//!for convenience when locking each of the functions
typedef LockScope<ProcessID::NumProcesses> AutoLock;

SoundManager::SoundManager()
	: mixerBuffer(0), mixerBufferSize(0),
		sndlist(),playlist(),chanlist(),mix_mode(Fast),queue_mode(Override),max_chan(4),lock()
{
}

SoundManager::~SoundManager() {
	delete[] mixerBuffer;
}

void
SoundManager::InitAccess(OSubject* subj) {
	subjs[ProcessID::getID()]=subj;
}

//!@todo this does one more copy than it really needs to
SoundManager::Snd_ID
SoundManager::LoadFile(std::string const &name) {
	AutoLock autolock(lock,ProcessID::getID());
	if (name.size() == 0) {
	  cout << "SoundManager::LoadFile() null filename" << endl;
	  return invalid_Snd_ID;
	};
	std::string path(config->sound.makePath(name));
	Snd_ID id=lookupPath(path);
	if(id!=invalid_Snd_ID) {
		//		cout << "add reference to pre-existing" << endl;
		sndlist[id].ref++;
	} else {
		//		cout << "load new file" << endl;
		struct stat buf;
		if(stat(path.c_str(),&buf)==-1) {
			cout << "SoundManager::LoadFile(): Sound file not found: " << path << endl;
			return invalid_Snd_ID;
		}
		byte * sndbuf=new byte[buf.st_size];
		std::ifstream file(path.c_str());
		file.read(reinterpret_cast<char*>(sndbuf),buf.st_size);
		WAV wav;
		WAVError error = wav.Set(sndbuf);
		if (error != WAV_SUCCESS) {
			OSYSLOG1((osyslogERROR, "%s : %s %d: '%s'","SoundManager::LoadFile()","wav.Set() FAILED",
				  error, path.c_str()));
			return invalid_Snd_ID;
		}
		if(wav.GetSamplingRate()!=config->sound.sample_rate || wav.GetBitsPerSample()!=config->sound.sample_bits) {
			OSYSLOG1((osyslogERROR, "%s : %s %s","SoundManager::LoadFile()","bad sample rate/bits", error));
			return invalid_Snd_ID;
		}
		id=LoadBuffer(reinterpret_cast<char*>(wav.GetDataStart()),wav.GetDataEnd()-wav.GetDataStart());
		delete [] sndbuf;
		strncpy(sndlist[id].name,path.c_str(),MAX_NAME_LEN);
	}
	return id;
}

SoundManager::Snd_ID
SoundManager::LoadBuffer(const char buf[], unsigned int len) {
	cout << "SoundManager::LoadBuffer() of " << len << " bytes" << endl;
	if(buf==NULL || len==0)
		return invalid_Snd_ID;
	AutoLock autolock(lock,ProcessID::getID());
	//setup region
	RCRegion * region=initRegion(len+SoundManagerMsg::MSG_SIZE);
	//setup message
	SoundManagerMsg * msg=new (reinterpret_cast<SoundManagerMsg*>(region->Base())) SoundManagerMsg;
	msg->setAdd(sndlist.new_front());
	//init sound structure
	sndlist[msg->getID()].rcr=NULL;  // set by SoundPlay upon reception
	sndlist[msg->getID()].data=NULL; // set by SoundPlay upon reception
	sndlist[msg->getID()].ref=1;
	sndlist[msg->getID()].len=len;
	//copy buffer, do any filtering needed here
	const byte* src=reinterpret_cast<const byte*>(buf);
	byte* dest=reinterpret_cast<byte*>(region->Base())+SoundManagerMsg::MSG_SIZE;
	byte* end=dest+len;
	if (config->sound.sample_bits==8)
		while (dest < end)
			*dest++ = *src++ ^ 0x80; // offset binary -> signed char 
	else
		while (dest < end)
			*dest++ = *src++;
	//send message
	if(ProcessID::getID()==ProcessID::SoundProcess) {
		//if SoundPlay is preloading files, don't need to do inter-object comm
		sndlist[msg->getID()].rcr=region;
		sndlist[msg->getID()].data=reinterpret_cast<byte*>(region->Base()+SoundManagerMsg::MSG_SIZE);		
	} else {
		subjs[ProcessID::getID()]->SetData(region);
		subjs[ProcessID::getID()]->NotifyObservers();
	}
	return msg->getID();
}
	
void
SoundManager::ReleaseFile(std::string const &name) {
	AutoLock autolock(lock,ProcessID::getID());
	Release(lookupPath(name));
}

void
SoundManager::Release(Snd_ID id) {
	if(id==invalid_Snd_ID)
		return;
	if(sndlist[id].ref==0) {
		cout << "SoundManager::Release() " << id << " extra release" << endl;
		return;
	}
	AutoLock autolock(lock,ProcessID::getID());
	sndlist[id].ref--;
	if(sndlist[id].ref==0) {
		//cout << "releasing snd_id " << id << endl;
		//setup region
		RCRegion * region=initRegion(SoundManagerMsg::MSG_SIZE);
		//setup message
		SoundManagerMsg * msg=new (reinterpret_cast<SoundManagerMsg*>(region->Base())) SoundManagerMsg;
		msg->setDelete(sndlist[id].rcr);
		//clean up sound data structure
		sndlist.erase(id);
		//send message
		if(ProcessID::getID()==ProcessID::SoundProcess) {
			msg->region->RemoveReference();
		} else {
			subjs[ProcessID::getID()]->SetData(region);
			subjs[ProcessID::getID()]->NotifyObservers();
		}
	}
}

SoundManager::Play_ID
SoundManager::PlayFile(std::string const &name) {
  if(playlist.size()>=playlist_t::MAX_ENTRIES)
		return invalid_Play_ID;	
	AutoLock autolock(lock,ProcessID::getID());
	Snd_ID sndid=LoadFile(name);
	if(sndid==invalid_Snd_ID)
		return invalid_Play_ID;
	sndlist[sndid].ref--;
	return Play(sndid);
}

SoundManager::Play_ID
SoundManager::PlayBuffer(const char buf[], unsigned int len) {
	if(playlist.size()>=playlist_t::MAX_ENTRIES || buf==NULL || len==0)
		return invalid_Play_ID;	
	AutoLock autolock(lock,ProcessID::getID());
	Snd_ID sndid=LoadBuffer(buf,len);
	if(sndid==invalid_Snd_ID)
		return invalid_Play_ID;
	sndlist[sndid].ref--;
	return Play(sndid);
}
	
SoundManager::Play_ID
SoundManager::Play(Snd_ID id) {
	//	cout << "Play " << id << endl;
	if(id==invalid_Snd_ID)
		return invalid_Play_ID;
	AutoLock autolock(lock,ProcessID::getID());
	Play_ID playid=playlist.new_front();
	if(playid!=invalid_Play_ID) {
		sndlist[id].ref++;
		playlist[playid].snd_id=id;
		playlist[playid].offset=0;
		//playlist.size() should be greater than or equal to chanlist.size
		//so if we got a playid, we can get a channel slot.
		chanlist.push_front(playid);

		//setup message to "wake-up" 
		//(only really need if chanlist was empty)
		//		if(chanlist.size()==1) { //commented out because sometimes doesn't wake up, thinks it's playing but isn't
		if(ProcessID::getID()!=ProcessID::SoundProcess) {
			RCRegion * region=initRegion(SoundManagerMsg::MSG_SIZE);
			ASSERT(region!=NULL,"initRegion returned NULL");
			SoundManagerMsg * msg=new (reinterpret_cast<SoundManagerMsg*>(region->Base())) SoundManagerMsg;
			msg->setWakeup();
			subjs[ProcessID::getID()]->SetData(region);
			subjs[ProcessID::getID()]->NotifyObservers();
		}
		//		}
		
		if(sndlist[id].rcr!=NULL)
			erouter->postEvent(EventBase::audioEGID,playid,EventBase::activateETID,0);
	}
	return playid;
}
	
SoundManager::Play_ID
SoundManager::ChainFile(Play_ID base, std::string const &next) {
       if(base==invalid_Play_ID)
		return base;
	Play_ID orig=base;
	while(playlist[base].next_id!=invalid_Play_ID)
		base=playlist[base].next_id;
	Play_ID nplay=playlist.new_front();
	if(nplay==invalid_Play_ID)
		return nplay;
	Snd_ID nsnd=LoadFile(next);
	if(nsnd==invalid_Snd_ID) {
		playlist.pop_front();
		return invalid_Play_ID;
	}
	playlist[nplay].snd_id=nsnd;
	playlist[base].next_id=nplay;
	return orig;
}

SoundManager::Play_ID
SoundManager::ChainBuffer(Play_ID base, const char buf[], unsigned int len) {
	if(base==invalid_Play_ID || buf==NULL || len==0)
		return base;
	Play_ID orig=base;
	while(playlist[base].next_id!=invalid_Play_ID)
		base=playlist[base].next_id;
	Play_ID nplay=playlist.new_front();
	if(nplay==invalid_Play_ID)
		return nplay;
	Snd_ID nsnd=LoadBuffer(buf,len);
	if(nsnd==invalid_Snd_ID) {
		playlist.pop_front();
		return invalid_Play_ID;
	}
	playlist[nplay].snd_id=nsnd;
	playlist[base].next_id=nplay;
	return orig;
}

SoundManager::Play_ID
SoundManager::Chain(Play_ID base, Snd_ID next) {
	if(base==invalid_Play_ID || next==invalid_Snd_ID)
		return base;
	Play_ID orig=base;
	while(playlist[base].next_id!=invalid_Play_ID)
		base=playlist[base].next_id;
	Play_ID nplay=playlist.new_front();
	if(nplay==invalid_Play_ID)
		return nplay;
	playlist[nplay].snd_id=next;
	playlist[base].next_id=nplay;
	return orig;
}

void
SoundManager::StopPlay() {
	while(!playlist.empty())
		StopPlay(playlist.begin());
}

void
SoundManager::StopPlay(Play_ID id) {
	//	cout << "Stopping sound " << id << endl;
	if(id==invalid_Play_ID)
		return;
	AutoLock autolock(lock,ProcessID::getID());
	//we start at the back (oldest) since these are the most likely to be removed...
	for(chanlist_t::index_t it=chanlist.prev(chanlist.end()); it!=chanlist.end(); it=chanlist.prev(it))
		if(chanlist[it]==id) {
			Release(playlist[id].snd_id);
			playlist.erase(id);
			chanlist.erase(it);
			playlist[id].cumulative+=playlist[id].offset;
			unsigned int ms=playlist[id].cumulative/(config->sound.sample_bits/8)/(config->sound.sample_rate/1000);
			erouter->postEvent(EventBase::audioEGID,id,EventBase::deactivateETID,ms);
			return;
		}
	cout << "SoundManager::StopPlay(): " << id << " does not seem to be playing" << endl;
}

void
SoundManager::PausePlay(Play_ID id) {
	if(id==invalid_Play_ID)
		return;
	AutoLock autolock(lock,ProcessID::getID());
	for(chanlist_t::index_t it=chanlist.begin(); it!=chanlist.end(); it=chanlist.next(it))
		if(chanlist[it]==id) {
			chanlist.erase(it);
			return;
		}
}
	
void
SoundManager::ResumePlay(Play_ID id) {
	if(id==invalid_Play_ID)
		return;
	AutoLock autolock(lock,ProcessID::getID());
	for(chanlist_t::index_t it=chanlist.begin(); it!=chanlist.end(); it=chanlist.next(it))
		if(chanlist[it]==id)
			return;
	chanlist.push_front(id);
}
	
void
SoundManager::SetMode(unsigned int max_channels, MixMode_t mixer_mode, QueueMode_t queuing_mode) {
	AutoLock autolock(lock,ProcessID::getID());
	max_chan=max_channels;
	mix_mode=mixer_mode;
	queue_mode=queuing_mode;
}

unsigned int
SoundManager::GetRemainTime(Play_ID id) const {
	AutoLock autolock(lock,ProcessID::getID());
	unsigned int t=0;
	while(id!=invalid_Play_ID) {
		t+=sndlist[playlist[id].snd_id].len-playlist[id].offset;
		id=playlist[id].next_id;
	}
	const unsigned int bytesPerMS=config->sound.sample_bits/8*config->sound.sample_rate/1000;
	return t/bytesPerMS;
}

void
SoundManager::MixChannel(Play_ID channelId, void* buf, size_t destSize) {
	char *dest = (char*) buf;
	
	PlayState& channel = playlist[channelId];
	while (destSize > 0) {
		const SoundData& buffer = sndlist[channel.snd_id];
		const char* samples = ((char*) (buffer.data)) + channel.offset;
		const unsigned int samplesSize = buffer.len - channel.offset;
		if (samplesSize > destSize) {
			memcpy(dest, samples, destSize);
			channel.offset += destSize;
			dest += destSize;
			destSize = 0;
			return;
		} else {
			memcpy(dest, samples, samplesSize);
			channel.offset += samplesSize;
			dest += samplesSize;
			destSize -= samplesSize;
			if (endPlay(channelId)) {
				break;
			}
		}
	}
	if (destSize > 0) {
		memset(dest, 0, destSize);
	}
}

void
SoundManager::MixChannelAdditively(Play_ID channelId, int bitsPerSample, MixMode_t mode,
                                   short scalingFactor, void* buf, size_t destSize)
{
	PlayState& channel = playlist[channelId];
	while (destSize > 0) {
		const SoundData& buffer = sndlist[channel.snd_id];
		const unsigned int samplesSize = buffer.len - channel.offset;
		const unsigned int mixedSamplesSize =
			((mode == Fast)
				? ((samplesSize > destSize) ? destSize : samplesSize)
				: ((samplesSize > destSize / 2) ? destSize / 2 : samplesSize)); 
		
		if (bitsPerSample == 8) {
			//  8-bit mode
			const char* samples = (char*) (buffer.data + channel.offset);
			if (mode == Fast) {
				// 8-bit mixing
				char *dest = (char*) buf;
				for (size_t i = 0; i < mixedSamplesSize; i++) {
					*dest += samples[i] / scalingFactor;
					dest++;
				}
				destSize -= (char*) dest - (char*) buf;
				buf = dest;
			} else {
				// 16-bit mixing
				short* dest = (short*) buf;
				for (size_t i = 0; i < mixedSamplesSize; i++) {
					*dest += samples[i];
					*dest++;
				}
				destSize -= (char*) dest - (char*) buf;
				buf = dest;
			}
		} else {
			// 16-bit mode
			const short* samples = (short*) (buffer.data + channel.offset);
			if (mode == Fast) {
				// 16-bit mixing
				short* dest = (short*) buf;
				for (size_t i = 0; i < mixedSamplesSize / 2; i++) {
					*dest += samples[i] / scalingFactor;
					*dest++;
				}
				destSize -= (char*) dest - (char*) buf;
				buf = dest;
			} else {
				// 32-bit mixing
				int* dest = (int*) buf;
				for (size_t i = 0; i < mixedSamplesSize / 2; i++) {
					*dest += samples[i];
					*dest++;
				}
				destSize -= (char*) dest - (char*) buf;
				buf = dest;
			}
		}
		channel.offset += mixedSamplesSize;
		if (destSize == 0) {
			return;
		} else {
			if (endPlay(channelId)) {
				return;
			}
		}
	}
}

unsigned int
SoundManager::CopyTo(OSoundVectorData* data) {
	AutoLock autolock(lock,ProcessID::getID());

	size_t destSize = data->GetInfo(0)->dataSize;	
	void* dest = data->GetData(0);
	
	if(chanlist.size() == 0) {
		memset(dest, 0, destSize);
		return 0;
	}

	std::vector<Play_ID> channels;
	selectChannels(channels);

	if (channels.size() == 0) {
		// No channels to mix
		memset(dest, 0, destSize); 
	} else if (channels.size() == 1) {
		// One channel to mix
		MixChannel(channels.front(), dest, destSize);
	} else {
		// Several channels to mix	
		const MixMode_t mode = mix_mode;
		const int bitsPerSample = config->sound.sample_bits;
		if (mode == Quality) {
			// Quality mixing uses an intermediate buffer
			if ((mixerBuffer == 0) || (mixerBufferSize < destSize * 2)) {
				delete[] mixerBuffer;
				mixerBuffer = 0;
				mixerBufferSize = destSize * 2;
				mixerBuffer = new int[(mixerBufferSize / 4) + 1]; // makes sure it's int-aligned
			}
			memset(mixerBuffer, 0,  mixerBufferSize);
			dest = mixerBuffer;
			destSize *= 2;
		} else {
			// Fast mixing does not use the intermeridate buffer
			memset(dest, 0, destSize);
		}
		
		const int channelCount = channels.size();
		const short scalingFactor = (short) ((mode == Fast) ? channelCount : 1);  
		for(std::vector<Play_ID>::iterator i = channels.begin(); i != channels.end(); i++)
			MixChannelAdditively(*i, bitsPerSample, mode, scalingFactor, dest, destSize);
		
		if (mode == Quality) {
			// Quality mixing uses an intermediate buffer
			// Scale the buffer
			destSize /= 2;
			if (bitsPerSample == 8) {
				//  8-bit mode
				char* destChar = (char*) data->GetData(0);
				short* mixerBufferShort = (short*) mixerBuffer;
				for (size_t i = 0; i < destSize; i++) {
					destChar[i] = (char) (mixerBufferShort[i] / channelCount);
				} 
			} else {
				// 16-bit mode
				short* destShort = (short*) data->GetData(0);
				const size_t destSampleCount = destSize / 2; 
				for (size_t i = 0; i < destSampleCount; i++) {
					destShort[i] = (short) (mixerBuffer[i] / channelCount);
				}
			}
		}
	}
	
	updateChannels(channels, data->GetInfo(0)->dataSize);
	return channels.size(); 
}

void
SoundManager::ReceivedMsg(const ONotifyEvent& event) {
	for(int x=0; x<event.NumOfData(); x++) {
		RCRegion * rcr = event.RCData(x);
		SoundManagerMsg * msg = reinterpret_cast<SoundManagerMsg*>(rcr->Base());
		switch(msg->type) {
		case SoundManagerMsg::add: {
			rcr->AddReference();
			sndlist[msg->id].rcr=rcr;
			sndlist[msg->id].data=reinterpret_cast<byte*>(rcr->Base()+SoundManagerMsg::MSG_SIZE);
			//look to see if there's any play's for the sound we just finished loading
			for(playlist_t::index_t it=playlist.begin();it!=playlist.end();it=playlist.next(it))
				if(playlist[it].snd_id==msg->id)
					//send an event if there are
					erouter->postEvent(EventBase::audioEGID,it,EventBase::activateETID,0);
		} break;
		case SoundManagerMsg::del: {
			msg->region->RemoveReference();
		} break;
		case SoundManagerMsg::wakeup: {
			//doesn't need to do anything, just causes SoundPlay to check activity
		} break;
		default:
			printf("*** WARNING *** unknown SoundManager msg type received\n");
		}
	}
}

//protected:

RCRegion*
SoundManager::initRegion(unsigned int size) {
	unsigned int pagesize=4096;
	sError err=GetPageSize(&pagesize);
	ASSERT(err==sSUCCESS,"Error "<<err<<" getting page size");
	unsigned int pages=(size+pagesize-1)/pagesize;
	return new RCRegion(pages*pagesize);
}

SoundManager::Snd_ID 
SoundManager::lookupPath(std::string const &name) const {
        std::string const path(config->sound.makePath(name));
	for(sndlist_t::index_t it=sndlist.begin(); it!=sndlist.end(); it=sndlist.next(it))
		if(strncasecmp(path.c_str(),sndlist[it].name,MAX_NAME_LEN)==0)
			return it;
	return invalid_Snd_ID;
}

void
SoundManager::selectChannels(std::vector<Play_ID>& mix) {
	unsigned int selected=0;
	switch(queue_mode) {
	case Enqueue: { //select the oldest channels
		for(chanlist_t::index_t it=chanlist.prev(chanlist.end());it!=chanlist.end();it=chanlist.prev(it)) {
			if(sndlist[playlist[chanlist[it]].snd_id].data!=NULL) {
				mix.push_back(chanlist[it]);
				selected++;
				if(selected==max_chan)
					return;
			}
		}
	} break;
	case Override:
	case Pause: { //select the youngest channels (difference between these two is in the final update)
		for(chanlist_t::index_t it=chanlist.begin(); it!=chanlist.end(); it=chanlist.next(it)) {
			if(sndlist[playlist[chanlist[it]].snd_id].data!=NULL) {
				mix.push_back(chanlist[it]);
				selected++;
				if(selected==max_chan)
					return;
			}
		}
	} break;
	case Stop: { //select the youngest, stop anything that remains
		unsigned int numkeep=0;
		chanlist_t::index_t it=chanlist.begin();
		for(;it!=chanlist.end(); it=chanlist.next(it), numkeep++) {
			if(sndlist[playlist[chanlist[it]].snd_id].data!=NULL) {
				mix.push_back(chanlist[it]);
				selected++;
				if(selected==max_chan) {
					for(unsigned int i=chanlist.size()-numkeep-1; i>0; i--)
						endPlay(chanlist.back());
					return;
				}
			}
		}
	} break;
	default:
		cout << "SoundManager::selectChannels(): Illegal queue mode" << endl;
	}
}

void
SoundManager::updateChannels(const std::vector<Play_ID>& mixs,size_t used) {
	switch(queue_mode) {
	case Enqueue:
	case Pause:
	case Stop: 
		break;
	case Override: { //increase offset of everything that wasn't selected
		//assumes mode hasn't changed since the mix list was created... (so order is same as chanlist)
		chanlist_t::index_t it=chanlist.begin(); 
		std::vector<Play_ID>::const_iterator mixit=mixs.begin();
		for(;it!=chanlist.end(); it=chanlist.next(it)) {
			for(;mixit!=mixs.end(); mixit++) //some mixs may have been stopped during play
				if(*mixit==chanlist[it])
					break;
			if(mixit==mixs.end())
				break;
		}
		for(;it!=chanlist.end(); it=chanlist.next(it)) {
			const Play_ID channelId = chanlist[it];
			PlayState &channel = playlist[channelId];
			size_t skip = used;
			while (skip > 0) {
				SoundData &buffer = sndlist[channel.snd_id];
				// FIXME: Don't know why the buffer.data != 0 check is done 
				if (buffer.data != 0) {
					size_t remain = buffer.len - channel.offset;
					if (remain < skip) {
						channel.offset = buffer.len;
						skip -= buffer.len;
						if (endPlay(channelId)) {
							break;
						}
					} else {
						channel.offset += skip;
						skip = 0;
					}
				} else {
					break;
				}
			}
		}
	} break;
	default:
		cout << "SoundManager::updateChannels(): Illegal queue mode" << endl;
	}
}

bool
SoundManager::endPlay(Play_ID id) {
	if(playlist[id].next_id==invalid_Play_ID) {
		StopPlay(id);
		return true;
	} else {
		//copies the next one into current so that the Play_ID consistently refers to the same "sound"
		Play_ID next=playlist[id].next_id;
		//		cout << "play " << id << " moving from " << playlist[id].snd_id << " to " << playlist[next].snd_id << endl;
		Release(playlist[id].snd_id);
		playlist[id].snd_id=playlist[next].snd_id;
		playlist[id].cumulative+=playlist[id].offset;
		playlist[id].offset=0;
		playlist[id].next_id=playlist[next].next_id;
		playlist.erase(next);
		unsigned int ms=playlist[id].cumulative/(config->sound.sample_bits/8)/(config->sound.sample_rate/1000);
		erouter->postEvent(EventBase::audioEGID,id,EventBase::statusETID,ms);
		return false;
	}
}

SoundManager::SoundData::SoundData()
	: rcr(NULL), data(NULL), len(0), ref(0)
{
	name[0]='\0';
}

SoundManager::PlayState::PlayState()
	: snd_id(invalid_Snd_ID), offset(0), cumulative(0), next_id(invalid_Play_ID)
{}


/*! @file
 * @brief Implements SoundManager, which provides sound effects and caching services, as well as mixing buffers for the SoundPlay process
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_2_1 $
 * $Revision: 1.18 $
 * $State: Exp $
 * $Date: 2004/11/04 05:24:03 $
 */




//This is a faster mix algo using bit shifting, but it doesn't work with
// non power of two number of channels, despite my efforts... eh, maybe
// i'll fix it later...
		// NOT WORKING
		/*
		if(mixs.size()==2 || mix_mode==Fast) {
			unsigned int shift=0;
			unsigned int offset=0;
			unsigned int tmp=mixs.size();
			while(tmp>1) {
				tmp>>=1;
				shift++;
			}
			unsigned int mask;
			if(config->sound.sample_bits==8) {
				unsigned int c=(unsigned char)~0;
				c>>=shift;
				mask=(c<<24)|(c<<16)|(c<<8)|c;
				offset=(1<<7)|(1<<15)|(1<<23)|(1<<31);
			} else {
				unsigned int c=(unsigned short)~0;
				c>>=shift;
				mask=(c<<16)|c;
				offset=(1<<31)|(1<<15);
			}
			memset(dest,0,avail);
			for(unsigned int c=0; c<mixs.size(); c++) {
				if(ends[c]-srcs[c]>avail) {
					for(unsigned int * beg=(unsigned int*)dest;beg<(unsigned int*)end;beg++) {
						const unsigned int src=*(unsigned int*)srcs[c];
						if(beg==(unsigned int*)dest) {
							cout << src <<' '<< (void*)src << endl;
							unsigned int x=((src^offset)>>shift)&mask;
							cout << x <<' '<< (void*)x << endl;
							cout << "****" << endl;
						}
						*beg+=((src^offset)>>shift)&mask;
						if(beg==(unsigned int*)dest)
							cout << *beg <<' '<< (void*)*beg << endl << "########" << endl;
						srcs[c]+=sizeof(int);
					}
					playlist[mixs[c]].offset+=avail;
				} else {
					unsigned int * beg=(unsigned int*)dest;
					for(;srcs[c]<ends[c];srcs[c]+=sizeof(int)) {
						const unsigned int src=*(unsigned int*)srcs[c];
						*beg+=((src^offset)>>shift)&mask;
						beg++;
					}
					for(;beg<(unsigned int*)end; beg++)
						*beg+=offset>>shift;
					playlist[mixs[c]].offset=sndlist[playlist[mixs[c]].snd_id].len;
					StopPlay(mixs[c]);
				}
			}
			unsigned int leftover=(offset>>shift)*((1<<shift)-mixs.size());
			for(unsigned int * beg=(unsigned int*)dest;beg<(unsigned int*)end;beg++)
				*beg=*(beg+leftover)^offset;
			updateChannels(avail);
			return mixs.size();
			} else*/
