//-*-c++-*-
#ifndef __MUTEX_LOCK_ET__
#define __MUTEX_LOCK_ET__

#include "debuget.h"
#include <iostream>

//#define MUTEX_LOCK_ET_USE_SPINCOUNT

//! A software only mutual exclusion lock.
/*! Use this to prevent more than one process from accessing a data structure
 *  at the same time (which often leads to unpredictable and unexpected results)
 *
 *  The template parameter specifies the maximum number of different processes
 *  which need to be protected.  This needs to be allocated ahead of time, as
 *  there doesn't seem to be a way to dynamically scale as needed without
 *  risking possible errors if two processes are both trying to set up at the
 *  same time.  Also, by using a template parameter, all data structures are
 *  contained within the class's memory allocation, so no pointers are involved.
 *
 *  Locks in this class can be recursive or non-recursive, depending
 *  whether you call release() or unlock().  If you lock 5 times, then
 *  you should call release() 5 times as well before it will be
 *  unlocked.  However, if you lock 5 times, just one call to unlock()
 *  will undo all 5 levels of locking.
 *
 *  Just remember, release() releases one level.  But unlock() completely unlocks.
 *
 *  Note that there is no check that the process doing the unlocking is the one
 *  that actually has the lock.  Be careful about this.
 *
 *  @warning Doing mutual exclusion in software is tricky business, be careful about any
 *  modifications you make!
 *
 * Implements a first-come-first-served Mutex as laid out on page 11 of: \n
 * "A First Come First Served Mutal Exclusion Algorithm with Small Communication Variables" \n
 * Edward A. Lycklama, Vassos Hadzilacos - Aug. 1991
*/
template<unsigned int num_doors>
class MutexLock {
 public:
	static const unsigned int NO_OWNER=-1U; //!< marks as unlocked

	//! constructor, just calls the init() function.
	MutexLock() : doors_used(0), owner_index(NO_OWNER), lockcount(0) { init();	}

	//! blocks (by busy looping on do_try_lock()) until a lock is achieved
	/*! You should pass some process-specific ID number as the input - just
	 *  make sure no other process will be using the same value.
	 *
	 *  This is not a recursive lock - repeated locks still only require one
	 *  release to undo them.
	 *  @todo - I'd like to not use a loop here */
	void lock(int id);

	//! attempts to get a lock, returns true if it succeeds
	/*! You should pass some process-specific ID number as the input - just
	 *  make sure no other process will be using the same value.
	 *
	 *  This is not a recursive lock - repeated locks still only require one
	 *  release to undo them. */
	bool try_lock(int id);

	//! releases one recursive lock-level from whoever has the current lock
	void release();

	//! completely unlocks, regardless of how many times a recursive lock has been obtained
	inline void unlock() { lockcount=1; release(); }

	//! returns the lockcount
	unsigned int get_lock_level() const { return lockcount;	}

	//! returns the current owner's id
	inline int owner() { return owner_index==NO_OWNER ? NO_OWNER : doors[owner_index].id; }

	//! allows you to reset one of the possible owners, so another process can take its place.  This is not tested
	void forget(int id);

#ifdef MUTEX_LOCK_ET_USE_SPINCOUNT
	inline unsigned int getSpincount() { return spincount; } //!< returns the number of times the spin() function has been called
	inline unsigned int resetSpincount() { spincount=0; } //!< resets the counter of the number of times the spin() function has been called
#endif
	
 protected:
	//! Does the work of trying to get a lock
	/*! Pass @c true for @a block if you want it to use FCFS blocking
	 *  instead of just returning right away if another process has the lock */
	bool do_try_lock(unsigned int index, bool block);

	//! returns the internal index mapping to the id number supplied by the process
	unsigned int lookup(int id); //may create a new entry

#ifdef MUTEX_LOCK_ET_USE_SPINCOUNT
	volatile unsigned int spincount; //!< handy to track how much time we're wasting
	void init() { spincount=0; }//memset((void*)doors,0,sizeof(doors)); } //!< just resets spincount
	inline void spin() { spincount++; } //!< if you find a way to sleep for a few microseconds instead of busy waiting, put it here
#else
	void init() { } //!< Doesn't do anything if you have the MUTEX_LOCK_ET_USE_SPINCOUNT undef'ed.  Used to do a memset, but that was causing problems....
	//memset((void*)doors,0,sizeof(doors)); } 
	inline void spin() {} //!< If you find a way to sleep for a few microseconds instead of busy waiting, put it here
#endif
		
	//! Holds per process shared info, one of these per process
	struct door_t {
		door_t() : id(NO_OWNER), FCFS_in_use(false), BL_ready(false), BL_in_use(false), turn('\0'), next_turn_bit('\0') {} //!< constructor
		//door_t(int i) : id(i), FCFS_in_use(false), BL_ready(false), BL_in_use(false), next_turn_bit('\0') {}
		int id; //!< process ID this doorway is assigned to
		volatile bool FCFS_in_use; //!< In FCFS doorway, corresponds to 'c_i'
		volatile bool BL_ready; //!< Signals past FCFS doorway, ready for BL doorway, corresponds to 'v_i'
		volatile bool BL_in_use; //!< Burns-Lamport doorway, corresponds to 'x_i'
		volatile unsigned char turn; //!< clock pulse, initial value doesn't matter
		unsigned char next_turn_bit; //!< selects which bit of turn will be flipped next
	};

	door_t doors[num_doors]; //!< holds all the doors
	unsigned int doors_used; //!< counts the number of doors used
	unsigned int owner_index; //!< holds the door index of the current lock owner
	unsigned int lockcount; //!< the depth of the lock, 0 when unlocked
};


template<unsigned int num_doors>
void
MutexLock<num_doors>::lock(int id) {
	if(owner()!=id)
		while(!do_try_lock(lookup(id),true))
			spin();
	lockcount++;
}


template<unsigned int num_doors>
bool
MutexLock<num_doors>::try_lock(int id) {
	if(owner()==id) {
		lockcount++;
		return true;
	} else {
		if(do_try_lock(lookup(id),false)) {
			lockcount++;
			return true;
		} else
			return false;
	}
}


template<unsigned int num_doors>
void
MutexLock<num_doors>::release() {
	if(lockcount>0)
		if(--lockcount==0)
			if(owner_index!=NO_OWNER) {
				unsigned int tmp = owner_index;
				owner_index=NO_OWNER;
				doors[tmp].BL_in_use=false;
				doors[tmp].BL_ready=false;
				// *** Lock has been released *** //
			}
}


//! If you define this to do something more interesting, can use it to see what's going on in the locking process
#define mutexdebugout(i,c) {}
//#define mutexdebugout(i,c) { std::cout << ((char)(i==0?c:((i==1?'M':'a')+(c-'A')))) << std::flush; }


template<unsigned int num_doors>
bool
MutexLock<num_doors>::do_try_lock(unsigned int i, bool block) {
	if(i==NO_OWNER) {
		std::cerr << "WARNING: new process attempted to lock beyond num_doors ("<<num_doors<<")" << std::endl;
		return false;
	}
	unsigned char S[num_doors]; // a local copy of everyone's doors
	// *** Entering FCFS doorway *** //
//	pprintf(TextOutputStream,"**%d**\n",i);
mutexdebugout(i,'A');
	doors[i].FCFS_in_use=true;
	for(unsigned int j=0; j<num_doors; j++)
		S[j]=doors[j].turn;
	doors[i].next_turn_bit=1-doors[i].next_turn_bit;
	doors[i].turn^=(1<<doors[i].next_turn_bit);
	doors[i].BL_ready=true;
	doors[i].FCFS_in_use=false;
	// *** Leaving FCFS doorway *** //
mutexdebugout(i,'B');
	for(unsigned int j=0; j<num_doors; j++) {
mutexdebugout(i,'C');
		while(doors[j].FCFS_in_use || (doors[j].BL_ready && S[j]==doors[j].turn))
			if(block)
				spin();
			else {
				doors[i].BL_ready=false;
				return false;
			}
mutexdebugout(i,'D');
	}
	// *** Entering Burns-Lamport *** //
mutexdebugout(i,'E');
	do {
		doors[i].BL_in_use=true;
		for(unsigned int t=0; t<i; t++)
			if(doors[t].BL_in_use) {
				doors[i].BL_in_use=false;
				if(!block) {
					doors[i].BL_ready=false;
					return false;
				}
mutexdebugout(i,'F');
				while(doors[t].BL_in_use)
					spin();
mutexdebugout(i,'G');
				break;
			}
	} while(!doors[i].BL_in_use);
	for(unsigned int t=i+1; t<num_doors; t++)
		while(doors[t].BL_in_use)
			spin();
	// *** Leaving Burns-Lamport ***//
	// *** Lock has been given *** //
mutexdebugout(i,'H');
	owner_index=i;
	return true;
}


template<unsigned int num_doors>
unsigned int
MutexLock<num_doors>::lookup(int id) {
	// TODO - this could break if two new processes are adding themselves at the same time
	//        or an id is being forgotten at the same time
	//I'm expecting a very small number of processes to be involved
	//probably not worth overhead of doing something fancy like a sorted array
	unsigned int i;
	for(i=0; i<doors_used; i++)
		if(doors[i].id==id)
			return i;
	if(i==num_doors)
		return NO_OWNER;
	doors[i].id=id;
	doors_used++;
	return i;
}


template<unsigned int num_doors>
void
MutexLock<num_doors>::forget(int id) { //not tested thoroughly
	unsigned int i = lookup(id);
	do_try_lock(i,true);
	doors[i].id=doors[--doors_used].id;
	doors[doors_used].id=NO_OWNER;
	release();
}


/*! @file 
 * @brief Defines MutexLock, a software only mutual exclusion lock.
 * @author ejt (Creator), Edward A. Lycklama, Vassos Hadzilacos (paper from which this was based)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_2_1 $
 * $Revision: 1.9 $
 * $State: Exp $
 * $Date: 2004/02/12 17:48:59 $
 */

#endif
