#include "Kinematics.h"
#include "Shared/Config.h"
#include "roboop/robot.h"
#include "roboop/config.h"
#include "Shared/WorldState.h"
#include <sstream>

using namespace std;

Kinematics * kine = NULL;
ROBOOP::Config * Kinematics::roconfig = NULL;
Kinematics::InterestPointMap * Kinematics::ips = NULL;

void
Kinematics::init() {
	unsigned int nchains=::config->motion.kinematic_chains.size();
	if(nchains==0) {
		serr->printf("ERROR Kinematics::init(): no kinematic chains were selected\n");
		return;
	}
	jointMaps[BaseFrameOffset]=JointMap(0,0);
	chains.resize(nchains);
	chainMaps.resize(nchains);
	if(roconfig==NULL)
		roconfig=new ROBOOP::Config(::config->motion.makePath(::config->motion.kinematics),true);
	for(unsigned int i=0; i<nchains; i++) {
		string section=::config->motion.kinematic_chains[i];
		chains[i]=new ROBOOP::Robot(*roconfig,section);
		int dof=0;
		if(roconfig->select_int(section,"dof",dof)!=0) {
			serr->printf("ERROR Kinematics::init(): unable to find 'dof' in chain %d (%s)\n",i,section.c_str());
			return;
		}
		chainMaps[i].resize(dof+1);
		for(int l=0; l<=dof; l++)
			chainMaps[i][l]=-1U;
		for(int l=1; l<=dof; l++) {
			ostringstream ostr;
			ostr << section << "_LINK" << l;
			string link = ostr.str();
			if(roconfig->parameter_exists(link,"tekkotsu_output")) {
				int tkout=-1U;
				roconfig->select_int(link,"tekkotsu_output",tkout);
				if((unsigned int)tkout>=NumOutputs) {
					serr->printf("WARNING Kinematics::init(): invalid tekkotsu_output %d on chain %d (%s), link %d (%s)\n",tkout,i,section.c_str(),l,link.c_str());
				} else {
					jointMaps[tkout]=JointMap(i,l);
					chainMaps[i][l]=tkout;
				}
			}
			if(roconfig->parameter_exists(link,"tekkotsu_frame")) {
				int tkout=-1U;
				roconfig->select_int(link,"tekkotsu_frame",tkout);
				tkout+=NumOutputs;
				if((unsigned int)tkout>=NumReferenceFrames)
					serr->printf("WARNING Kinematics::init(): invalid tekkotsu_frame %d on chain %d (%s), link %d (%s)\n",tkout,i,section.c_str(),l,link.c_str());
				else
					jointMaps[tkout]=JointMap(i,l);
			}
		}
	}

	if(ips==NULL) {
		int numIP=0;
		roconfig->select_int("InterestPoints","Length",numIP);
		ips=new InterestPointMap(numIP);
		pair<string,InterestPoint> ip;
		string chain;
		for(int i=1; i<=numIP; i++) {
			char section[100];
			snprintf(section,100,"InterestPoint%d",i);
			roconfig->select_float(section,"x",ip.second.x);
			roconfig->select_float(section,"y",ip.second.y);
			roconfig->select_float(section,"z",ip.second.z);
			roconfig->select_string(section,"chain",chain);
			roconfig->select_int(section,"link",ip.second.link);
			roconfig->select_string(section,"name",ip.first);
			for(ip.second.chain=0; ip.second.chain<(::config->motion.kinematic_chains.size()); ip.second.chain++)
				if(::config->motion.kinematic_chains[ip.second.chain]==chain)
					break;
			if(ip.second.chain==::config->motion.kinematic_chains.size())
				serr->printf("WARNING: Chain %s is not recognized for interest point %d\n",chain.c_str(),i);
			else
				ips->insert(ip);
		}
	}

	/*cout << "Joint Maps" << endl;
	for(unsigned int i=0; i<NumOutputs; i++)
		cout << outputNames[i] << ": " << jointMaps[i].chain << ' ' << jointMaps[i].link << endl;
	for(unsigned int i=NumOutputs; i<NumReferenceFrames; i++)
		cout << i << ": " << jointMaps[i].chain << ' ' << jointMaps[i].link << endl;
	cout << "Chain Maps" << endl;
	for(unsigned int i=0; i<chains.size(); i++)
		for(unsigned int j=1; j<chainMaps[i].size(); j++)
		cout << "chainMaps["<<i<<"]["<<j<<"] == " << chainMaps[i][j] << endl;*/
}

NEWMAT::ReturnMatrix
Kinematics::frameToBase(unsigned int j) {
	unsigned int c=-1U,l=-1U;
	if(!lookup(j,c,l)) {
		NEWMAT::Matrix A(4,4);
		A<<ROBOOP::fourbyfourident;
		A.Release(); return A;
	}
	update(c,l);
	return chains[c]->convertFrame(l,0);
}

NEWMAT::ReturnMatrix
Kinematics::linkToBase(unsigned int j) {
	unsigned int c=-1U,l=-1U;
	if(!lookup(j,c,l)) {
		NEWMAT::Matrix A(4,4);
		A<<ROBOOP::fourbyfourident;
		A.Release(); return A;
	}
	update(c,l);
	return chains[c]->convertLink(l,0);
}

NEWMAT::ReturnMatrix
Kinematics::baseToFrame(unsigned int j) {
	unsigned int c=-1U,l=-1U;
	if(!lookup(j,c,l)) {
		NEWMAT::Matrix A(4,4);
		A<<ROBOOP::fourbyfourident;
		A.Release(); return A;
	}
	update(c,l);
	return chains[c]->convertFrame(0,l);
}

NEWMAT::ReturnMatrix
Kinematics::baseToLink(unsigned int j) {
	unsigned int c=-1U,l=-1U;
	if(!lookup(j,c,l)) {
		NEWMAT::Matrix A(4,4);
		A<<ROBOOP::fourbyfourident;
		A.Release(); return A;
	}
	update(c,l);
	return chains[c]->convertLink(0,l);
}

NEWMAT::ReturnMatrix
Kinematics::frameToFrame(unsigned int ij, unsigned int oj) {
	unsigned int ci=-1U,li=-1U,co=-1U,lo=-1U;
	if(ij==oj || !lookup(ij,ci,li) || !lookup(oj,co,lo)) {
		NEWMAT::Matrix A(4,4);
		A<<ROBOOP::fourbyfourident;
		A.Release(); return A;
	}
	if(ci==co) {
		update(ci,li>lo?li:lo);
		return chains[ci]->convertFrame(li,lo);
	} else if(li==0) {
		update(co,lo);
		return chains[co]->convertFrame(0,lo);
	} else if(lo==0) {
		update(ci,li);
		return chains[ci]->convertFrame(li,0);
	} else {
		update(ci,li);
		update(co,lo);
		NEWMAT::Matrix ans=chains[co]->convertFrame(0,lo)*chains[ci]->convertFrame(li,0);
		ans.Release(); return ans;
	}
}

NEWMAT::ReturnMatrix
Kinematics::frameToLink(unsigned int ij, unsigned int oj) {
	unsigned int ci=-1U,li=-1U,co=-1U,lo=-1U;
	if(ij==oj || !lookup(ij,ci,li) || !lookup(oj,co,lo)) {
		NEWMAT::Matrix A(4,4);
		A<<ROBOOP::fourbyfourident;
		A.Release(); return A;
	}
	if(ci==co) {
		update(ci,li>lo?li:lo);
		return chains[ci]->convertFrameToLink(li,lo);
	} else if(li==0) {
		update(co,lo);
		return chains[co]->convertFrameToLink(0,lo);
	} else if(lo==0) {
		update(ci,li);
		return chains[ci]->convertFrameToLink(li,0);
	} else {
		update(ci,li);
		update(co,lo);
		NEWMAT::Matrix ans=chains[co]->convertLink(0,lo)*chains[ci]->convertFrame(li,0);
		ans.Release(); return ans;
	}
}

NEWMAT::ReturnMatrix
Kinematics::linkToFrame(unsigned int ij, unsigned int oj) {
	unsigned int ci=-1U,li=-1U,co=-1U,lo=-1U;
	if(ij==oj || !lookup(ij,ci,li) || !lookup(oj,co,lo)) {
		NEWMAT::Matrix A(4,4);
		A<<ROBOOP::fourbyfourident;
		A.Release(); return A;
	}
	if(ci==co) {
		update(ci,li>lo?li:lo);
		return chains[ci]->convertLinkToFrame(li,lo);
	} else if(li==0) {
		update(co,lo);
		return chains[co]->convertLinkToFrame(0,lo);
	} else if(lo==0) {
		update(ci,li);
		return chains[ci]->convertLinkToFrame(li,0);
	} else {
		update(ci,li);
		update(co,lo);
		NEWMAT::Matrix ans=chains[co]->convertFrame(0,lo)*chains[ci]->convertLink(li,0);
		ans.Release(); return ans;
	}
}

NEWMAT::ReturnMatrix
Kinematics::linkToLink(unsigned int ij, unsigned int oj) {
	unsigned int ci=-1U,li=-1U,co=-1U,lo=-1U;
	if(ij==oj || !lookup(ij,ci,li) || !lookup(oj,co,lo)) {
		NEWMAT::Matrix A(4,4);
		A<<ROBOOP::fourbyfourident;
		A.Release(); return A;
	}
	if(ci==co) {
		update(ci,li>lo?li:lo);
		return chains[ci]->convertLink(li,lo);
	} else if(li==0) {
		update(co,lo);
		return chains[co]->convertLink(0,lo);
	} else if(lo==0) {
		update(ci,li);
		return chains[ci]->convertLink(li,0);
	} else {
		update(ci,li);
		update(co,lo);
		NEWMAT::Matrix ans=chains[co]->convertLink(0,lo)*chains[ci]->convertLink(li,0);
		ans.Release(); return ans;
	}
}

void
Kinematics::getInterestPoint(const std::string& name, unsigned int& j, NEWMAT::Matrix& ip) {
	unsigned int c=-1U,l=-1U;
	getInterestPoint(name,c,l,ip);
	j=chainMaps[c][l];
}

void
Kinematics::getInterestPoint(const std::string& name, unsigned int& c, unsigned int& l, NEWMAT::Matrix& ip) {
	InterestPointMap::iterator it=ips->find(name);
	ip=NEWMAT::ColumnVector(4);
	if(it==ips->end()) {
		serr->printf("ERROR: '%s' is not a recognized interest point\n",name.c_str());
		ip=0;
		c=l=-1U;
	}
	c=(*it).second.chain;
	l=(*it).second.link;
	ip=pack((*it).second.x,(*it).second.y,(*it).second.z);
	//cout << ci << ' ' << li << ' ' << co << ' ' << lo << endl;
}

NEWMAT::ReturnMatrix
Kinematics::getFrameInterestPoint(unsigned int frame, const std::string& name) {
	NEWMAT::ColumnVector ans(4);
	ans=0;
	unsigned int co=-1U,lo=-1U;
	if(!lookup(frame,co,lo)) {
		ans.Release(); return ans;
	}
	for(unsigned int pos=0,len=0; pos<name.size(); pos+=len+1) {
		len=name.find(',',pos);
		if(len==string::npos)
			len=name.size();
		len-=pos;
		unsigned int ci=-1U,li=-1U;
		NEWMAT::ColumnVector ip;
		getInterestPoint(name.substr(pos,len),ci,li,ip);
		if(ci==-1U) {
			ip.Release(); return ip;
		}
		if(ci==co) {
			update(ci,li>lo?li:lo);
			ans+=chains[ci]->convertLinkToFrame(li,lo)*ip;
		} else if(li==0) {
			update(co,lo);
			ans+=chains[co]->convertLinkToFrame(0,lo)*ip;
		} else if(lo==0) {
			update(ci,li);
			ans+=chains[ci]->convertLinkToFrame(li,0)*ip;
		} else {
			update(ci,li);
			update(co,lo);
			ans+=chains[co]->convertFrame(0,lo)*chains[ci]->convertLink(li,0)*ip;
		}
	}
	ans/=ans(4); //not strictly necessary, but may save some people headaches
	ans.Release(); return ans;
}

LegOrder_t
Kinematics::findUnusedLeg(const NEWMAT::ColumnVector& down) {
	float height[NumLegs]; //not actually the real height, but proportional to it, which is all we need
	for(unsigned int i=0; i<NumLegs; i++) {
		height[i]=NEWMAT::DotProduct(frameToBase(PawFrameOffset+i).SubMatrix(1,3,4,4),down.SubMatrix(1,3,1,1));
		//cout << "height["<<i<<"]=="<<height[i]<<endl;
	}
	
	//Find the highest foot
	unsigned int highleg=0;
	for(unsigned int i=1; i<NumLegs; i++)
		if(height[highleg]>height[i])
			highleg=i;
	//cout << "High: " << highleg << endl;
	return static_cast<LegOrder_t>(highleg);
}

NEWMAT::ReturnMatrix
Kinematics::calculateGroundPlane() {
  return calculateGroundPlane(pack(state->sensors[BAccelOffset],state->sensors[LAccelOffset],state->sensors[DAccelOffset]));
}

NEWMAT::ReturnMatrix
Kinematics::calculateGroundPlane(const NEWMAT::ColumnVector& down) {
	//Find the unused foot
	unsigned int highleg=findUnusedLeg(down);
	
	//Fit a plane to the remaining 3 feet
	NEWMAT::Matrix legs(3,3);
	for(unsigned int i=0; i<highleg; i++)
		legs.Column(i+1)=frameToBase(PawFrameOffset+i).SubMatrix(1,3,4,4);
	for(unsigned int i=highleg+1; i<NumLegs; i++)
		legs.Column(i)=frameToBase(PawFrameOffset+i).SubMatrix(1,3,4,4);
	//cout << legs;
	NEWMAT::ColumnVector ones(3); ones=1;
	NEWMAT::ColumnVector p(4);
	p.SubMatrix(1,3,1,1) = legs.t().i()*ones;
	p(4)=1;
	p.Release(); return p;
}

NEWMAT::ReturnMatrix
Kinematics::projectToPlane(unsigned int j, const NEWMAT::ColumnVector& r_j,
                           unsigned int b, const NEWMAT::ColumnVector& p_b,
                           unsigned int f)
{
	/*! Mathematcal implementation:
	 *  
	 *  Need to convert @a p_b to @a p_j
	 *  
	 *  Once we have the transformation Tb_j from b to j, we need:\n
	 *    T2=inv(Tb_j)'; T2(3,1:end-1)=-T2(3,1:end-1);\n
	 *  but since we know a few things about the structure of T,
	 *  we don't have to explicitly calculate that inverse. */
	NEWMAT::Matrix T2=linkToLink(b,j);
	//cout << "Transform b->j:\n"<<T2;
	T2.SubMatrix(4,4,1,3)=T2.SubMatrix(1,3,4,4).t()*T2.SubMatrix(1,3,1,3);
	T2.SubMatrix(1,3,4,4)=0;
	//cout << "Transform plane b->j:\n"<<T2;
	NEWMAT::ColumnVector p_j=T2*p_b;
	//cout << "p_j:\n"<<p_j.t();

	
	/*! After we obtain @a p_j, we can find the point of intersection of
	 *  @a r_j and @a p_j using:
	 *  @f[ \frac{p_d}{p_{xyz} \cdot r}r @f]
	 *  Where @f$p_{xyz}@f$ is the first three elemnts of @a p_j, and
	 *  @f$p_d@f$ is the fourth (hopefully last) element of p_j.
	 *
	 *  Of course, if @f$p_{xyz} \cdot r@f$ is 0, then r and p are parallel
	 *  (since @a p_j is the normal of the plane, so a line perpendicular to
	 *  the normal is parallel to the plane), so we set the resulting
	 *  homogenous coordinates accordingly to represent an interesection at
	 *  infinity. */

	float denom=0;
	for(unsigned int i=1; i<=3; i++)
		denom+=r_j(i)*p_j(i);
	NEWMAT::ColumnVector intersect=r_j;
	if(denom==0)
		intersect(4)=0;
	else {
		float s=p_j(4)/denom;
		for(unsigned int i=1; i<=3; i++)
			intersect(i)*=s;
		intersect(4)=1;
	}
	//cout << "Intersect_j:\n"<<intersect.t();
	NEWMAT::Matrix ans=linkToLink(j,f)*intersect;
	ans.Release(); return ans;
}

void
Kinematics::update(unsigned int c, unsigned int l) {
	for(unsigned int j=1; j<=l; j++) {
		unsigned int tkout=chainMaps[c][j];
		if(tkout<NumOutputs)
			chains[c]->set_q(state->outputs[tkout],j);
	}
}


/*! @file
 * @brief 
 * @author ejt (Creator)
 *
 * $Author: dst $
 * $Name: tekkotsu-2_2_1 $
 * $Revision: 1.19 $
 * $State: Exp $
 * $Date: 2004/10/28 23:06:10 $
 */

