#include "EventRouter.h"
#include "Shared/Profiler.h"
#include <algorithm>

EventRouter * erouter=NULL;

EventRouter::EventRouter()
	: timers(), events(), doSendBufferLock(false), lastBufClear(0), buffertime(0), trappers(), listeners()
{}

//! @todo handle recursive calls
void EventRouter::processTimers() {
	//		cout << "processTimers..." << flush;
	unsigned int curtime=get_time();
	if(curtime-lastBufClear>=buffertime || buffertime==1)
		processEventBuffer();

	TimerEntry curTimer(curtime);
	timer_it_t last_it=upper_bound(timers.begin(),timers.end(),&curTimer,TimerEntryPtrCmp());
	std::vector<TimerEntry*> process(timers.begin(),last_it); //copy these out for safe keeping
	for(timer_it_t it=process.begin(); it!=process.end(); it++) //increment the timers we're processing
		if((*it)->repeat)
			(*it)->next+=(*it)->delay;
		else
			(*it)->next=(unsigned int)-1;
	sort(timers.begin(),last_it,TimerEntryPtrCmp()); //re-sort the timers we're processing (at the beginning of timers)
	inplace_merge(timers.begin(),last_it,timers.end(),TimerEntryPtrCmp()); //now do a merge of the sorted processed stuff and the rest of the list (which is still sorted)
	//	if(process.size()>0) chkTimers();
	for(timer_it_t it=process.begin(); it!=process.end(); it++) // process the timers we say we're going to, can no longer assume anything about the state of the world
		(*it)->el->processEvent(EventBase(EventBase::timerEGID,(*it)->sid,EventBase::statusETID,(*it)->next));
	//	if(process.size()>0) chkTimers();
	static const TimerEntry deadTimer((unsigned int)-1); // matches all the dead ones as set in the incrementation phase
	last_it=lower_bound(timers.begin(),timers.end(),&deadTimer,TimerEntryPtrCmp()); //find the beginning of all the non-repeating timers we're clearing
	for(timer_it_t it=last_it; it!=timers.end(); it++) // delete all of them
		delete *it;
	timers.erase(last_it,timers.end()); //and then remove them from the timer list
	//	if(process.size()>0) chkTimers();
	//		cout << "done" << endl;
}

/*! timers are unique by EventListener and source ID - can't have two timers for the same el and sid\n
 *  a delay of 0 with repeating will cause an event to be sent at every opportunity, use sparingly\n
 *  a delay of -1U will call removeTimer() if it already exists, otherwise is ignored\n
 *
 *  To add a timer, you can also call addListener() with EventBase::timerEGID and the sid
 *  and delay (in the EventBase::duration field) - this method will simply cause this
 *  function to be called internally.
 *
 *  @param el the EventListener to send the timer event to
 *  @param sid the source ID to use on that event (if you need to send more info, send a pointer to a struct of your devising, typecasted as int)
 *  @param delay the delay between the first (and future) calls
 *  @param repeat set to true if you want to keep receiving this event, otherwise it will only send once */
void EventRouter::addTimer(EventListener* el, unsigned int sid, unsigned int delay, bool repeat) {
	if(delay==-1U) {
		removeTimer(el,sid);
		return;
	}
	for(timer_it_t it=timers.begin(); it!=timers.end(); it++)
		if((*it)->el==el && (*it)->sid==sid) {
			(*it)->Set(delay,repeat);
			// now put that timer back into the correct place in the ordering (think before touching this! ;)
			if(it!=timers.begin() && (*it)->next<(*(it-1))->next)
				rotate(upper_bound(timers.begin(),it,*it,TimerEntryPtrCmp()),it,it+1);
			else if(it+1!=timers.end() && (*it)->next>(*(it+1))->next)
				rotate(it,it+1,lower_bound(it+1,timers.end(),*it,TimerEntryPtrCmp()));
			return;
		}
	//didn't find a pre-existing one
	TimerEntry * add=new TimerEntry(el,sid,delay,repeat);
	timers.insert(lower_bound(timers.begin(),timers.end(),add,TimerEntryPtrCmp()),add);
	//	chkTimers();
}

void EventRouter::removeTimer(EventListener* el) {
	for(timer_it_t it=timers.begin(); it!=timers.end(); it++)
		if((*it)->el==el) {
			delete *it;
			*it=NULL;
		}
	timers.erase(remove(timers.begin(),timers.end(),(const TimerEntry*)NULL),timers.end());
}

void EventRouter::removeTimer(EventListener* el, unsigned int sid) {
	for(timer_it_t it=timers.begin(); it!=timers.end(); it++)
		if((*it)->el==el && (*it)->sid==sid) {
			delete *it;
			timers.erase(it);
			return;
		}
}

void EventRouter::removeAllTimers() {
	for(timer_it_t it=timers.begin(); it!=timers.end(); it++)
		delete *it;
	timers.erase(timers.begin(),timers.end());
}

void EventRouter::addListener(EventListener* el, EventBase::EventGeneratorID_t egid) {
	bool hadListener=hasListeners(egid);
	listeners.addMapping(el,egid); 
	if(!hadListener)
		postEvent(new EventBase(EventBase::erouterEGID,egid,EventBase::activateETID));
	else
		postEvent(new EventBase(EventBase::erouterEGID,egid,EventBase::statusETID));
}
void EventRouter::addListener(EventListener* el, EventBase::EventGeneratorID_t egid, unsigned int sid) {
	bool hadListener=hasListeners(egid);
	for(unsigned int et=0; et<EventBase::numETIDs; et++)
		listeners.addMapping(el,egid,sid,(EventBase::EventTypeID_t)et);
	if(!hadListener)
		postEvent(new EventBase(EventBase::erouterEGID,egid,EventBase::activateETID));
	else
		postEvent(new EventBase(EventBase::erouterEGID,egid,EventBase::statusETID));
}
void EventRouter::addListener(EventListener* el, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid) {
	bool hadListener=hasListeners(egid);
	listeners.addMapping(el,egid,sid,etid);
	if(!hadListener)
		postEvent(new EventBase(EventBase::erouterEGID,egid,EventBase::activateETID));
	else
		postEvent(new EventBase(EventBase::erouterEGID,egid,EventBase::statusETID));
}
void EventRouter::addListener(EventListener* el, const EventBase& e) {
	if(e.getGeneratorID()==EventBase::timerEGID)
		addTimer(el,e.getSourceID(),e.getDuration());
	else {
		bool hadListener=hasListeners(e.getGeneratorID());
		listeners.addMapping(el,e.getGeneratorID(),e.getSourceID(),e.getTypeID());
		if(!hadListener)
			postEvent(new EventBase(EventBase::erouterEGID,e.getGeneratorID(),EventBase::activateETID));
		else
			postEvent(new EventBase(EventBase::erouterEGID,e.getGeneratorID(),EventBase::statusETID));
	}
}

void EventRouter::removeListener(EventListener* el) {
	for(unsigned int eg=0; eg<EventBase::numEGIDs; eg++)
		removeListener(el,(EventBase::EventGeneratorID_t)eg);
}
void EventRouter::removeListener(EventListener* el, EventBase::EventGeneratorID_t egid) {
	if(egid==EventBase::timerEGID)
		removeTimer(el);
	else {
		if(!listeners.removeMapping(el,egid))
			return; //nothing was removed, don't want to clean up or throw an event
		listeners.clean(egid);
		if(!hasListeners(egid))
			postEvent(new EventBase(EventBase::erouterEGID,egid,EventBase::deactivateETID));
		else
			postEvent(new EventBase(EventBase::erouterEGID,egid,EventBase::statusETID));
	}
}
void EventRouter::removeListener(EventListener* el, EventBase::EventGeneratorID_t egid, unsigned int sid) {
	if(egid==EventBase::timerEGID)
		removeTimer(el,sid);
	else {
		unsigned int removed=0;
		for(unsigned int et=0; et<EventBase::numETIDs; et++)
			removed+=listeners.removeMapping(el,egid,sid,(EventBase::EventTypeID_t)et);
		if(!removed)
			return; //nothing was removed, don't want to clean up or throw an event
		listeners.clean(egid);
		if(!hasListeners(egid))
			postEvent(new EventBase(EventBase::erouterEGID,egid,EventBase::deactivateETID));
		else
			postEvent(new EventBase(EventBase::erouterEGID,egid,EventBase::statusETID));
	}
}
void EventRouter::removeListener(EventListener* el, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid) {
	if(egid==EventBase::timerEGID) {
		if(etid==EventBase::statusETID)
			removeTimer(el,sid);
	} else {
		if(!listeners.removeMapping(el,egid,sid,etid))
			return; //nothing was removed, don't want to clean up or throw an event
		listeners.clean(egid);
		if(!hasListeners(egid))
			postEvent(new EventBase(EventBase::erouterEGID,egid,EventBase::deactivateETID));
		else
			postEvent(new EventBase(EventBase::erouterEGID,egid,EventBase::statusETID));
	}
}
void EventRouter::removeListener(EventListener* el, const EventBase& e) {
	if(e.getGeneratorID()==EventBase::timerEGID)
		removeTimer(el,e.getSourceID());
	else {
		if(!listeners.removeMapping(el,e.getGeneratorID(),e.getSourceID(),e.getTypeID()))
			return; //nothing was removed, don't want to clean up or throw an event
		listeners.clean(e.getGeneratorID());
		if(!hasListeners(e.getGeneratorID()))
			postEvent(new EventBase(EventBase::erouterEGID,e.getGeneratorID(),EventBase::deactivateETID));
		else
			postEvent(new EventBase(EventBase::erouterEGID,e.getGeneratorID(),EventBase::statusETID));
	}
}

void EventRouter::forgetListener(EventListener* el) {
	removeListener(el);
}

void EventRouter::addTrapper(EventTrapper* el, const EventBase& e) {
	bool hadListener=hasListeners(e.getGeneratorID());
	trappers.addMapping(el,e.getGeneratorID(),e.getSourceID(),e.getTypeID());
	if(!hadListener)
		postEvent(new EventBase(EventBase::erouterEGID,e.getGeneratorID(),EventBase::activateETID));
	else
		postEvent(new EventBase(EventBase::erouterEGID,e.getGeneratorID(),EventBase::statusETID));
}
/*! Note that since timers are not broadcast, they cannot be trapped.  Only the EventListener which requested the timer will receive that timer. */
void EventRouter::addTrapper(EventTrapper* el, EventBase::EventGeneratorID_t egid) {
	bool hadListener=hasListeners(egid);
	trappers.addMapping(el,egid);
	if(!hadListener)
		postEvent(new EventBase(EventBase::erouterEGID,egid,EventBase::activateETID));
	else
		postEvent(new EventBase(EventBase::erouterEGID,egid,EventBase::statusETID));
}
/*! Note that since timers are not broadcast, they cannot be trapped.  Only the EventListener which requested the timer will receive that timer. */
void EventRouter::addTrapper(EventTrapper* el, EventBase::EventGeneratorID_t egid, unsigned int sid) {
	bool hadListener=hasListeners(egid);
	for(unsigned int et=0; et<EventBase::numETIDs; et++)
		trappers.addMapping(el,egid,sid,(EventBase::EventTypeID_t)et);
	if(!hadListener)
		postEvent(new EventBase(EventBase::erouterEGID,egid,EventBase::activateETID));
	else
		postEvent(new EventBase(EventBase::erouterEGID,egid,EventBase::statusETID));
}
/*! Note that since timers are not broadcast, they cannot be trapped.  Only the EventListener which requested the timer will receive that timer. */
void EventRouter::addTrapper(EventTrapper* el, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid) {
	bool hadListener=hasListeners(egid);
	trappers.addMapping(el,egid,sid,etid);
	if(!hadListener)
		postEvent(new EventBase(EventBase::erouterEGID,egid,EventBase::activateETID));
	else
		postEvent(new EventBase(EventBase::erouterEGID,egid,EventBase::statusETID));
}

/*! Note that since timers are not broadcast, they cannot be trapped.  Only the EventListener which requested the timer will receive that timer. */
void EventRouter::addTrapper(EventTrapper* el) {
	for(unsigned int eg=0; eg<EventBase::numEGIDs; eg++)
		addTrapper(el,(EventBase::EventGeneratorID_t)eg);
}


void EventRouter::removeTrapper(EventTrapper* el, const EventBase& e) {
	if(!trappers.removeMapping(el,e.getGeneratorID(),e.getSourceID(),e.getTypeID()))
		return; //nothing was removed, don't want to clean up or throw an event
	trappers.clean(e.getGeneratorID());
	if(!hasListeners(e.getGeneratorID()))
		postEvent(new EventBase(EventBase::erouterEGID,e.getGeneratorID(),EventBase::deactivateETID));
	else
		postEvent(new EventBase(EventBase::erouterEGID,e.getGeneratorID(),EventBase::statusETID));
}
void EventRouter::removeTrapper(EventTrapper* el, EventBase::EventGeneratorID_t egid) {
	if(!trappers.removeMapping(el,egid))
		return; //nothing was removed, don't want to clean up or throw an event
	trappers.clean(egid);
	if(!hasListeners(egid))
		postEvent(new EventBase(EventBase::erouterEGID,egid,EventBase::deactivateETID));
	else
		postEvent(new EventBase(EventBase::erouterEGID,egid,EventBase::statusETID));
}
void EventRouter::removeTrapper(EventTrapper* el, EventBase::EventGeneratorID_t egid, unsigned int sid) {
	int removed=0;
	for(unsigned int et=0; et<EventBase::numETIDs; et++)
		removed+=trappers.removeMapping(el,egid,sid,(EventBase::EventTypeID_t)et);
	if(!removed)
		return; //nothing was removed, don't want to clean up or throw an event
	trappers.clean(egid);
	if(!hasListeners(egid))
		postEvent(new EventBase(EventBase::erouterEGID,egid,EventBase::deactivateETID));
	else
		postEvent(new EventBase(EventBase::erouterEGID,egid,EventBase::statusETID));
}
void EventRouter::removeTrapper(EventTrapper* el, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid) {
	if(!trappers.removeMapping(el,egid,sid,etid))
		return; //nothing was removed, don't want to clean up or throw an event
	trappers.clean(egid);
	if(!hasListeners(egid))
		postEvent(new EventBase(EventBase::erouterEGID,egid,EventBase::deactivateETID));
	else
		postEvent(new EventBase(EventBase::erouterEGID,egid,EventBase::statusETID));
}

void EventRouter::removeTrapper(EventTrapper* el) {
	for(unsigned int eg=0; eg<EventBase::numEGIDs; eg++)
		removeTrapper(el,(EventBase::EventGeneratorID_t)eg);
}

void EventRouter::processEventBuffer() { //clears buffered events
	if(events.size()>0)
		doSendBuffer();
}
void EventRouter::doSendBuffer() {
	if(doSendBufferLock) {
		std::cout << "*** WARNING recursive call to doSendBuffer()" << std::endl;
		return;
	}
	doSendBufferLock=true;
	unsigned int start=get_time();
	//	doSendEvent(EventBase(EventBase::eventRouterEGID,0,EventBase::activateETID,0));
	//important to use indexes instead of iterators in case the event listeners decide to post more events during processing
	for(unsigned int i=0; i<events.size(); i++) {
		doSendEvent(*events[i]);
		delete events[i];
	}
	events.erase(events.begin(),events.end());
	doSendBufferLock=false;
	lastBufClear=start;
	//	doSendEvent(EventBase(EventBase::eventRouterEGID,0,EventBase::deactivateETID,get_time()-start));
}

void EventRouter::processEvent(const EventBase& e) {
	// want to make sure we don't send events out of order...
	if(events.size()>0)
		doSendBuffer();
	doSendEvent(e);
}
void EventRouter::doSendEvent(const EventBase& e) {
	//	cout << "doSendEvent("<<e.getName()<<")..." << flush;
	std::vector<EventTrapper*> t;
	trappers.getMapping(e,t);
	for(std::vector<EventTrapper*>::iterator it=t.begin(); it!=t.end(); it++)
		if(trappers.verifyMapping(*it,e))
			if((*it)->trapEvent(e))
				return;
	std::vector<EventListener*> l;
	listeners.getMapping(e,l);
	for(std::vector<EventListener*>::iterator it=l.begin(); it!=l.end(); it++)
		if(listeners.verifyMapping(*it,e))
			(*it)->processEvent(e);
	//	cout << "done." << flush;
}

EventRouter::EventMapper::EventMapper() {
	for(unsigned int eg=0; eg<EventBase::numEGIDs; eg++)
		for(unsigned int et=0; et<EventBase::numETIDs; et++)
			filteredevents[eg][et]=NULL;
}

void EventRouter::EventMapper::addMapping(void* el, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid) {
	if(filteredevents[egid][etid]==NULL) //if this is the first subscriber to this EGID and ETID
		filteredevents[egid][etid]=new SIDtoListenerVectorMap_t(); 
	SIDtoListenerVectorMap_t::iterator it=filteredevents[egid][etid]->find(sid); // now find subscribers to the source id as well
	std::vector<void*>* elv=NULL;
	if(it==filteredevents[egid][etid]->end()) { // if this is the first subscriber to the source ID
		std::pair<const unsigned int,std::vector<void*> > p(sid,std::vector<void*>());
		//		p.first=sid; //p.second is a vector, only needs to be constructed
		filteredevents[egid][etid]->insert(p);
		elv=&(*filteredevents[egid][etid]->find(sid)).second;
	} else {
		elv=&(*it).second;
	}
	elv->push_back(el); // now that everything's set up, we can add the listener
}

bool EventRouter::EventMapper::removeMapping(void* el, EventBase::EventGeneratorID_t egid) {
	// remove listener from allevents
	unsigned int numlist=allevents[egid].size();
	allevents[egid].erase(remove(allevents[egid].begin(),allevents[egid].end(),el),allevents[egid].end());
	bool hadListener=allevents[egid].size()!=numlist;
	
	// now remove listener from all of the filtered events
	for(unsigned int et=0; et<EventBase::numETIDs; et++) {
		SIDtoListenerVectorMap_t* mapping=filteredevents[egid][et];
		if(mapping!=NULL) { // if there are subscribers to this egid/etid
			SIDtoListenerVectorMap_t::iterator mapit=mapping->begin();
			for(mapit=mapping->begin(); mapit!=mapping->end(); mapit++) {// go through each sourceID, delete EL
				std::vector<void*> * v=&(*mapit).second;
				std::vector<void*>::iterator last=remove(v->begin(),v->end(),el);
				if(last!=v->end()) {
					hadListener=true;
					v->erase(last,v->end());
				}
			}
		}
	}
	return hadListener;
}

bool EventRouter::EventMapper::removeMapping(void* el, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid) {
	bool hadListener=false;
	SIDtoListenerVectorMap_t* mapping=filteredevents[egid][etid];
	if(mapping!=NULL) { // if there are subscribers to this egid/etid
		SIDtoListenerVectorMap_t::iterator mapit=mapping->find(sid);
		if(mapit!=mapping->end()) {
			std::vector<void*> * v=&(*mapit).second;
			std::vector<void*>::iterator last=remove(v->begin(),v->end(),el);
			if(last!=v->end()) {
				hadListener=true;
				v->erase(last,v->end());
			}
		}
	}
	return hadListener;
}

void EventRouter::EventMapper::clean() {
	for(unsigned int eg=0; eg<EventBase::numEGIDs; eg++)
		clean((EventBase::EventGeneratorID_t)eg);
}
void EventRouter::EventMapper::clean(EventBase::EventGeneratorID_t egid) {
	// first, remove any empty sid vectors from all the mappings
	for(unsigned int et=0; et<EventBase::numETIDs; et++) {
		SIDtoListenerVectorMap_t* mapping=filteredevents[egid][et];
		if(mapping!=NULL) { // if there are subscribers to this egid/etid
			SIDtoListenerVectorMap_t::iterator mapit=mapping->begin();
			bool done=false;
			while(!done) {
				done=true;
				for(mapit=mapping->begin(); mapit!=mapping->end(); mapit++) { // go through each sourceID vector
					if((*mapit).second.size()==0) {
						mapping->erase(mapit);
						done=false;
						break;
					}
				}
			}
		}
	}
	// now remove any empty mappings
	for(unsigned int et=0; et<EventBase::numETIDs; et++) {
		SIDtoListenerVectorMap_t* mapping=filteredevents[egid][et];
		if(mapping!=NULL) { // if there are subscribers to this egid/etid
			if(mapping->size()==0) {
				delete mapping;
				filteredevents[egid][et]=NULL;
			}
		}
	}
}

void EventRouter::EventMapper::clear() {
	for(unsigned int eg=0; eg<EventBase::numEGIDs; eg++) {
		for(unsigned int et=0; et<EventBase::numETIDs; et++) {
			SIDtoListenerVectorMap_t* mapping=filteredevents[eg][et];
			if(mapping!=NULL) { // don't beat a dead horse!
				mapping->erase(mapping->begin(),mapping->end());
				delete mapping;
				filteredevents[eg][et]=NULL;
			}
		}
	}
}

bool EventRouter::EventMapper::hasMapping(EventBase::EventGeneratorID_t egid) {
	if(allevents[egid].size()>0)
		return true;
	for(unsigned int et=0; et<EventBase::numETIDs; et++) {
		SIDtoListenerVectorMap_t* mapping=filteredevents[egid][et];
		if(mapping!=NULL) {
			SIDtoListenerVectorMap_t::iterator mapit=mapping->begin();
			for(mapit=mapping->begin(); mapit!=mapping->end(); mapit++)
				if((*mapit).second.size()>0)
					return true;
		}
	}
	return false;
}

bool EventRouter::EventMapper::hasMapping(EventBase::EventGeneratorID_t egid, unsigned int sid) {
	if(allevents[egid].size()>0)
		return true;
	for(unsigned int et=0; et<EventBase::numETIDs; et++) {
		SIDtoListenerVectorMap_t* mapping=filteredevents[egid][et];
		if(mapping!=NULL) {
			SIDtoListenerVectorMap_t::iterator mapit=mapping->find(sid);
			if(mapit!=mapping->end() && (*mapit).second.size()>0)
				return true;
		}
	}
	return false;
}

bool EventRouter::EventMapper::hasMapping(EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid) {
	if(allevents[egid].size()>0)
		return true;
	SIDtoListenerVectorMap_t* mapping=filteredevents[egid][etid];
	if(mapping!=NULL) {
		SIDtoListenerVectorMap_t::iterator mapit=mapping->find(sid);
		if(mapit!=mapping->end())
			return ((*mapit).second.size()>0);
	}
	return false;
}

template<class T>
void EventRouter::EventMapper::getMapping(const EventBase& e, std::vector<T*>& ls) {
	// first get all the filtered subscribers (tricky!)
	std::vector<void*>* elv=NULL;
	SIDtoListenerVectorMap_t* sidtovm=filteredevents[e.getGeneratorID()][e.getTypeID()];
	if(sidtovm!=NULL) { // if there's a map (at least one EL is filtering on this EGID and ETID)
		SIDtoListenerVectorMap_t::iterator mapit=sidtovm->find(e.getSourceID()); // find listening for this source id
		if(mapit!=sidtovm->end()) { // if there's at least one is filtering on this sourceID as well
			elv=&(*mapit).second; // now go through them all
			for(std::vector<void*>::iterator elit=elv->begin(); elit!=elv->end(); elit++)
				ls.push_back(static_cast<T*>(*elit));
		}
	}
	// now get the 'all events' subscribers
	elv=&allevents[e.getGeneratorID()];
	for(std::vector<void*>::iterator elit=elv->begin(); elit!=elv->end(); elit++)
		ls.push_back(static_cast<T*>(*elit));
}

bool EventRouter::EventMapper::verifyMapping(void * listener, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid) {
	// first get all the filtered subscribers (tricky!)
	std::vector<void*>* elv=NULL;
	SIDtoListenerVectorMap_t* sidtovm=filteredevents[egid][etid];
	if(sidtovm!=NULL) { // if there's a map (at least one EL is filtering on this EGID and ETID)
		SIDtoListenerVectorMap_t::iterator mapit=sidtovm->find(sid); // find listening for this source id
		if(mapit!=sidtovm->end()) { // if there's at least one is filtering on this sourceID as well
			elv=&(*mapit).second; // now go through them all
			for(std::vector<void*>::iterator elit=elv->begin(); elit!=elv->end(); elit++)
				if(*elit==listener)
					return true;
		}
	}
	// now get the 'all events' subscribers
	elv=&allevents[egid];
	for(std::vector<void*>::iterator elit=elv->begin(); elit!=elv->end(); elit++)
		if(*elit==listener)
			return true;

	// if we haven't found it, doesn't exist:
	return false;
}


/*! @file
 * @brief Implements EventRouter class, for distribution and trapping of events to listeners
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_2_1 $
 * $Revision: 1.13 $
 * $State: Exp $
 * $Date: 2004/10/07 22:14:17 $
 */







// Use hasListeners(*) it's faster, i doubt anyone would really care how many... (but just in case...)
/*
unsigned int EventRouter::numListeners(EventBase::EventGeneratorID_t egid) {
	unsigned int ans=allevents[egid].size();
	for(unsigned int et=0; et<EventBase::numETIDs; et++) {
		SIDtoListenerVectorMap_t* mapping=filteredevents[egid][et];
		if(mapping!=NULL) {
			SIDtoListenerVectorMap_t::iterator mapit=mapping->begin();
			for(mapit=mapping->begin(); mapit!=mapping->end(); mapit++)
				ans+=(*mapit).second.size();
		}
	}
	return ans;
}

bool EventRouter::numListeners(EventBase::EventGeneratorID_t egid, unsigned int sid) {
	unsigned int ans=allevents[egid].size();
	for(unsigned int et=0; et<EventBase::numETIDs; et++) {
		SIDtoListenerVectorMap_t* mapping=filteredevents[egid][et];
		if(mapping!=NULL) {
			SIDtoListenerVectorMap_t::iterator mapit=mapping->find(sid);
			if(mapit!=mapping->end())
				ans+=(*mapit).second.size();
		}
	}
	return false;
}

unsigned int EventRouter::numListeners(EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid) {
	unsigned int ans=allevents[egid].size();
	SIDtoListenerVectorMap_t* mapping=filteredevents[egid][etid];
	if(mapping!=NULL) {
		SIDtoListenerVectorMap_t::iterator mapit=mapping->find(sid);
		if(mapit!=mapping->end())
			ans+=(*mapit).second.size();
	}
	return ans;
}
*/
