//-*-c++-*-
#ifndef INCLUDED_EventBase_h
#define INCLUDED_EventBase_h

#include "Shared/get_time.h"
#include "Shared/LoadSave.h"
#include <string>

//! The basis of events passed around the high level system
/*! Contains the list of 'known' event generators in #EventGeneratorID_t
 *  and #EventGeneratorNames[].  If you want to make a new generator, all
 *  you have to do is add a new entry to the ID list (#EventGeneratorID_t)
 *  and then put its name in the #EventGeneratorNames[] array.
 *
 *  The #EventGeneratorID_t list should contain links to the
 *  generators' documentation, or will directly give information about
 *  what to expect in events from that generator.
 *  
 *  Alternatively, there is an 'unlicensed spectrum' available under
 *  the #unknownEGID.  You can send out events from that generator just
 *  like any other, but it should only be used for quick tests and hacking
 *  around...
 *
 *  The #sourceID (just an unsigned int) is generator specific.  A SID
 *  of @a i from the button generator (#buttonEGID) will refer to a
 *  particular button, whereas a SID from a vision detector (say
 *  #visObjEGID) refers to seeing a particular object.  These SIDs are
 *  usually defined in the generators themselves.
 *
 *  If more information needs to be sent along with the event, the
 *  cleanest solution is to create a subclass of EventBase to hold the
 *  additional information.  For example, you can see the existing
 *  subclasses in the inheritance diagram above.  If you want to use a
 *  quick hack however, you could assume the pointer size is the same
 *  as an unsigned int and just pass a pointer to data as the SID.
 *  Your funeral, err, I mean your call. ;)
 *
 *  The #duration field is also generator specific - some may refer to
 *  the time since the last activation event (e.g. button events)
 *  where as others refer to time since last status (e.g. sensors
 *  updates)
 *
 */
class EventBase : public LoadSave {
 public:
	//! Lists all possible event generator ids
	/*! An event generator is a abstract source of events, used for listening to and parsing certain classes of events
	 *
	 *  IF YOU ADD AN EVENT GENERATOR, DON'T FORGET TO NAME IT (EventBase::EventGeneratorNames, actual names are in EventBase.cc)*/
	enum EventGeneratorID_t {
		unknownEGID=0,    //!< default EGID, used if you forget to set it, probably not a good idea to use this for anything except errors or testing quick hacks
		aiEGID,           //!< not being used, yet (might use this when AI makes decisions?)
		audioEGID,        //!< Sends an event when a sound starts/ends playback, status events as chained sounds end; SID is SoundManager::Play_ID; duration is playtime
		buttonEGID,       //!< Sends activate event for button down, deactivate for button up.  Status events only for when pressure sensitive buttons' reading changes. (on sensorEGID updates); SIDs are from ButtonOffset_t in the namespace of the target model (e.g. ERS210Info::ButtonOffset_t); duration is button down time
		erouterEGID,      //!< Sends activate event on first listener, deactivate on last listener, and status on other listener changes.; SID is the generator ID affected
		estopEGID,        //!< Sends an event when the estop is turned on or off; SID is the MotionManager::MC_ID of the EmergencyStopMC; duration is length of estop activation
		locomotionEGID,   //!< Sends events regarding transportation in the world; you can/should assume these will all be LocomotionEvent classes; SID is MotionManager::MC_ID of posting MotionCommand; duration is the time since last velocity change of that MC. (You could generate these for things other than walking...)
		micOSndEGID,      //!< Sends a DataEvent<OSoundVectorData> for every audio buffer received from the system; SID and duration are always 0 (This is generated by the MainObj instantiation of MMCombo)
		micRawEGID,       //!< reserved for future use
		micFFTEGID,       //!< reserved for future use
		motmanEGID,       //!< Sends events when a MotionCommand is added or removed, SID is is the MotionManager::MC_ID, duration is always 0; individual MotionCommands may throw status events to signal intermediary status
		powerEGID,        //!< Sends events for low power warnings, temperature, etc. see PowerSourceID::PowerSourceID_t
		sensorEGID,       //!< Sends a status event when new sensor readings are available. see SensorSourceID::SensorSourceID_t
		stateMachineEGID, //!< Sends an event upon entering and leaving a StateNode; SID is pointer to the StateNode; duration is always 0
    textmsgEGID,      //!< Sends status events when a text msg is received on console; generated by the Controller, SID is always -1; durations is always 0 (see Controller for more information)
		timerEGID,        //!< Sends timer events; you set timers explicitly, you don't have to listen as well. (See EventRouter::addTimer()) There's no cross-talk, only the listener which requested the timer will receive it; SID is whatever you requested it to be; duration is the time (since boot, in ms) that the timer was supposed to go off; these are always status
		visOFbkEGID,      //!< Sends a DataEvent < OFbkImageVectorData > for every camera image received from the system; SID and duration are always 0 (This is generated by the MainObj instantiation of MMCombo)
		visRawCameraEGID, //!< Sends a FilterBankEvent when new raw camera images are available; SID is whatever value you gave during setup (typically in StartupBehavior_SetupVision.cc), duration is always 0
		visInterleaveEGID,//!< Sends a FilterBankEvent when new interleaved images are available; SID is whatever value you gave during setup (typically in StartupBehavior_SetupVision.cc), duration is always 0
		visJPEGEGID,      //!< Sends a FilterBankEvent when JPEG compressed images are available; SID is whatever value you gave during setup (typically in StartupBehavior_SetupVision.cc), duration is always 0
		visSegmentEGID,   //!< Sends a SegmentedColorFilterBankEvent when color segmentated images are available; SID is whatever value you gave during setup (typically in StartupBehavior_SetupVision.cc), duration is always 0
		visRLEEGID,       //!< Sends a SegmentedColorFilterBankEvent when RLE encoded color segmentated images are available; SID is whatever value you gave during setup (typically in StartupBehavior_SetupVision.cc), duration is always 0
		visRegionEGID,    //!< Sends a SegmentedColorFilterBankEvent when color regions are available; SID is whatever value you gave during setup (typically in StartupBehavior_SetupVision.cc), duration is always 0
		visObjEGID,       //!< Sends VisionObjectEvents for objects detected in camera images; SID is whatever value you gave during setup (typically in StartupBehavior_SetupVision.cc), duration is always 0
		wmVarEGID,        //!< Sends an event when a watched memory is changed; source id is pointer to WMEntry
		worldModelEGID,   //!< not being used, yet (for when objects are detected/lost?)
		numEGIDs          //!< the number of generators available
	};

	//! Holds string versions of each of the generator's names, handy for debugging so you can output the events as readable strings (you'll find this in EventBase.cc since it can't go in the header or we get multiply-defined errors during linking)
	static const char* const EventGeneratorNames[numEGIDs];
	
	//! an event type id is used to denote whether it's the first in a sequence (button down), in a sequence (button still down), or last (button up)
	enum EventTypeID_t {
		activateETID,   //!< Start of an event sequence, e.g. button down
		statusETID,     //!< Indicates a value has changed, e.g. new sensor readings
		deactivateETID, //!< Last of a series of events, e.g. button up
		numETIDs        //!< the number of different event types
	};

	/*! @name Constructors/Destructors */
	//! constructor
	/*! @see EventRouter::postEvent() */
	EventBase(); 
	EventBase(EventGeneratorID_t gid, unsigned int sid, EventTypeID_t tid, unsigned int dur=0);
	EventBase(EventGeneratorID_t gid, unsigned int sid, EventTypeID_t tid, unsigned int dur, const std::string& n, float mag);
	virtual ~EventBase() {} //!< destructor
	//@}

	/*! @name Methods */
	virtual const std::string& getName() const { return stim_id; } //!< gets the name of the event - useful for debugging, outputs
	virtual EventBase& setName(const std::string& n) { nameisgen=false; stim_id=n; return *this; } //!< sets name to a given string, prevents overwriting by generated names

	virtual float getMagnitude() const { return magnitude; } //!< gets "strength" of event - by default 1 for activate and status events, 0 for deactivate events
	virtual EventBase& setMagnitude(float m) { magnitude=m; return *this; }//!< sets "strength" of event - you may want to override the default values (see getMagnitude())

	virtual unsigned int getTimeStamp() const { return timestamp; } //!< time event was created

	virtual EventGeneratorID_t getGeneratorID() const { return genID; } /*!< @brief gets the generator ID for this event @see EventGeneratorID_t */
	virtual EventBase& setGeneratorID(EventGeneratorID_t gid) { genID=gid; genName(); return *this; } /*!< @brief sets the generator ID for this event @see EventGeneratorID_t */
	
	virtual unsigned int getSourceID() const { return sourceID; } /*!< @brief gets the source ID for this event @see sourceID */
	virtual EventBase& setSourceID(unsigned int sid) { sourceID=sid; genName(); return *this; } /*!< @brief sets the source ID for this event @see sourceID */
	
	virtual EventTypeID_t getTypeID() const { return typeID; } /*!< @brief gets the type ID @see EventTypeID_t */
	virtual EventBase& setTypeID(EventTypeID_t tid) { typeID=tid; genName(); return *this; } /*!< @brief sets the type ID @see EventTypeID_t */

	virtual unsigned int getDuration() const { return duration; } /*!< @brief gets the time since the beginning of this sequence (the timestamp of the activate event) @see duration */
	virtual EventBase& setDuration(unsigned int d) { duration = d; return *this; }/*!< @brief sets the time since the beginning of this sequence (the timestamp of the activate event) @see duration */

	virtual const std::string& resetName() { nameisgen=true; genName(); return stim_id; } //!< resets name to generated form, overwriting any previous name
	virtual bool isCustomName() const { return !nameisgen; } //!< returns true if not using the generated name

	//! generates a description of the event with variable verbosity 
	/*! @param showTypeSpecific will signal subclasses to add additional information
	 *  @param verbosity can be one of the following values:
	 *    - 0 - the name and type
	 *    - 1 - the name, type, duration, and timestamp
	 *    - 2 and above - the name, type, duration, and magnitude
	 *  if showTypeSpecific, additional fields will be added after the common fields listed above. */
	virtual std::string getDescription(bool showTypeSpecific=true, unsigned int verbosity=0) const; 

	inline bool operator<(const EventBase& e) const { return timestamp<e.timestamp; }

	//! is true if the genID, typeID, and sourceID's all match
	virtual bool operator==(const EventBase& eb) const {
		return (sourceID==eb.sourceID && genID==eb.genID && typeID==eb.typeID);
	}
	//!tests to see if events have the same generator and source IDs
	bool sameGenSource(const EventBase& eb) const { return genID==eb.genID && sourceID==eb.sourceID; }
	
	bool longerThan(const EventBase& eb) const { return duration>eb.duration && *this==eb; } //!< compares event duration and ensures same event generator, source, and type - useful for event masks
	bool shorterThan(const EventBase& eb) const { return duration<eb.duration && *this==eb; }//!< compares event duration and ensures same event generator, source, and type - useful for event masks
	bool equalOrLongerThan(const EventBase& eb) const { return duration>=eb.duration && *this==eb; }//!< compares event duration and ensures same event generator, source, and type - useful for event masks
	bool equalOrShorterThan(const EventBase& eb) const { return duration<=eb.duration && *this==eb; }//!< compares event duration and ensures same event generator, source, and type - useful for event masks
	//@}

	//! Useful for serializing events to send between processes
	/*! @name LoadSave interface */
	virtual unsigned int getBinSize() const;
	virtual unsigned int LoadBuffer(const char buf[], unsigned int len);
	virtual unsigned int SaveBuffer(char buf[], unsigned int len) const;
	//@}
 protected:
  std::string stim_id; //!< the name of the event, use the same name consistently or else will be seen as different stimuli
  float magnitude; //!< the current "strength" of the event/stimuli... MAKE SURE this gets set to ZERO IF event is DEACTIVATE
  unsigned int timestamp; //!< the time the event was created - set automatically by constructor

	bool nameisgen; //!< tracks whether the current name (stim_id) is generated or set
	virtual void genName(); //!< This does the actual generation of names based on genID, sourceID, and typeID

	EventGeneratorID_t genID; //!< generator ID, see EventGeneratorID_t
	EventTypeID_t typeID; //!< type ID, see EventTypeID_t
	unsigned int sourceID; /*!< @brief the source ID for this event
													* Source IDs are defined by the generator that made it.  This should
													* give authors flexibility to design their modules without having to
													* worry about ID space collision */
	unsigned int duration; /*!< @brief the time since this sequence started (like, how long the
													*   button has been pressed); not all generators will set this;
													*   Typically, this would be 0 for activate,
													*   (activate.timestamp-::get_time()) for status and deactivate */
};

/*! @file
 * @brief Describes EventBase, the basic class for sending events around the system
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_2_1 $
 * $Revision: 1.27 $
 * $State: Exp $
 * $Date: 2004/10/18 16:59:20 $
 */

#endif
