#include "CameraBehavior.h"
#include "Events/EventRouter.h"
#include "Shared/ERS210Info.h"
#include "Shared/ERS220Info.h"
#include "Shared/ERS7Info.h"
#include "Wireless/Socket.h"
#include "Shared/WorldState.h"
#include "SoundPlay/SoundManager.h"
#include "Shared/Config.h"
#include "Shared/ProjectInterface.h"
#include "Motion/LedMC.h"
#include "Motion/MMAccessor.h"

#include "Vision/FilterBankGenerator.h"
#include "Vision/RawCameraGenerator.h"
#include "Vision/InterleavedYUVGenerator.h"
#include "Vision/JPEGGenerator.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>

void
CameraBehavior::DoStart() {
	BehaviorBase::DoStart();
	if(state->robotDesign&WorldState::ERS210Mask) {
		camera_click.setSourceID(ERS210Info::HeadFrButOffset);
	} else if(state->robotDesign&WorldState::ERS220Mask) {
		camera_click.setSourceID(ERS220Info::HeadFrButOffset);
	} else if(state->robotDesign&WorldState::ERS7Mask) {
		camera_click.setSourceID(ERS7Info::HeadButOffset);
	}
	initIndex();
	sndman->LoadFile("camera.wav");
	ledID=motman->addPersistentMotion(SharedObject<LedMC>());
	erouter->addListener(this,camera_click);
}

void CameraBehavior::DoStop() {
	erouter->removeListener(this);
	sndman->ReleaseFile("camera.wav");
	motman->removeMotion(ledID);
	BehaviorBase::DoStop();
}

	
/*! The format used depends on the current config settings.  If JPEG
 *  is the current choice, then a JPEG file will be written.
 *  Otherwise, RawCameraGenerator::SaveFile() will be called.
 */
void
CameraBehavior::processEvent(const EventBase& e) {
	if(e.shorterThan(camera_click))
		return;

	{
		MMAccessor<LedMC> leds(ledID);
		leds->cset(FaceLEDMask,2.0/3.0);
		leds->set(TopBrLEDMask,1);
	}

	if(config->vision.rawcam_compression==Config::vision_config::COMPRESS_NONE) {
		//this is our own odd little format, would be nice to save a TIFF or something instead

		// open file
		FILE * f=openNextFile(".raw");
		if(f==NULL) //error message already displayed in openNextFile()
			return;

		//! write actual image data
		if(config->vision.rawcam_encoding==Config::vision_config::ENCODE_COLOR) {
			FilterBankGenerator * gen=ProjectInterface::defInterleavedYUVGenerator; // just an alias for readability
			gen->selectSaveImage(ProjectInterface::doubleLayer,InterleavedYUVGenerator::CHAN_YUV);
			unsigned int len=gen->SaveFileStream(f);
			if(len==0) {
				serr->printf("Error saving file\n");
				sndman->PlayFile(config->controller.error_snd);
				return;
			}
		} else if(config->vision.rawcam_encoding==Config::vision_config::ENCODE_SINGLE_CHANNEL) {
			FilterBankGenerator * gen=ProjectInterface::defRawCameraGenerator; // just an alias for readability
			gen->selectSaveImage(ProjectInterface::doubleLayer,config->vision.rawcam_channel);
			unsigned int len=gen->SaveFileStream(f);
			if(len==0) {
				serr->printf("Error saving file\n");
				sndman->PlayFile(config->controller.error_snd);
				return;
			}
		}
		
		// close file
		fclose(f);

	} else if(config->vision.rawcam_compression==Config::vision_config::COMPRESS_JPEG) {
		//save a JPEG image
		JPEGGenerator * jpeg=NULL; // we'll set this to pick between the color jpeg or a single channel grayscale jpeg
		unsigned int chan=0; // and this will hold the channel to use out of that jpeg generator
		if(config->vision.rawcam_encoding==Config::vision_config::ENCODE_COLOR)
			jpeg=dynamic_cast<JPEGGenerator*>(ProjectInterface::defColorJPEGGenerator);
		else if(config->vision.rawcam_encoding==Config::vision_config::ENCODE_SINGLE_CHANNEL) {
			jpeg=dynamic_cast<JPEGGenerator*>(ProjectInterface::defGrayscaleJPEGGenerator);
			chan=config->vision.rawcam_channel;
		}
		if(jpeg!=NULL) {
			unsigned int tmp_q=jpeg->getQuality(); //temporary storage so we can reset the default
			jpeg->setQuality(92);
			
			// open file
			FILE * f=openNextFile(".jpg");
			if(f==NULL) //error message already displayed in openNextFile()
				return;

			//! write actual image data
			unsigned char * imgbuf=jpeg->getImage(ProjectInterface::doubleLayer,chan);
			unsigned int writ=fwrite(imgbuf,jpeg->getImageSize(ProjectInterface::doubleLayer,chan),1,f);
			if(writ==0) {
				serr->printf("Error saving file\n");
				sndman->PlayFile(config->controller.error_snd);
				return;
			}

			// close file
			fclose(f);
			
			jpeg->setQuality(tmp_q);
		}
	}


	{
		MMAccessor<LedMC> leds(ledID);
		leds->clear();
		leds->flash(TopBrLEDMask,700);
		leds->flash(TopLLEDMask|TopRLEDMask,500);
		leds->flash(MidLLEDMask|MidRLEDMask,300);
		leds->flash(BotLLEDMask|BotRLEDMask,100);
	}

	sout->printf("done\n");
}

FILE *
CameraBehavior::openNextFile(const std::string& ext) {
	FILE * f=fopen(getNextName(ext).c_str(),"w+");
	if(f==NULL) {
		serr->printf("Error opening file\n");
		sndman->PlayFile(config->controller.error_snd);
		return NULL;
	}
	sndman->PlayFile("camera.wav");
	return f;
}


std::string
CameraBehavior::getNextName(const std::string& ext) {
	char tmp[100];
	snprintf(tmp,100,"/ms/data/img%05d%s",index++,ext.c_str());
	sout->printf("Saving %s...",tmp);
	return tmp;
}

void
CameraBehavior::initIndex() {
	std::string path="/ms/data/";
	DIR* dir=opendir(path.c_str());
	if(dir==NULL) {
		serr->printf("bad path: %s\n",path.c_str());
		return;
	}
	struct dirent * ent=readdir(dir);
	while(ent!=NULL) {
		struct stat s;
		std::string fullpath=path+ent->d_name;
		int err=stat(fullpath.c_str(),&s);
		if(err!=0) {
			serr->printf("File disappeared: %s\n");
			return;
		}
		if((s.st_mode&S_IFDIR)==0 && strncasecmp(ent->d_name,"IMG",3)==0) {
			unsigned int cur=atoi(&ent->d_name[3]);
			if(cur>index)
				index=cur;
		}
		ent=readdir(dir);
	}
	closedir(dir);
	index++; //set index to next unused
	sout->printf("The next saved image will go to %simg%05d\n",path.c_str(),index);
}


/*! @file
 * @brief Implements CameraBehavior, for taking pictures
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_2_1 $
 * $Revision: 1.7 $
 * $State: Exp $
 * $Date: 2004/10/17 01:16:10 $
 */

