#include "PostureEditor.h"
#include "Motion/MMAccessor.h"
#include "Motion/EmergencyStopMC.h"
#include "Motion/MotionSequenceMC.h"
#include "Motion/LedMC.h"
#include "Events/EventRouter.h"
#include "ValueEditControl.h"
#include "NullControl.h"
#include "SoundPlay/SoundManager.h"

typedef MotionSequenceMC<MotionSequence::SizeSmall> reach_t; //!< just to save some typing

PostureEditor::PostureEditor(MotionManager::MC_ID estop_ID)
	: ControlBase("Posture Editor","Allows you to load, save, and numerically edit the posture"), 
	  pose(), reachID(MotionManager::invalid_MC_ID),
	  estopID(estop_ID), lastSlot(NULL), loadPose(NULL), savePose(NULL), pauseCalled(false)
{
	// add load and save menus
	pushSlot(loadPose=new FileInputControl("Load Posture","Select a posture to open",config->motion.root));
	loadPose->setFilter("*.pos");
	pushSlot(savePose=new StringInputControl("Save Posture","Please enter the filename to save to (in "+config->motion.root+")"));

	// add submenu for weight editors
	ControlBase * weights;
	pushSlot(weights=new ControlBase("Weights","Set the weights for outputs"));
	for(unsigned int i=0; i<NumOutputs; i++)
		weights->pushSlot(new ValueEditControl<float>(outputNames[i],&pose(i).weight));

	pushSlot(NULL); // a separator for clarity

	// add actual value editors
	for(unsigned int i=0; i<NumOutputs; i++)
		pushSlot(new ValueEditControl<float>(outputNames[i],&pose(i).value));
}

ControlBase *
PostureEditor::activate(MotionManager::MC_ID disp_id, Socket * gui) {
	//cout << "activate" << endl;
	// start off with current pose
	pose.takeSnapshot();
	// clear the LEDs though
	for(unsigned int i=LEDOffset; i<LEDOffset+NumLEDs; i++)
		pose.setOutputCmd(i,0);
	// add it the motion sequence we'll be using to move to changes
	SharedObject<reach_t> reach;
	reachID=motman->addPersistentMotion(reach);
	// we'll need to know when estop is turned on or off
	erouter->addListener(this,EventBase::estopEGID);
	// call super class
	return ControlBase::activate(disp_id,gui);
}

void
PostureEditor::refresh() {
	//cout << "refresh" << endl;
	if(isEStopped()) {
		erouter->addTimer(this,0,500);
		options[0]=NULL;
	} else {
		options[0]=loadPose;
	}
	if(lastSlot==loadPose) {
		// we just got back from the load menu
		pose.LoadFile(loadPose->getLastInput().c_str());
		updatePose(moveTime);
	} else if(lastSlot==savePose || savePose->getLastInput().size()>0) {
		// we just got back from the save menu
		pose.SaveFile(config->motion.makePath(savePose->getLastInput()).c_str());
		savePose->takeInput("");
	} else {
		updatePose(moveTime/2);
	}
	lastSlot=NULL;
	pauseCalled=false;
	ControlBase::refresh();
}

void
PostureEditor::pause() {
	//cout << "paused" << endl;
	pauseCalled=true;
	erouter->removeListener(this,EventBase::timerEGID);
}

void
PostureEditor::deactivate() {
	//cout << "deactivate" << endl;
	motman->removeMotion(reachID);
	reachID=MotionManager::invalid_MC_ID;
	erouter->removeListener(this);
	ControlBase::deactivate();
}

ControlBase*
PostureEditor::doSelect() {
	// record the option that is being selected, in case it's the load or save
	lastSlot=options[hilights.front()];
	// but do what we'd normally do (select that option)
	return ControlBase::doSelect();
}

void
PostureEditor::processEvent(const EventBase& e) {
	if(e.getGeneratorID()==EventBase::estopEGID) {
		if(e.getTypeID()==EventBase::deactivateETID) {
			MMAccessor<reach_t>(reachID)->play();
			erouter->removeListener(this,EventBase::timerEGID);
			if(!pauseCalled)
				refresh();
		} else {
			if(!pauseCalled) {
				erouter->addTimer(this,0,500);
				processEvent(EventBase(EventBase::timerEGID,0,EventBase::statusETID));
			}
		}
	} else if(e.getGeneratorID()==EventBase::timerEGID) {
		//doing a manual copy instead of just takeSnapshot() so we don't disturb the LED settings
		for(unsigned int i=PIDJointOffset; i<PIDJointOffset+NumPIDJoints; i++)
			pose(i).value=state->outputs[i];
		refresh();
	} else {
		serr->printf("WARNING: PostureEditor unexpected event: %s\n",e.getName().c_str());
	}
}

bool
PostureEditor::isEStopped() {
	return MMAccessor<EmergencyStopMC>(estopID)->getStopped();
}

void
PostureEditor::updatePose(unsigned int delay) {
	bool paused=isEStopped();
	MMAccessor<reach_t> reach_acc(reachID);
	reach_acc->clear();
	reach_acc->setPlayTime(delay);
	reach_acc->setPose(pose);
	reach_acc->setPlayTime(delay+100);
	reach_acc->setPose(pose);
	if(paused)
		reach_acc->pause();
	else
		reach_acc->play();
}


/*! @file
 * @brief Describes PostureEditor, which allows numeric control of joints and LEDs
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_2_1 $
 * $Revision: 1.7 $
 * $State: Exp $
 * $Date: 2004/10/18 19:53:02 $
 */
