/*
 * Measurement update code for the AIBO FastSLAM.
 * See header file for details.
 */

#ifdef __cplusplus
extern "C" {
#endif

#include "afsParticle.h"
#include "afsTriangulator.h"
#include "afsUtility.h"

#include <math.h>
#include <stdio.h>

/* Prototypes */
void doPriming(afsParticle *p, int landmark, double theta);
void doKalmanUpdate(afsParticle *p, int landmark, double theta);

/* The main measurement update routine. Calls one of the other specific
 * update routines and doesn't do much by itself. */
void afsMeasurementUpdate(afsParticle *p, int landmark, double theta)
{
  /* Add theta to the current heading of the robot to put it in
   * world coordinates. */
  theta += p->pose.theta;
  /* Constrain it to [0, 2pi) */
  while(theta < 0) theta += 2*M_PI;
  while(theta >= 2*M_PI) theta -= 2*M_PI;

  if(p->landmarks[landmark].state == PRIMING) doPriming(p, landmark, theta);
  else /* state = PRIMED */ doKalmanUpdate(p, landmark, theta);
}

/* This routine is used to generate an initial fix on the location of the
 * landmark by priming it with a position estimation derived from
 * triangulation. */
void doPriming(afsParticle *p, int landmark, double theta)
{
  /* We do this for brevity's sake. */
  afsLastObservation *l = &(p->landmarks[landmark].priming);

  /* If this is the very first time the landmark has been observed, we
   * simply store information about the observation. */
  if(l->empty) {
    l->x = p->pose.x;
    l->y = p->pose.y;
    l->theta = theta;
    l->empty = 0;
  }
  /* If the landmark has already been observed once before, we try to do
   * the triangulation. */
  else {
    if(afsTriangulator(l->x,l->y,l->theta, p->pose.x,p->pose.y,theta,
		       &(p->landmarks[landmark]))) {
      /* Succesful triangulation--the triangulator has taken care of
       * everything. */
      p->landmarks[landmark].state = PRIMED;
    }
    else {
      /* Priming was not successful--chances are we were in a bad location
       * for triangulation. Just store the last measurement away for now--
       * we'll try again later. */
      l->x = p->pose.x;
      l->y = p->pose.y;
      l->theta = theta;
    }
  }

  /* We didn't make any weighty contribution */
  p->gotweight = 0;
  /* Finished */
}

/* We already have some idea of where the landmark is. Use Kalman filtering
 * to integrate information from this latest sensor reading into a position
 * estimate for the landmark. */
/* Thanks to Mike "Mr. FastSLAM" Montemerlo for help with this. */
void doKalmanUpdate(afsParticle *p, int landmark, double theta)
{
  /* Here are all the matrices 'n stuff we need to do the Kalman filter */
  double H[2], K[2], expected_theta;
  /* Shorthand */
  afsLandmarkLoc *l = &(p->landmarks[landmark]);
  /* This q variable is used to compute the Jacobean, H. It's also used
   * later to alter the measurement covariance based on distance. See the
   * note below. */
  double q;

  /* What we now have is a great deal of 2D matrix math. */
  /* First we compute the Jacobean, H, which is as I understand it a
   * transformation from 2D location space to "measurement space" (?)
   * or at least the kind of numbers that bearings only sensors give us.
   * Actually, it's a linear approximation of such a transformation. */
  {
    double rel_x, rel_y;

    rel_x = l->mean.x - p->pose.x;
    rel_y = l->mean.y - p->pose.y;

    q = rel_x*rel_x + rel_y*rel_y;
    /* Check against divide by 0--hopefully it won't ever happen */
    if(q < 0.0000001) {
      fputs("RARE ERROR: near divide by 0 in doKalmanUpdate\n", stderr);
      p->gotweight = 0;
      return;
    }

    H[0] = -rel_y / q;
    H[1] = rel_x / q;

    /* While we're at it, let's also find the value of theta we expected to
     * see for this landmark. */
    expected_theta = atan2(rel_y, rel_x);
  }

  {
    /* Now we compute K, the Kalman gain. */
    double H_Sigma[2];
    double divisor;
    double dtheta;

    /* This is the measurement covariance we use to compute the Kalman gain.
     * Here, in bearings-only FastSLAM, it's dynamic. We increase the
     * covariance value linearly with the distance of the object. Measurements
     * about distant objects therefore have less effect on where the objects
     * are placed in the particle's map and don't drastically affect the
     * particle's weighting. This is a kludge, but then so is the EKF. */
    /* AFS_MEASURE_VARIANCE is now the measurement variance at a
     * distance of a meter. AFS_VARIANCE_MULTIPLIER is multiplied by the
     * distance and added to AFS_MEASUREMENT_VARIANCE to describe a linear
     * change in covariance. It is a Good Idea to set this value such that
     * covariance never becomes negative! */
    /* If you don't like any of this, set AFS_VARIANCE_MULTIPLIER to 1
     * and don't think about it. */
    double R = AFS_VARIANCE_MULTIPLIER*(sqrt(q) - 1000) + AFS_MEASURE_VARIANCE;
    /* We should never let R get below AFS_MEASURE_VARIANCE, so we correct
     * that here. */
    if(R < AFS_MEASURE_VARIANCE) R = AFS_MEASURE_VARIANCE;

    H_Sigma[0] = H[0] * l->variance.x  + H[1] * l->variance.xy;
    H_Sigma[1] = H[0] * l->variance.xy + H[1] * l->variance.y;

    /* Note measurement variance fudge factor */
    divisor = H_Sigma[0] * H[0] + H_Sigma[1] * H[1] + R;

    K[0] = H_Sigma[0] / divisor;
    K[1] = H_Sigma[1] / divisor;

    /* Hooray--now we can compute the new mean for the landmark */
    dtheta = find_dtheta(expected_theta, theta);
    l->mean.x += K[0] * dtheta;
    l->mean.y += K[1] * dtheta;

    /* Since we have the divisor and expected value already, we can go
     * ahead and compute the weight for this particle. */
    p->gotweight = 1;
    /* 2 * M_PI is removed -- scaling doesn't need it */
    /* Why do we multiply here? If we have multiple measurements before a
     * resampling, we want to be able to use them all to influence the
     * resampling. We multiply together all the weights from the measurements
     * to create a final weight that is used for resampling. DANGER: May
     * fall to zero after too many improbable measurements. 300 weights of
     * 0.05 is zero in double precision. The lesson: for now, move around
     * and resample frequently. */
    p->weight *= (1/sqrt(divisor)) * exp(-0.5*dtheta*dtheta/divisor);
  }

  /* As well as the new covariance! */
  {
    double KH_x, KH_y, KH_tr, KH_bl;
    double KHP_x, KHP_y, KHP_xy;

    KH_x  = K[0]*H[0];
    KH_y  = K[1]*H[1];
    KH_tr = K[0]*H[1];
    KH_bl = K[1]*H[0];

    KHP_x  = KH_x * l->variance.x + KH_tr * l->variance.xy;
    KHP_y  = KH_bl * l->variance.xy + KH_y * l->variance.y;
    KHP_xy = KH_bl * l->variance.x + KH_y * l->variance.xy;

    l->variance.x -= KHP_x;
    l->variance.y -= KHP_y;
    l->variance.xy -= KHP_xy;
  }

  /* Kalman filters are rad, even if I don't understand them very well */
}

#ifdef __cplusplus
}
#endif
