#ifndef _AFS_FIND_BEST_POSE_H_
#define _AFS_FIND_BEST_POSE_H_

/*! @file
 *
 * Finds the proper orientation of the landmark and position information
 * within a particle based on the known locations of landmarks.
 */

#include "afsParticle.h"

#ifdef __cplusplus
extern "C" {
#endif

/*! A structure that contains the relative radial position of a landmark
 * (the robot is at the origin). */
typedef struct _afsRRP {
  double theta; //!< angle
  double distance; //!< distance
} afsRRP;

/*! A structure that simply contains the x,y coordinates of landmarks. */
typedef struct _afsXY {
  double x; //!< distance along x
	double y; //!< distance along y
} afsXY;

/*! This routine determines the best orientation of the particle p given the
 * supplied robot pose and landmark positions. The latter are supplied in an
 * array pointed to by the last argument (note: the landmark ID numbers will
 * be used to index the array. */
afsPose afsFindBestPose(afsParticle *p, afsPose pose, afsXY *real_landmarks);

/*! This routine takes the location and orientation in the particle p and
 * places the same information in new_p after applying the translation and
 * rotation recommended in best_pose (which you can get from 
 * afsFindBestPose. You can make p and new_p point to the same thing
 * safely, but why would you want to? */
void afsApplyBestPose(afsParticle *p, afsParticle *new_p, afsPose best_pose);

/*! This routine finds the error value for a particle given real map
 * information. The function we use to calculate the error here is the
 * same one we minimized in afsFindBestPose, only there we were just
 * concerned with its derivative.
 * NOTE: The more unprimed landmarks a particle has, the lower its error is
 * likely to be. This does not affect the performance of the algorithm */
double afsParticleError(afsParticle *p, afsPose pose, afsXY *real_landmarks);

/*! This routine makes an estimated map of landmarks in the environment and
 * comes up with an estimated robot pose for situations where ground truth
 * is not known. It does all of this by averaging. The first argument is
 * a collection of all the particles in the filter. g in the arguments
 * stands for "guess". */
void afsGuessState(afsParticle *P, afsPose *g_pose, afsXY *g_landmarks);

#ifdef __cplusplus
}
#endif
#endif
