#include "RegionGenerator.h"
#include "Events/EventRouter.h"
#include "Wireless/Wireless.h"
#include "Shared/WorldState.h"

#include "Vision/RLEGenerator.h"

#include "Shared/debuget.h"

RegionGenerator::RegionGenerator(EventBase::EventGeneratorID_t gid, unsigned int sid, unsigned int mysid)
	: FilterBankGenerator("RegionGenerator",gid,sid,EventBase::visRegionEGID,mysid), srcNumColors(0), srcColors(0), regions(NULL)
{ }

void
RegionGenerator::freeCaches() {
	FilterBankGenerator::freeCaches();
	for(unsigned int l=0; l<numLayers; l++)
		for(unsigned int c=0; c<numChannels; c++) {
			delete [] regions[l][c];
			regions[l][c]=NULL;
		}
}

void
RegionGenerator::processEvent(const EventBase& event) {
	FilterBankGenerator::processEvent(event);
	const RLEGenerator * rle=dynamic_cast<const RLEGenerator*>(src);
	if(NULL==rle) {
		serr->printf("RegionGenerator's event %s is not from RLEGenerator\n",event.getName().c_str());
		return;
	}
	const SegmentedColorFilterBankEvent * segev=dynamic_cast<const SegmentedColorFilterBankEvent*>(&event);
	if(NULL==segev) {
		serr->printf("RegionGenerator's event %s is not SegmentedColorFilterBankEvent\n",event.getName().c_str());
		return;
	}
	if(srcNumColors!=segev->getNumColors())
		freeCaches();
	srcNumColors=segev->getNumColors();
	srcColors=segev->getColors();
	erouter->postEvent(new SegmentedColorFilterBankEvent(this,getGeneratorID(),getSourceID(),*segev));
}

unsigned int
RegionGenerator::getBinSize() const {
	unsigned int used=FilterBankGenerator::getBinSize();
	used+=strlen("RegionImage")+LoadSave::stringpad;
	used+=sizeof(unsigned int);
	unsigned int xmit_bytes_per_run=5*sizeof(int)+2*sizeof(float)+2*sizeof(int);
	if(imageValids[selectedSaveLayer][selectedSaveChannel])
		used+=xmit_bytes_per_run;//* ? todo
	else
		used+=xmit_bytes_per_run*MAX_REGIONS;
	return used;
}

unsigned int
RegionGenerator::LoadBuffer(const char buf[], unsigned int len) {
	unsigned int origlen=len;
	unsigned int used;
	std::string tmp;
	if(0==(used=FilterBankGenerator::LoadBuffer(buf,len))) return 0;
	len-=used; buf+=used;
	if(0==(used=decode(tmp,buf,len))) return 0;
	len-=used; buf+=used;
	if(tmp!="RegionImage") {
		serr->printf("Unhandled image type for RegionGenerator: %s",tmp.c_str());
		return 0;
	} else {
		unsigned int tmpNumClr=0;
		if(0==(used=decode(tmpNumClr,buf,len))) return 0;
		len-=used; buf+=used;
		if(tmpNumClr!=srcNumColors)
			freeCaches();
		srcNumColors=tmpNumClr;
		if(images[selectedSaveLayer][selectedSaveChannel]==NULL)
			images[selectedSaveLayer][selectedSaveChannel]=createImageCache(selectedSaveLayer,selectedSaveChannel);
		region_stats * stats=reinterpret_cast<region_stats*>(images[selectedSaveLayer][selectedSaveChannel]);
		if(stats==NULL)
			return 0;
		for(unsigned int i=0; i<srcNumColors; i++) {
			unsigned int tmpNumReg=0;
			if(0==(used=decode(tmpNumReg,buf,len))) return 0;
			len-=used; buf+=used;
			if(MAX_REGIONS<=tmpNumReg)
				return 0;
			for(unsigned int j=0; j<tmpNumReg; j++) {
				region * curreg=&regions[selectedSaveLayer][selectedSaveChannel][j];
				if(0==(used=decode(curreg->color,buf,len))) return 0;
				len-=used; buf+=used;
				if(0==(used=decode(curreg->x1,buf,len))) return 0;
				len-=used; buf+=used;
				if(0==(used=decode(curreg->y1,buf,len))) return 0;
				len-=used; buf+=used;
				if(0==(used=decode(curreg->x2,buf,len))) return 0;
				len-=used; buf+=used;
				if(0==(used=decode(curreg->y2,buf,len))) return 0;
				len-=used; buf+=used;
				if(0==(used=decode(curreg->cen_x,buf,len))) return 0;
				len-=used; buf+=used;
				if(0==(used=decode(curreg->cen_y,buf,len))) return 0;
				len-=used; buf+=used;
				if(0==(used=decode(curreg->area,buf,len))) return 0;
				len-=used; buf+=used;
				if(0==(used=decode(curreg->run_start,buf,len))) return 0;
				len-=used; buf+=used;
				if(j==tmpNumReg-1)
					curreg->next=NULL;
				else
					curreg->next=&regions[selectedSaveLayer][selectedSaveChannel][j+1];
			}
			//we're only going to save the region part (not the color info)
			if(0==(used=decode(stats[i].min_area,buf,len))) return 0;
			len-=used; buf+=used;
			if(0==(used=decode(stats[i].total_area,buf,len))) return 0;
			len-=used; buf+=used;
			if(0==(used=decode(stats[i].merge_threshold,buf,len))) return 0;
			len-=used; buf+=used;
			stats[i].list=regions[selectedSaveLayer][selectedSaveChannel];
			stats[i].num=tmpNumReg;
		}
		imageValids[selectedSaveLayer][selectedSaveChannel]=true;
		return origlen-len;	
	}
}

unsigned int
RegionGenerator::SaveBuffer(char buf[], unsigned int len) const {
	unsigned int origlen=len;
	unsigned int used;
	if(0==(used=FilterBankGenerator::SaveBuffer(buf,len))) return 0;
	len-=used; buf+=used;
	if(0==(used=encode("RegionImage",buf,len))) return 0;
	len-=used; buf+=used;
	
	region_stats * stats=reinterpret_cast<region_stats*>(getImage(selectedSaveLayer,selectedSaveChannel));
	if(stats==NULL)
		return 0;
	if(0==(used=encode(srcNumColors,buf,len))) return 0;
	len-=used; buf+=used;
	for(unsigned int i=0; i<srcNumColors; i++) {
		if(0==(used=encode(stats[i].num,buf,len))) return 0;
		len-=used; buf+=used;
		region * curreg=stats[i].list;
		for(int j=0; j<stats[i].num; j++) {
			if(0==(used=encode(curreg->color,buf,len))) return 0;
			len-=used; buf+=used;
			if(0==(used=encode(curreg->x1,buf,len))) return 0;
			len-=used; buf+=used;
			if(0==(used=encode(curreg->y1,buf,len))) return 0;
			len-=used; buf+=used;
			if(0==(used=encode(curreg->x2,buf,len))) return 0;
			len-=used; buf+=used;
			if(0==(used=encode(curreg->y2,buf,len))) return 0;
			len-=used; buf+=used;
			if(0==(used=encode(curreg->cen_x,buf,len))) return 0;
			len-=used; buf+=used;
			if(0==(used=encode(curreg->cen_y,buf,len))) return 0;
			len-=used; buf+=used;
			if(0==(used=encode(curreg->area,buf,len))) return 0;
			len-=used; buf+=used;
			if(0==(used=encode(curreg->run_start,buf,len))) return 0;
			len-=used; buf+=used;
			curreg=curreg->next;
		}
		//we're only going to save the region part (not the color info)
		if(0==(used=encode(stats[i].min_area,buf,len))) return 0;
		len-=used; buf+=used;
		if(0==(used=encode(stats[i].total_area,buf,len))) return 0;
		len-=used; buf+=used;
		if(0==(used=encode(stats[i].merge_threshold,buf,len))) return 0;
		len-=used; buf+=used;
	}
	return origlen-len;
}

void
RegionGenerator::setNumImages(unsigned int nLayers, unsigned int nChannels) {
	if(nLayers==numLayers && nChannels==numChannels)
		return;
	FilterBankGenerator::setNumImages(nLayers,nChannels);
	regions = new region**[numLayers];
	for(unsigned int i=0; i<numLayers; i++) {
		regions[i] = new region*[numChannels];
		for(unsigned int j=0; j<numChannels; j++)
			regions[i][j]=NULL;
	}
}

unsigned char *
RegionGenerator::createImageCache(unsigned int layer, unsigned int chan) const {
	//this is where we'll store the linked list of regions for this image
	regions[layer][chan]=new region[MAX_REGIONS];

	//this is where we store the head of each list as well as some general stats of each image
	//this is what will be returned as getImage()
	region_stats * stats=new region_stats[srcNumColors];
	
	//initialize the region parameters (held in color definition...)
	memcpy(stats,srcColors,srcNumColors*sizeof(region_stats));
	
	return reinterpret_cast<unsigned char*>(stats);
}

void
RegionGenerator::calcImage(unsigned int layer, unsigned int chan) const {
	PROFSECTION("RegionGenerator::calcImage(...)",state->mainProfile);
	
	//some shorthand to make the rest of the code more readable
	const RLEGenerator& srcRLE=dynamic_cast<const RLEGenerator&>(*src); //source generator
	RLEGenerator::run * rmap=reinterpret_cast<RLEGenerator::run*>(srcRLE.getImage(layer,chan)); //the RLE encoded image from source
	region_stats * stats=reinterpret_cast<region_stats*>(images[layer][chan]); //our top level region stats array (which is what users get from getImage())

	//do the actual calculations
  CMVision::ConnectComponents(rmap,srcRLE.getNumRuns(layer,chan));
  unsigned int numReg = CMVision::ExtractRegions(regions[layer][chan],MAX_REGIONS,rmap,srcRLE.getNumRuns(layer,chan));
  unsigned int maxArea = CMVision::SeparateRegions(stats,srcNumColors,regions[layer][chan],numReg);
  CMVision::SortRegions(stats,srcNumColors,maxArea);
  CMVision::MergeRegions(stats,(int)srcNumColors,rmap); //yes that (int) is necessary or compiler complains of ambiguous function call

	//and set the flag so we don't recompute if getImage() is called again before the next frame
	imageValids[layer][chan]=true;
}

void
RegionGenerator::destruct() {
	FilterBankGenerator::destruct();
	for(unsigned int i=0; i<numLayers; i++)
		delete [] regions[i];
	delete [] regions;
	regions=NULL;
}

/*! @file
 * @brief Implements RegionGenerator, which connects regions of CMVision format runs in RLEGenerator
 * @author alokl (Creator)
 * @author ejt (reorganized)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_1 $
 * $Revision: 1.7 $
 * $State: Exp $
 * $Date: 2004/02/18 21:13:32 $
 */

