#include "JPEGGenerator.h"
#include "InterleavedYUVGenerator.h"
#include "Events/DataEvent.h"
#include "Events/EventRouter.h"
#include "Events/FilterBankEvent.h"
#include "Wireless/Wireless.h"
#include <write_jpeg.h>
#include "Shared/Config.h"
#include "Shared/WorldState.h"

#include <jpeg_mem_dest.h>


#include "Shared/debuget.h"

JPEGGenerator::JPEGGenerator(EventBase::EventGeneratorID_t gid, unsigned int sid, unsigned int mysid)
	: FilterBankGenerator("JPEGGenerator",gid,sid,EventBase::visJPEGEGID,mysid), srcMode(SRC_AUTO), curMode(SRC_AUTO), bytesUsed(NULL), cinfo(), jerr(), quality(-1U)
{
	// We set the err object before we create the compress...  the idea
	// is if the creation fails, we can still get the error as to why it
	// failed.
	cinfo.err = jpeg_std_error(&jerr);
	jpeg_create_compress(&cinfo);
}

JPEGGenerator::JPEGGenerator(EventBase::EventGeneratorID_t gid, unsigned int sid, unsigned int mysid, src_mode_t mode)
	: FilterBankGenerator("JPEGGenerator",gid,sid,EventBase::visJPEGEGID,mysid), srcMode(mode), curMode(SRC_AUTO), bytesUsed(NULL), cinfo(), jerr(), quality(-1U)
{
	// We set the err object before we create the compress...  the idea
	// is if the creation fails, we can still get the error as to why it
	// failed.
	cinfo.err = jpeg_std_error(&jerr);
	jpeg_create_compress(&cinfo);
}

JPEGGenerator::~JPEGGenerator() {
	freeCaches();
	destruct();
	jpeg_destroy_compress(&cinfo);
}


/*! The const casts in this function are regretable but necessary
 *  since the corresponding OPEN-R functions require mutable
 *  arguments, even though they shouldn't be modifying the data
 */
void
JPEGGenerator::processEvent(const EventBase& event) {
	FilterBankGenerator::processEvent(event);
	if(getSourceMode()==SRC_AUTO) { //if not auto, curMode was already set when srcMode was set
		if(dynamic_cast<const InterleavedYUVGenerator*>(src)!=NULL)
			curMode=SRC_COLOR;
		else
			curMode=SRC_GRAYSCALE;
	}
	erouter->postEvent(new FilterBankEvent(this,getGeneratorID(),getSourceID()));
}

unsigned int
JPEGGenerator::getBinSize() const {
	unsigned int used=FilterBankGenerator::getBinSize();
	used+=strlen("JPEGColor")+LoadSave::stringpad;
	if(bytesUsed[selectedSaveLayer][selectedSaveChannel]!=0)
		used+=bytesUsed[selectedSaveLayer][selectedSaveChannel];
	else
		used+=widths[selectedSaveLayer]*heights[selectedSaveLayer]*3+JPEG_HEADER_PAD;
	return used;
}

unsigned int
JPEGGenerator::LoadBuffer(const char buf[], unsigned int len) {
	unsigned int origlen=len;
	unsigned int used;
	std::string tmp;
	if(0==(used=FilterBankGenerator::LoadBuffer(buf,len))) return 0;
	len-=used; buf+=used;
	if(0==(used=decode(tmp,buf,len))) return 0;
	len-=used; buf+=used;
	if(tmp!="JPEGColor" && tmp!="JPEGGray") {
		serr->printf("Unhandled image type for JPEGGenerator: %s",tmp.c_str());
		return 0;
	} else {
		if(tmp=="JPEGColor" && getCurrentSourceFormat()==SRC_GRAYSCALE)
			serr->printf("Warning: loading grayscale into color image");
		if(tmp=="JPEGGrayscale" && getCurrentSourceFormat()==SRC_COLOR)
			serr->printf("Warning: loading color into grayscale image");
		unsigned int tmpL;
		if(0==(used=decode(tmpL,buf,len))) return 0;
		len-=used; buf+=used;
		if(tmpL>len)
			return 0;
		if(images[selectedSaveLayer][selectedSaveChannel]!=NULL)
			delete [] images[selectedSaveLayer][selectedSaveChannel];
		images[selectedSaveLayer][selectedSaveChannel]=createImageCache(selectedSaveLayer,selectedSaveChannel);
		used=bytesUsed[selectedSaveLayer][selectedSaveChannel]=tmpL;
		unsigned char* img=getImage(selectedSaveLayer,selectedSaveChannel);
		if(img==NULL)
			return 0;
		memcpy(img,buf,used);
		len-=used; buf+=used;
		imageValids[selectedSaveLayer][selectedSaveChannel]=true;
		return origlen-len;	
	}
}

unsigned int
JPEGGenerator::SaveBuffer(char buf[], unsigned int len) const {
	unsigned int origlen=len;
	unsigned int used;
	if(0==(used=FilterBankGenerator::SaveBuffer(buf,len))) return 0;
	len-=used; buf+=used;

	char * type;
	if(getCurrentSourceFormat()==SRC_COLOR)
		type="JPEGColor";
	else if(getCurrentSourceFormat()==SRC_GRAYSCALE)
		type="JPEGGrayscale";
	else {
		serr->printf("SaveBuffer failed - unsuitable or unknown mode/generator pair");
		return 0;
	}
	if(0==(used=encode(type,buf,len))) return 0;
	len-=used; buf+=used;
	
	unsigned char* img=getImage(selectedSaveLayer,selectedSaveChannel);
	if(img==NULL)
		return 0;
	if(0==(used=encode(bytesUsed[selectedSaveLayer][selectedSaveChannel],buf,len))) return 0;
	len-=used; buf+=used;
	used=bytesUsed[selectedSaveLayer][selectedSaveChannel];
	if(used>len)
		return 0;
	memcpy(buf,img,used);
	len-=used;
	return origlen-len;
}

void
JPEGGenerator::setNumImages(unsigned int nLayers, unsigned int nChannels) {
	if(nLayers==numLayers && nChannels==numChannels)
		return;
	FilterBankGenerator::setNumImages(nLayers,nChannels);
	for(unsigned int i=0; i<numLayers; i++)
		strides[i]=skips[i]=0;
	bytesUsed=new unsigned int*[numLayers];
	for(unsigned int res=0; res<numLayers; res++) {
		increments[res]=3;
		bytesUsed[res]=new unsigned int[numChannels];
		for(unsigned int i=0; i<numChannels; i++)
			bytesUsed[res][i]=0;
	}
}

unsigned char *
JPEGGenerator::createImageCache(unsigned int layer, unsigned int /*chan*/) const {
	return new unsigned char[widths[layer]*heights[layer]*3+JPEG_HEADER_PAD];
}

/*! This function is taken pretty directly from the write_jpeg_mem()
 *  function from Sony's W3AIBO sample code.
 *
 *  I have adapted it for this object, and added the ability to
 *  process greyscale images as well as color.
 */
void
JPEGGenerator::calcImage(unsigned int layer, unsigned int chan) const {
	PROFSECTION("JPEGGenerator::calcImage(...)",state->mainProfile);

	//pass the destination buffer and buffer size here
	jpeg_mem_dest(&cinfo, images[layer][chan], widths[layer]*heights[layer]*3+JPEG_HEADER_PAD);

	// mode setup
	cinfo.image_width = widths[layer];
	cinfo.image_height = heights[layer];
	if(getCurrentSourceFormat()==SRC_COLOR ) {
		cinfo.input_components = 3;		/* # of color components per pixel */
		cinfo.in_color_space = JCS_YCbCr; /* colorspace of input image */
	} else if(getCurrentSourceFormat()==SRC_GRAYSCALE) {
		cinfo.input_components = 1;		/* # of color components per pixel */
		cinfo.in_color_space = JCS_GRAYSCALE; /* colorspace of input image */
	} else {
		serr->printf("Compression failed - unsuitable or unknown mode/generator pair");
		jpeg_destroy_compress(&cinfo);
		return;
	}

	// parameter setup
	jpeg_set_defaults(&cinfo);
	unsigned int qual=(quality==-1U?config->vision.rawcam_compress_quality:quality);
	jpeg_set_quality(&cinfo, qual, false); /* limit to baseline-JPEG values */
	cinfo.dct_method=config->vision.jpeg_dct_method;
	if(cinfo.in_color_space==JCS_YCbCr) {
		unsigned int ysamp=1;
		unsigned int uvsamp=1;
		const unsigned int maxsamp=2;  // according to jpeg docs, this should be able to go up to 4, but I get error: "Sampling factors too large for interleaved scan"
		if(config->vision.rawcam_y_skip>config->vision.rawcam_uv_skip) {
			uvsamp=config->vision.rawcam_y_skip-config->vision.rawcam_uv_skip+1;
			if(uvsamp>maxsamp)
				uvsamp=maxsamp;
		} else {
			ysamp=config->vision.rawcam_uv_skip-config->vision.rawcam_y_skip+1;
			if(ysamp>maxsamp)
				ysamp=maxsamp;
		}
		cinfo.comp_info[0].h_samp_factor=ysamp;
		cinfo.comp_info[0].v_samp_factor=ysamp;
		cinfo.comp_info[1].h_samp_factor=uvsamp;
		cinfo.comp_info[1].v_samp_factor=uvsamp;
		cinfo.comp_info[2].h_samp_factor=uvsamp;
		cinfo.comp_info[2].v_samp_factor=uvsamp;
	}

	// compression
	jpeg_start_compress(&cinfo, TRUE);
	unsigned int row_stride = src->getStride(layer);	/* JSAMPLEs per row in image_buffer */
	JSAMPROW row_pointer[1] = { const_cast<JSAMPROW>(src->getImage(layer,chan)) };
	while (cinfo.next_scanline < cinfo.image_height) {
		jpeg_write_scanlines(&cinfo, row_pointer, 1);
		row_pointer[0]+=row_stride;
	}
	jpeg_finish_compress(&cinfo);

	// results
	bytesUsed[layer][chan]=jpeg_mem_size(&cinfo);
	imageValids[layer][chan]=true;
}

void
JPEGGenerator::destruct() {
	FilterBankGenerator::destruct();
	for(unsigned int res=0; res<numLayers; res++)
		delete [] bytesUsed[res];
	delete [] bytesUsed;
	bytesUsed=NULL;
}

/*! @file
 * @brief Implements JPEGGenerator, which generates FilterBankEvents containing JPEG compressed images
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_1 $
 * $Revision: 1.8 $
 * $State: Exp $
 * $Date: 2004/02/18 21:13:32 $
 */

