//-*-c++-*-
#ifndef INCLUDED_ERS7Info_h
#define INCLUDED_ERS7Info_h

#include <math.h>
#ifndef PLATFORM_APERIOS
typedef unsigned short word; //!< otherwise defined in Types.h
#else
#include <Types.h>
#endif

#include "CommonInfo.h"
using namespace RobotInfo;

//! Contains information about the ERS-7 Robot, such as number of joints, PID defaults, timing information, etc.
namespace ERS7Info {

	// *******************************
	//       ROBOT CONFIGURATION
	// *******************************


	const unsigned int FrameTime=8;        //!< time between frames in the motion system (milliseconds)
	const unsigned int NumFrames=4;        //!< the number of frames per buffer (don't forget also double buffered)
	const unsigned int SlowFrameTime=8;  //!< time between frames for the ears (ERS-7 doesn't seem to have any "slow" joints; this only applied for the ears on the ERS-210)
	const unsigned int NumSlowFrames=4;    //!< the number of frames per buffer being sent to ears (double buffered as well)
	const unsigned int SoundBufferTime=32; //!< the number of milliseconds per sound buffer... I'm not sure if this can be changed
	
	//!Corresponds to entries in ERS7Info::PrimitiveName, defined at the end of this file, these are the primary grouping
	/*!Right now all binary joints are slow, but perhaps this won't always be the case... hence the IsFast/Slow bitmasks to select which type, in order to be more general */
	//!@name Output Types Information
	const unsigned NumPIDJoints   = 18; //!< The number of joints which use PID motion - everything except ears
	const unsigned NumLEDs        = 27; //!< The number LEDs which can be controlled
	const unsigned NumBinJoints   =  2; //!< The number of binary joints - just the ears
	const unsigned NumOutputs     = NumPIDJoints + NumBinJoints + NumLEDs; //!< the total number of outputs

	//! true for joints which can be updated every 32 ms (all but the ears)
	const bool IsFastOutput[NumOutputs] = {
		true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true, //PID joints
		true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true, //leds
		true,true //ears
	};
	//! we need this so you can tell programmatically which joints are "real" and which are "fake" in a compatability mode
	const bool IsRealERS7[NumOutputs] = { 
		true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true, //PID joints
		true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,false, //leds
		true,true //ears
	};
	//@}

	const unsigned JointsPerLeg   =  3; //!< The number of joints per leg
	const unsigned NumLegs        =  4; //!< The number of legs
	const unsigned NumLegJoints   =  JointsPerLeg*NumLegs; //!< the TOTAL number of joints on ALL legs
	const unsigned NumHeadJoints  =  3; //!< The number of joints in the neck
	const unsigned NumTailJoints  =  2; //!< The number of joints assigned to the tail
	const unsigned NumMouthJoints =  1; //!< the number of joints that control the mouth
	const unsigned NumEarJoints   =  2; //!< The number of joints which control the ears (NOT per ear, is total)
	const unsigned NumButtons     =  2+4+3+1; //!< the number of buttons that are available, 2 head, 4 paws, 3 back, 1 underbelly see ERS7Info::ButtonOffset_t
	const unsigned NumSensors     =  3+3+5;  //!< 3 IR (distance), 3 accel (force), 5 from power, see ERS7Info::SensorOffset_t
	

	// *******************************
	//         OUTPUT OFFSETS
	// *******************************


	//!Corresponds to entries in ERS7Info::PrimitiveName, defined at the end of this file
	//!@name Output Offsets

	const unsigned PIDJointOffset = 0; //!< The beginning of the PID Joints
	const unsigned LegOffset   = PIDJointOffset;           //!< the offset of the beginning of the leg joints
	const unsigned HeadOffset  = LegOffset+NumLegJoints;   //!< the offset of the beginning of the head joints
	const unsigned TailOffset  = HeadOffset+NumHeadJoints; //!< the offset of the beginning of the tail joints
	const unsigned MouthOffset = TailOffset+NumTailJoints; //!< the offset of the beginning of the mouth joint

	const unsigned LEDOffset   = PIDJointOffset + NumPIDJoints; //!< the offset of LEDs in WorldState::outputs and MotionCommand functions

	const unsigned BinJointOffset = LEDOffset + NumLEDs; //!< The beginning of the binary joints
	const unsigned EarOffset   = BinJointOffset;           //!< the offset of the beginning of the ear joints - note that ears aren't sensed.  They can be flicked by the environment and you won't know.  Nor will they be flicked back

	//! The offsets of the individual legs
	enum LegOffset_t {
		LFrLegOffset = LegOffset+LFrLegOrder*JointsPerLeg, //!< beginning of left front leg
		RFrLegOffset = LegOffset+RFrLegOrder*JointsPerLeg, //!< beginning of right front leg
		LBkLegOffset = LegOffset+LBkLegOrder*JointsPerLeg, //!< beginning of left back leg
		RBkLegOffset = LegOffset+RBkLegOrder*JointsPerLeg  //!< beginning of right back leg
	};
	
	//@}
	
	//! The offsets of the individual LEDs on the head and tail.  Note that left/right are robot's point of view.  See also LEDBitMask_t
	enum LEDOffset_t {
		HeadColorLEDOffset = LEDOffset,
		HeadWhiteLEDOffset,
		ModeRedLEDOffset,
		ModeGreenLEDOffset,
		ModeBlueLEDOffset,
		WirelessLEDOffset,
		FaceLEDPanelOffset, //!< the first LED in the panel - add to this to get the other 13 (14 total)
		FrBackColorLEDOffset = FaceLEDPanelOffset+14, // blue/purple
		FrBackWhiteLEDOffset,
		MdBackColorLEDOffset,                    // orange
		MdBackWhiteLEDOffset,
		RrBackColorLEDOffset,                    // red
		RrBackWhiteLEDOffset,
		LEDABModeOffset, // allows you to control A/B mode setting (this is a "virtual" LED)

		// aliases for 2xx cross-compatibility
		BotLLEDOffset = FaceLEDPanelOffset+0,//!< aliases for backward compatability (use mode A); bottom left
		BotRLEDOffset = FaceLEDPanelOffset+1, //!< bottom right
		MidLLEDOffset = FaceLEDPanelOffset+2, //!< middle left
		MidRLEDOffset = FaceLEDPanelOffset+3, //!< middle right
		TopLLEDOffset = FaceLEDPanelOffset+6, //!< top left
		TopRLEDOffset = FaceLEDPanelOffset+7, //!< top right
		TopBrLEDOffset= HeadColorLEDOffset,//!< top bar
		TlRedLEDOffset= RrBackColorLEDOffset,//!< red tail light
		TlBluLEDOffset= FrBackColorLEDOffset //!< blue tail light
	};
	
	//! Bitmasks for use when specifying combinations of LEDs (see LedEngine ) Note that left/right are robot's point of view
	//!@name LED Bitmasks
	typedef unsigned int LEDBitMask_t; //!< So you can be clear when you're refering to a LED bitmask
	
	const LEDBitMask_t HeadColorLEDMask = 1<<(HeadColorLEDOffset-LEDOffset); //!< mask corresponding to HeadColorLEDOffset
	const LEDBitMask_t HeadWhiteLEDMask = 1<<(HeadWhiteLEDOffset-LEDOffset); //!< mask corresponding to HeadWhiteLEDOffset
	const LEDBitMask_t ModeRedLEDMask = 1<<(ModeRedLEDOffset-LEDOffset); //!< mask corresponding to ModeRedLEDOffset
	const LEDBitMask_t ModeGreenLEDMask = 1<<(ModeGreenLEDOffset-LEDOffset); //!< mask corresponding to ModeGreenLEDOffset
	const LEDBitMask_t ModeBlueLEDMask = 1<<(ModeBlueLEDOffset-LEDOffset); //!< mask corresponding to ModeBlueLEDOffset
	const LEDBitMask_t WirelessLEDMask = 1<<(WirelessLEDOffset-LEDOffset); //!< mask corresponding to WirelessLEDOffset
	const LEDBitMask_t FaceLEDPanelMask = 1<<(FaceLEDPanelOffset-LEDOffset); //!< mask corresponding to FaceLEDPanelOffset, selects only the first of the panel - shift this to get the others
	const LEDBitMask_t FrBackColorLEDMask = 1<<(FrBackColorLEDOffset-LEDOffset); //!< mask corresponding to FrBackColorLEDOffset
	const LEDBitMask_t FrBackWhiteLEDMask = 1<<(FrBackWhiteLEDOffset-LEDOffset); //!< mask corresponding to FrBackWhiteLEDOffset
	const LEDBitMask_t MdBackColorLEDMask = 1<<(MdBackColorLEDOffset-LEDOffset); //!< mask corresponding to MdBackColorLEDOffset
	const LEDBitMask_t MdBackWhiteLEDMask = 1<<(MdBackWhiteLEDOffset-LEDOffset); //!< mask corresponding to MdBackWhiteLEDOffset
	const LEDBitMask_t RrBackColorLEDMask = 1<<(RrBackColorLEDOffset-LEDOffset); //!< mask corresponding to RrBackColorLEDOffset
	const LEDBitMask_t RrBackWhiteLEDMask = 1<<(RrBackWhiteLEDOffset-LEDOffset); //!< mask corresponding to RrBackWhiteLEDOffset
	const LEDBitMask_t LEDABModeMask = 1<<(LEDABModeOffset-LEDOffset); //!< mask corresponding to LEDABModeOffset

	const LEDBitMask_t BotLLEDMask = 1<<(BotLLEDOffset-LEDOffset); //!< bottom left
	const LEDBitMask_t BotRLEDMask = 1<<(BotRLEDOffset-LEDOffset); //!< bottom right
	const LEDBitMask_t MidLLEDMask = 1<<(MidLLEDOffset-LEDOffset); //!< middle left
	const LEDBitMask_t MidRLEDMask = 1<<(MidRLEDOffset-LEDOffset); //!< middle right
	const LEDBitMask_t TopLLEDMask = 1<<(TopLLEDOffset-LEDOffset); //!< top left
	const LEDBitMask_t TopRLEDMask = 1<<(TopRLEDOffset-LEDOffset); //!< top right
	const LEDBitMask_t TopBrLEDMask= 1<<(TopBrLEDOffset-LEDOffset); //!< top bar
	const LEDBitMask_t TlRedLEDMask= 1<<(TlRedLEDOffset-LEDOffset); //!< red tail light
	const LEDBitMask_t TlBluLEDMask= 1<<(TlBluLEDOffset-LEDOffset); //!< blue tail light

	const LEDBitMask_t FaceLEDMask = (FaceLEDPanelMask<<0)|(FaceLEDPanelMask<<1)|(FaceLEDPanelMask<<2)|(FaceLEDPanelMask<<3)|(FaceLEDPanelMask<<4)|(FaceLEDPanelMask<<5)|(FaceLEDPanelMask<<6)|(FaceLEDPanelMask<<7)|(FaceLEDPanelMask<<8)|(FaceLEDPanelMask<<9)|(FaceLEDPanelMask<<10)|(FaceLEDPanelMask<<11)|(FaceLEDPanelMask<<12)|(FaceLEDPanelMask<<13); //!< LEDs for the face panel
	const LEDBitMask_t HeadLEDMask = FaceLEDMask|HeadColorLEDMask|HeadWhiteLEDMask|ModeRedLEDMask|ModeGreenLEDMask|ModeBlueLEDMask|WirelessLEDMask; //!< LEDs for face (all but back lights)
	const LEDBitMask_t BackLEDMask = FrBackColorLEDMask|FrBackWhiteLEDMask|MdBackColorLEDMask|MdBackWhiteLEDMask|RrBackColorLEDMask|RrBackWhiteLEDMask; //!< LEDS on the back
	const LEDBitMask_t TailLEDMask = 0; //!< LEDs on tail (ERS-7 has none)
	const LEDBitMask_t AllLEDMask  = ~0; //!< selects all of the leds
	//@}


	// *******************************
	//          INPUT OFFSETS
	// *******************************


	//! The order in which inputs should be stored
	//!@name Input Offsets

	//! holds offsets to different buttons in WorldState::buttons[]
	/*! Should be a straight mapping to the ButtonSourceIDs
	 *
	 *  Note that the chest (power) button is not a normal button.  It kills
	 *  power to the motors at a hardware level, and isn't sensed in the
	 *  normal way.  If you want to know when it is pressed (and you are
	 *  about to shut down) see PowerSourceID::PauseSID.
	 *
	 *  @see WorldState::buttons @see ButtonSourceID_t */
	enum ButtonOffset_t {
		LFrPawOffset = LFrLegOrder,
		RFrPawOffset = RFrLegOrder,
		LBkPawOffset = LBkLegOrder,
		RBkPawOffset = RBkLegOrder,
		ChinButOffset= 4,
		HeadButOffset,
		FrontBackButOffset,
		MiddleBackButOffset,
		RearBackButOffset,
		WirelessSwOffset
	};

	//! holds offset to different sensor values in WorldState::sensors[]
	/*! @see WorldState::sensors[] */
	enum SensorOffset_t {
		NearIRDistOffset = 0,  //!< in millimeters
		FarIRDistOffset,  //!< in millimeters
		ChestIRDistOffset,  //!< in millimeters
		BAccelOffset, //!< backward acceleration, in @f$m/s^2@f$, negative if sitting on butt (positive for faceplant)
		LAccelOffset, //!< acceleration to the robot's left, in @f$m/s^2@f$, negative if lying on robot's left side
		DAccelOffset, //!< downward acceleration, in @f$m/s^2@f$, negative if standing up... be careful about the signs on all of these...
		PowerRemainOffset, //!< percentage, 0-1
		PowerThermoOffset, //!<  degrees Celcius
		PowerCapacityOffset, //!< milli-amp hours
		PowerVoltageOffset, //!< volts
		PowerCurrentOffset //!< milli-amp negative values (maybe positive while charging?)
	};

	//@}


	//! The length of the strings used for each of the outputs in outputNames (doesn't include null term)
	const unsigned outputNameLen = 9;
	//! A name of uniform length for referring to joints - handy for posture files, etc.
	const char* const outputNames[NumOutputs] = {
		"LFr:rotor",
		"LFr:elvtr",
		"LFr:knee~",
		"RFr:rotor",
		"RFr:elvtr",
		"RFr:knee~",
		"LBk:rotor",
		"LBk:elvtr",
		"LBk:knee~",
		"RBk:rotor",
		"RBk:elvtr",
		"RBk:knee~",
		
		"NECK:tilt",
		"NECK:pan~",
		"NECK:nod~",
		
		"TAIL:tilt",
		"TAIL:pan~",
		
		"MOUTH~~~~",
		
		"LED:headC",
		"LED:headW",
		"LED:modeR",
		"LED:modeG",
		"LED:modeB",
		"LED:wless",
		"LED:faceA",
		"LED:faceB",
		"LED:faceC",
		"LED:faceD",
		"LED:faceE",
		"LED:faceF",
		"LED:faceG",
		"LED:faceH",
		"LED:faceI",
		"LED:faceJ",
		"LED:faceK",
		"LED:faceL",
		"LED:faceM",
		"LED:faceN",
		"LED:bkFrC",
		"LED:bkFrW",
		"LED:bkMdC",
		"LED:bkMdW",
		"LED:bkRrC",
		"LED:bkRrW",
		"LED:ABmod",
		
		"EAR:left~",
		"EAR:right"
	};
	

	//! the joint identifier strings used to refer to specific joints in OPEN-R (but not needed for others)
	/*!@showinitializer 
	 * @warning IMPORTANT!!!!  DO NOT CHANGE THE ORDER OF ITEMS IN THIS TABLE!!!\n
	 *
	 * The offset consts defined in this file correspond to this table and will make life easier
	 * if you feel the need to reorder things, but they aren't used perfect @e everywhere \n
	 * In particular, assumptions are made that the pid joints will be in slots 0-numPIDJoints
	 * and that the fast outputs (ie NOT ears) will be in slots 0-NumFastOutputs\n
	 * There may be other assumptions not noted here!!!
	 * @note These entries DON'T correspond to the CPC index numbers defined in WorldState (this only lists joints, and in a different order defined by OPEN-R, that one has sensors as well*/
	const char* const PrimitiveName [NumOutputs] = {
		"PRM:/r2/c1-Joint2:21",       //!< the left front leg   the rotator
		"PRM:/r2/c1/c2-Joint2:22",    //!< the left front leg   the elevator 
		"PRM:/r2/c1/c2/c3-Joint2:23", //!< the left front leg   the knee 
		"PRM:/r4/c1-Joint2:41",       //!< the right front leg   the rotator
		"PRM:/r4/c1/c2-Joint2:42",    //!< the right front leg    the elevator 
		"PRM:/r4/c1/c2/c3-Joint2:43", //!< the right front leg   the knee 
				
		"PRM:/r3/c1-Joint2:31",       //!< the left hind leg   the rotator
		"PRM:/r3/c1/c2-Joint2:32",    //!< the left hind leg   the elevator 
		"PRM:/r3/c1/c2/c3-Joint2:33", //!< the left hind leg   the knee
		"PRM:/r5/c1-Joint2:51",       //!< the right hind leg   the rotator
		"PRM:/r5/c1/c2-Joint2:52",    //!< the right hind leg   the elevator 
		"PRM:/r5/c1/c2/c3-Joint2:53", //!< the right hind leg   the knee 
		  
		"PRM:/r1/c1-Joint2:11",       //!< the lower neck tilt (12)
		"PRM:/r1/c1/c2-Joint2:12",    //!< the neck pan 
		"PRM:/r1/c1/c2/c3-Joint2:13", //!< the upper neck tilt (nod)
				
		"PRM:/r6/c1-Joint2:61",       //!< the tail tilt
		"PRM:/r6/c2-Joint2:62",       //!< the tail rotate
				
		"PRM:/r1/c1/c2/c3/c4-Joint2:14", //!< the mouth (17)
				
		"PRM:/r1/c1/c2/c3/l1-LED2:l1", //!< Head light (color) (x6, 18)
		"PRM:/r1/c1/c2/c3/l2-LED2:l2", //!< Head light (white)
		"PRM:/r1/c1/c2/c3/l3-LED2:l3", //!< Red mode indicator
		"PRM:/r1/c1/c2/c3/l4-LED2:l4", //!< Green mode indicator
		"PRM:/r1/c1/c2/c3/l5-LED2:l5", //!< Blue mode indicator
		"PRM:/r1/c1/c2/c3/l6-LED2:l6", //!< wireless light
		
		"PRM:/r1/c1/c2/c3/la-LED3:la", //!< face lights... (x14, 24-37)
		"PRM:/r1/c1/c2/c3/lb-LED3:lb", 		
		"PRM:/r1/c1/c2/c3/lc-LED3:lc", 
		"PRM:/r1/c1/c2/c3/ld-LED3:ld", 
		"PRM:/r1/c1/c2/c3/le-LED3:le", 		
		"PRM:/r1/c1/c2/c3/lf-LED3:lf", 
		"PRM:/r1/c1/c2/c3/lg-LED3:lg", 
		"PRM:/r1/c1/c2/c3/lh-LED3:lh", 
		"PRM:/r1/c1/c2/c3/li-LED3:li", 
		"PRM:/r1/c1/c2/c3/lj-LED3:lj", 
		"PRM:/r1/c1/c2/c3/lk-LED3:lk", 
		"PRM:/r1/c1/c2/c3/ll-LED3:ll", 
		"PRM:/r1/c1/c2/c3/lm-LED3:lm", 
		"PRM:/r1/c1/c2/c3/ln-LED3:ln", //!< ...last face light (37)

		"PRM:/lu-LED3:lu", //!< front back light (color) (x6, 38)
		"PRM:/lv-LED3:lv", //!< front back light (white)
		"PRM:/lw-LED3:lw", //!< middle back light (color)
		"PRM:/lx-LED3:lx", //!< middle back light (white)
		"PRM:/ly-LED3:ly", //!< rear back light (color)
		"PRM:/lz-LED3:lz", //!< rear back light (white)
		"",  //!< the virtual mode A/B switcher
		  
		"PRM:/r1/c1/c2/c3/e5-Joint4:15", //!< left ear (44)
		"PRM:/r1/c1/c2/c3/e6-Joint4:16" //!< right ear
	};

	//! use to open speaker connection with the system
	const char* const SpeakerLocator="PRM:/s1-Speaker:S1";

	//! use to open camera connection with the system
	const char* const CameraLocator="PRM:/r1/c1/c2/c3/i1-FbkImageSensor:F1";

	//! This table holds the default PID values for each joint.  see PIDMC
	const float DefaultPIDs[NumPIDJoints][3] =
		{
			{ 0x1C/(double)(1<<(16-0xE)), 0x08/(double)(1<<(16-0x2)), 0x01/(double)(1<<(16-0xF)) },
			{ 0x14/(double)(1<<(16-0xE)), 0x04/(double)(1<<(16-0x2)), 0x01/(double)(1<<(16-0xF)) },
			{ 0x1C/(double)(1<<(16-0xE)), 0x08/(double)(1<<(16-0x2)), 0x01/(double)(1<<(16-0xF)) },
			{ 0x1C/(double)(1<<(16-0xE)), 0x08/(double)(1<<(16-0x2)), 0x01/(double)(1<<(16-0xF)) },
			{ 0x14/(double)(1<<(16-0xE)), 0x04/(double)(1<<(16-0x2)), 0x01/(double)(1<<(16-0xF)) },
			{ 0x1C/(double)(1<<(16-0xE)), 0x08/(double)(1<<(16-0x2)), 0x01/(double)(1<<(16-0xF)) },
			{ 0x1C/(double)(1<<(16-0xE)), 0x08/(double)(1<<(16-0x2)), 0x01/(double)(1<<(16-0xF)) },
			{ 0x14/(double)(1<<(16-0xE)), 0x04/(double)(1<<(16-0x2)), 0x01/(double)(1<<(16-0xF)) },
			{ 0x1C/(double)(1<<(16-0xE)), 0x08/(double)(1<<(16-0x2)), 0x01/(double)(1<<(16-0xF)) },
			{ 0x1C/(double)(1<<(16-0xE)), 0x08/(double)(1<<(16-0x2)), 0x01/(double)(1<<(16-0xF)) },
			{ 0x14/(double)(1<<(16-0xE)), 0x04/(double)(1<<(16-0x2)), 0x01/(double)(1<<(16-0xF)) },
			{ 0x1C/(double)(1<<(16-0xE)), 0x08/(double)(1<<(16-0x2)), 0x01/(double)(1<<(16-0xF)) },

			{ 0x0A/(double)(1<<(16-0xE)), 0x04/(double)(1<<(16-0x2)), 0x02/(double)(1<<(16-0xF)) },
			{ 0x08/(double)(1<<(16-0xE)), 0x02/(double)(1<<(16-0x2)), 0x04/(double)(1<<(16-0xF)) },
			{ 0x08/(double)(1<<(16-0xE)), 0x04/(double)(1<<(16-0x2)), 0x02/(double)(1<<(16-0xF)) },

			{ 0x0A/(double)(1<<(16-0xE)), 0x04/(double)(1<<(16-0x2)), 0x04/(double)(1<<(16-0xF)) },
			{ 0x0A/(double)(1<<(16-0xE)), 0x04/(double)(1<<(16-0x2)), 0x04/(double)(1<<(16-0xF)) },

			{ 0x08/(double)(1<<(16-0xE)), 0x00/(double)(1<<(16-0x2)), 0x04/(double)(1<<(16-0xF)) }
		};
		
	//! These will control the shift values given to the system.  see PIDMC
	/*! These are modified from the default values to give better range of values to the gains */
	const unsigned char DefaultPIDShifts[3] = {0x0E, 0x02-1, 0x0F-3};
		
	//!These values are Sony's recommended maximum joint velocities, in rad/ms
	/*! a value <= 0 means infinite speed (e.g. LEDs)
	 *  
	 *  These limits are <b>not</b> enforced by the framework.  They are simply available for you to use as you see fit.
	 *  HeadPointerMC is (as of v1.6) the only included MotionCommand to actually use these values.
	 *  
	 *  This is disabled for the ERS-7 until Sony gives us values to use */
	const float MaxOutputSpeed[NumOutputs] = {
		0,     //Legs LR,FB,REK
		0,
		0,
		0,
		0,
		0,
		0,
		0,
		0,
		0,
		0,
		0,
	
		0,     //Head TPR
		0,
		0,
	
		0,     //Tail
		0,
	
		0,     //Mouth
		
		0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, //LEDs
		
		0,0                //Ears
	};

	#ifndef RAD
		//!Just a little macro for converting degrees to radians
	#define RAD(deg) (((deg) * M_PI ) / 180.0)
		//!a flag so we undef these after we're done - do you have a cleaner solution?
	#define __RI_RAD_FLAG
	#endif
	
	//! Defines the indexes to use to access the min and max entries of ERS7Info::outputRanges and ERS7Info::mechanicalLimits
	enum MinMaxRange_t { MinRange,MaxRange };

	//! This table holds the software limits of each of the outputs
	const double outputRanges[NumOutputs][2] =
		{
			{ RAD(-115),RAD(130) },{ RAD(-10),RAD(88) },{ RAD(-25),RAD(122) }, //left front REK
			{ RAD(-115),RAD(130) },{ RAD(-10),RAD(88) },{ RAD(-25),RAD(122) }, //right front REK
			{ RAD(-115),RAD(130) },{ RAD(-10),RAD(88) },{ RAD(-25),RAD(122) }, //left back REK
			{ RAD(-115),RAD(130) },{ RAD(-10),RAD(88) },{ RAD(-25),RAD(122) }, //right back REK

			{ RAD(-75),RAD(0) },{ RAD(-88),RAD(88) },{ RAD(-15),RAD(45) }, //neck Tilt-pan-nod
				
			{ RAD(5),RAD(60) },{ RAD(-45),RAD(45) }, // tail tp

			{ RAD(-55),RAD(-3) }, //mouth

			{0,1},{0,1},{0,1},{0,1},{0,1},{0,1}, //misc LEDs
			{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1}, //Face LEDs
			{0,1},{0,1},{0,1},{0,1},{0,1},{0,1}, //back LEDs
			{0,1}, //virtual mode A/B switcher
		
			{0,1},{0,1} //ears
		};

	//! This table holds the mechanical limits of each of the outputs @todo same as outputLimits right now, don't know actual values yet
	const double mechanicalLimits[NumOutputs][2] =
		{
			{ RAD(-115),RAD(130) },{ RAD(-10),RAD(88) },{ RAD(-25),RAD(122) }, //left front REK
			{ RAD(-115),RAD(130) },{ RAD(-10),RAD(88) },{ RAD(-25),RAD(122) }, //right front REK
			{ RAD(-115),RAD(130) },{ RAD(-10),RAD(88) },{ RAD(-25),RAD(122) }, //left back REK
			{ RAD(-115),RAD(130) },{ RAD(-10),RAD(88) },{ RAD(-25),RAD(122) }, //right back REK

			{ RAD(-75),RAD(0) },{ RAD(-88),RAD(88) },{ RAD(-15),RAD(45) }, //neck Tilt-pan-nod
				
			{ RAD(5),RAD(60) },{ RAD(-45),RAD(45) }, // tail tp

			{ RAD(-55),RAD(-3) }, //mouth

			{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1}, //LEDs
		
			{0,1},{0,1} //ears
		};

#ifdef __RI_RAD_FLAG
#undef RAD
#undef __RI_RAD_FLAG
#endif


	/*! @name CPC IDs
	 * values defined by OPEN-R, used to interface with lower level OPEN-R code to read sensors - DOESN'T correspond to ERS7Info::PrimitiveName */
	static const int CPCJointMouth      =  0; //!< Mouth                           
	static const int CPCSwitchChin      =  1; //!< Chin sensor                     
	static const int CPCJointNeckNod    =  2; //!< Neck tilt2                      
	static const int CPCSensorHead      =  3; //!< Head sensor                     
	static const int CPCSensorNearPSD   =  4; //!< Head distance sensor(near)      
	static const int CPCSensorFarPSD    =  5; //!< Head distance sensor(far)       
	static const int CPCJointNeckPan    =  6; //!< Neck pan                        
	static const int CPCJointNeckTilt   =  7; //!< Neck tilt1                      
	static const int CPCSwitchLFPaw     =  8; //!< Left fore leg  paw sensor       
	static const int CPCJointLFKnee     =  9; //!< Left fore legJ3                 
	static const int CPCJointLFElevator = 10; //!< Left fore legJ2                 
	static const int CPCJointLFRotator  = 11; //!< Left fore legJ1                 
	static const int CPCSwitchLHPaw     = 12; //!< Left hind leg  paw sensor       
	static const int CPCJointLHKnee     = 13; //!< Left hind legJ3                 
	static const int CPCJointLHElevator = 14; //!< Left hind legJ2                 
	static const int CPCJointLHRotator  = 15; //!< Left hind legJ1                 
	static const int CPCSwitchRFPaw     = 16; //!< Right fore leg  paw sensor      
	static const int CPCJointRFKnee     = 17; //!< Right fore legJ3                
	static const int CPCJointRFElevator = 18; //!< Right fore legJ2                
	static const int CPCJointRFRotator  = 19; //!< Right fore legJ1                
	static const int CPCSwitchRHPaw     = 20; //!< Right hind leg  paw sensor      
	static const int CPCJointRHKnee     = 21; //!< Right hind legJ3                
	static const int CPCJointRHElevator = 22; //!< Right hind legJ2                
	static const int CPCJointRHRotator  = 23; //!< Right hind legJ1                
	static const int CPCJointTailTilt   = 24; //!< Tail tilt                       
	static const int CPCJointTailPan    = 25; //!< Tail pan                        
	static const int CPCSensorAccelFB   = 26; //!< Acceleration sensor(front-back) 
	static const int CPCSensorAccelLR   = 27; //!< Acceleration sensor(right-left) 
	static const int CPCSensorAccelUD   = 28; //!< Acceleration sensor(up-down)    
	static const int CPCSensorChestPSD  = 29; //!< Chest distance sensor           
	static const int CPCSwitchWireless  = 30; //!< Wireless LAN switch             
	static const int CPCSensorBackRear  = 31; //!< Back sensor(rear)               
	static const int CPCSensorBackMiddle= 32; //!< Back sensor(middle)             
	static const int CPCSensorBackFront = 33; //!< Back sensor(front)              
	//@}

}
	
/*! @file
 * @brief Defines RobotInfo namespace for ERS-210 models, gives some information about the robot's capabilities, such as joint counts, offsets, names and PID values
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_1 $
 * $Revision: 1.9 $
 * $State: Exp $
 * $Date: 2004/02/09 22:45:28 $
 */

#endif
