//-*-c++-*-
#ifndef INCLUDED_Config_h
#define INCLUDED_Config_h

#include <vector>
#include <string>
#include "RobotInfo.h"
#include <jpeglib.h>

//!provides global access to system configuration information
class Config {
 public:
	//!constructor
	Config(const char* filename)
		: wireless(), vision(), main(), behaviors(), controller(), motion(),
		worldmodel2(), sound()
		{ readConfig(filename); }
	//!destructor
	~Config() {}

	//!section IDs
	enum section_t {
		sec_wireless=0,  //!< denotes wireless section of config file
		sec_vision,      //!< denotes vision section of config file
		sec_main,        //!< denotes main section of config file, for misc. settings
		sec_behaviors,   //!< denotes behaviors section of config file
		sec_controller,  //!< denotes controller section of config file
		sec_motion,      //!< denotes motion section of config file
		sec_worldmodel2, //!< denotes worldmodel section of config file
		sec_sound,       //!< denotes sound section of config file
		sec_invalid      //!< denotes an invalid section of config file
	};

	//!wirless information
	struct wireless_config {
		int id; //!< id number (in case you have more than one AIBO)
      
		wireless_config () : id(1) {} //!< constructor
	} wireless;

	//!vision information
	struct vision_config {
		int white_balance;    //!< white balance
		int gain;             //!< gain
		int shutter_speed;    //!< shutter speed
		int resolution;       //!< resolution
		std::vector<std::string> thresh;      //!< thresholds
		char colors[50];      //!< colors
		int rawcam_port;      //!< port to send raw frames on
		int rle_port;         //!< port to send RLE frames on
		int obj_port;         //!< port to send object info on
		bool restore_image;   //!< if true, replaces pixels holding image info with actual image pixels (as much as possible anyway)
		J_DCT_METHOD jpeg_dct_method;  //!< pick between dct methods for jpeg compression

		//! type of information to send, stored in Config::vision_config::rawcam_encoding
		enum encoding_t {
			ENCODE_COLOR, //!< send Y, U, and V channels
			ENCODE_SINGLE_CHANNEL, //!< send only a single channel (which channel to send is stored in Config::vision_config::rawcam_channel) This is also used for all seg cam images
		};
		encoding_t rawcam_encoding; //!< holds whether to send color or single channel
		int rawcam_channel;    //!< RawCameraGenerator::channel_id_t, if raw_encoding is single channel, this holds the channel to send (computed from rawcam_encoding, not set directly)
		
		//! compression format to use, stored in Config::vision_config::rawcam_compression
		enum compression_t {
			COMPRESS_NONE, //!< no compression (other than subsampling)
			COMPRESS_JPEG, //!< JPEG compression
			COMPRESS_RLE   //!< RLE compression
		};
    compression_t rawcam_compression;//!< holds whether to send jpeg compression

    int rawcam_compress_quality;//!< 0-100, compression quality (currently only used by jpeg)
    int rawcam_y_skip;     //!< resolution level to transmit y channel at
    int rawcam_uv_skip;    //!< resolution level to transmit uv channel at (ignored for jpeg compression)
    int rlecam_skip;       //!< resolution level to transmit segmented images at
    int rlecam_channel;    //!< channel of RLEGenerator to send
		compression_t rlecam_compression; //!< what compression to use on the segmented image
      
		//!constructor
		vision_config() : white_balance(3), gain(2), shutter_speed(2), resolution(2), thresh(), colors(), rawcam_port(0), rle_port(0), obj_port(0), restore_image(true), jpeg_dct_method(JDCT_IFAST), rawcam_encoding(ENCODE_COLOR), rawcam_channel(0), rawcam_compression(COMPRESS_NONE), rawcam_compress_quality(75), rawcam_y_skip(0), rawcam_uv_skip(0), rlecam_skip(1), rlecam_channel(0), rlecam_compression(COMPRESS_RLE) {}
	} vision;
	
	//!core functionality information
	struct main_config {
		int console_port;  //!< port to send/receive "console" information on (separate from system console)
		int stderr_port;   //!< port to send error information to
		int error_level;   //!< controls amount of info to error port
		int debug_level;   //!< controls amount of debug info
		int verbose_level; //!< controls verbosity of info
		int wsjoints_port; //!< port to send joint positions on
		int wspids_port;   //!< port to send pid info on
		int headControl_port;	   //!< port for receiving head commands
		int walkControl_port;	   //!< port for receiving walk commands
		int estopControl_port;	   //!< port for receiving walk commands
    int aibo3d_port;   //!< port for send/receive of joint positions from Aibo 3D GUI
    int wmmonitor_port; //!< port for monitoring Watchable Memory
		bool use_VT100;    //!< if true, enables VT100 console codes (currently only in Controller menus - 1.3)

		//!constructor
		main_config()
			: console_port(0), stderr_port(0), error_level(0), debug_level(0),
				verbose_level(0),wsjoints_port(0),wspids_port(0),headControl_port(0),
				walkControl_port(0),estopControl_port(0),aibo3d_port(0),
        wmmonitor_port(0), use_VT100(true)
		{ }
	} main;

	//!placeholder
	struct behaviors_config {
	} behaviors;
    
	//!controller information
	struct controller_config {
		int gui_port;        //!< port to listen for the GUI to connect to aibo on
		char select_snd[50]; //!< sound file to use for "select" action
		char next_snd[50];   //!< sound file to use for "next" action
		char prev_snd[50];   //!< sound file to use for "prev" action
		char read_snd[50];   //!< sound file to use for "read from std-in" action
		char cancel_snd[50]; //!< sound file to use for "cancel" action
		char error_snd[50]; //!< sound file to use to signal errors

		//!constructor
		controller_config() : gui_port(0) {
			select_snd[0]=next_snd[0]=prev_snd[0]=read_snd[0]=cancel_snd[0]=error_snd[0]='\0';
		}
	} controller;
    
	//!motion information
	struct motion_config {
		std::string root;       //!< path on memory stick to "motion" files - for instance, position (.pos) and motion sequence (.mot)
		std::string walk;       //!< the walk parameter file to load by default for new WalkMC's
		char estop_on_snd[50];  //!< sound file to use when e-stop turned on
		char estop_off_snd[50]; //!< sound file to use when e-stop turned off
		float max_head_tilt_speed; //!< max speed for the head joints, used by HeadPointerMC; rad/s
		float max_head_pan_speed; //!< max speed for the head joints, used by HeadPointerMC; rad/s
		float max_head_roll_speed; //!< max speed for the head joints, used by HeadPointerMC; rad/s

		//!returns an absolute path if @a is relative (to root), otherwise just @a name
		std::string makePath(std::string name) { 
			if(name[0]=='/')
				return name;
			if(root[root.size()-1]=='/')
				return root+name;
			else
				return root+"/"+name;
		}

		//!constructor
		motion_config() : root(), walk(), max_head_tilt_speed(0), max_head_pan_speed(0), max_head_roll_speed(0) {
			estop_on_snd[0]=estop_off_snd[0]='\0';
			max_head_tilt_speed=0;
			max_head_pan_speed=0;
			max_head_roll_speed=0;
		}
	} motion;

	//!world model information
	struct worldmodel2_config {
		//@{
		//!ports to use for sending world model information
		int dm_port, hm_port, gm_port, fs_port; //@}

		//!constructor
		worldmodel2_config() : dm_port(0), hm_port(0), gm_port(0), fs_port(0) {}
	} worldmodel2;
	
	//!sound information
	struct sound_config {
		std::string root;         //!< path to sound clips
		unsigned int volume;      //!< volume in decibels - the value is interpreted as a signed short, where 0 is full volume, 0x8000 is mute
		unsigned int sample_rate; //!< sample rate to send to system, currently only 8000 or 16000 supported
		unsigned int sample_bits; //!< sample bit depth, either 8 or 16
		std::vector<std::string> preload; //!< list of sounds to preload at boot
			
		//!returns an absolute path if @a is relative (to root), otherwise just @a name
		std::string makePath(std::string name) { 
			if(name[0]=='/')
				return name;
			if(root[root.size()-1]=='/')
				return root+name;
			else
				return root+"/"+name;
		}

		//!constructor
		sound_config() : root(), volume(0xF600), sample_rate(0), sample_bits(0), preload() {}
	} sound;

	//! call this function when it's time to read the configuration file
	void readConfig(const char* filename);
	//! returns the section structure corresponding to the section name given
  section_t parseSection(const char* key);
	//! pass the section, item name string, item value string - sets the value and returns pointer to the item changed
  void* setValue(section_t section, const char *key, const char *value, bool updated=false);


protected:
	//! returns true if pattern matches model - pattern may have up to 1 '*', case insensitive
	bool matchNoCase(const std::string& model, const std::string& pattern);

	//! returns bool value corresponding to a @a value of "t", "f", "true", "false", "y", "n", "yes", "no", or zero/nonzero number
	static bool extractBool(const char* value);
};

//!allows global access to current settings
extern Config* config;

/*! @file
 * @brief Describes Config, which provides global access to system configuration information
 * @author alokl (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_1 $
 * $Revision: 1.28 $
 * $State: Exp $
 * $Date: 2004/01/30 01:30:07 $
 */

#endif
