//-*-c++-*-
#ifndef INCLUDED_WalkNode_h_
#define INCLUDED_WalkNode_h_

#include "Behaviors/StateNode.h"
#include "Motion/MotionManager.h"
#include "Motion/WalkMC.h"
#include "Motion/MMAccessor.h"

//! A StateNode for walking in a direction
class WalkNode : public StateNode {
public:
	//!constructor
	WalkNode(StateNode * p=NULL)
		: StateNode("WalkNode",p), walkid(MotionManager::invalid_MC_ID), walkidIsMine(true), x(0), y(0), a(0)
	{
		setRetain(false);
	}

	//!constructor, positive is counter-clockwise from above (to match coordinate system)
	WalkNode(float xvel, float yvel, float avel, StateNode * p=NULL)
		: StateNode("WalkNode",p), walkid(MotionManager::invalid_MC_ID), walkidIsMine(true), x(xvel), y(yvel), a(avel)
	{
		setRetain(false);
	}

	//! sets the velocity of the walk
	void setVelocity(float xvel, float yvel, float avel) {
		x=xvel;
		y=yvel;
		a=avel;
		updateWalk(x,y,a);
	}
	
	//! sets the velocity in x direction (positive is forward)
	void setXVelocity(float xvel) { x=xvel; updateWalk(x,y,a); }

	//! returns the velocity in x direction (positive is forward)
	float getXVelocity() { return x; }

	//! sets the velocity in y direction (positive is forward)
	void setYVelocity(float yvel) { y=yvel; updateWalk(x,y,a); }

	//! returns the velocity in y direction (positive is forward)
	float getYVelocity() { return y; }

	//! sets the velocity of the turn, positive is counter-clockwise from above (to match coordinate system)
	void setAVelocity(float avel) { a=avel; updateWalk(x,y,a); }

	//! returns the velocity of the turn, positive is counter-clockwise from above (to match coordinate system)
	float getAVelocity() { return a; }

	virtual void DoStart() {
		StateNode::DoStart();
		updateWalk(x,y,a);
	}

	virtual void DoStop() {
		updateWalk(0,0,0);
		StateNode::DoStop();
	}

	//! removes #walkid if #walkidIsMine
	virtual void teardown() {
		if(walkidIsMine) {
			motman->removeMotion(walkid);
			walkid=MotionManager::invalid_MC_ID;
		}
	}

	//! use this to force the WalkNode to use a shared WalkMC - set to MotionManager::invalid_MC_ID to reset to internally generated walk
	virtual void setWalkID(MotionManager::MC_ID id) {
		if(walkidIsMine) {
			motman->removeMotion(walkid);
			walkid=MotionManager::invalid_MC_ID;
		}
		walkid=id;
		walkidIsMine=(id==MotionManager::invalid_MC_ID);
	}

	//! use this to access the WalkMC that the WalkNode is using
	virtual MotionManager::MC_ID getWalkID() { return walkid; }

	//! returns true if #walkid was created (and will be destroyed) by this WalkNode - false if assigned by setWalkID()
	virtual bool ownsWalkID() { return walkidIsMine; }

protected:
	//!if the walk is invalid, create; then set xya
	void updateWalk(float xvel, float yvel, float avel) {
		if(walkid==MotionManager::invalid_MC_ID) {
			SharedObject<WalkMC> walk;
			walk->setTargetVelocity(xvel,yvel,avel);
			walkid=motman->addMotion(walk);
			walkidIsMine=true;
		} else {
			MMAccessor<WalkMC> walk(walkid);
			walk->setTargetVelocity(xvel,yvel,avel);
		}
	}

	MotionManager::MC_ID walkid; //!< the current WalkMC
	bool walkidIsMine; //!< true if the walk was created in updateWalk (instead of assigned externally)
	float x; //!< velocity in x direction (positive is forward)
	float y; //!< velocity in y direction (positive is dog's left)
	float a; //!< velocity of the turn, positive is counter-clockwise from above (to match coordinate system)
};

/*! @file
 * @brief Describes WalkNode, a  StateNode for walking in a direction
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_1 $
 * $Revision: 1.3 $
 * $State: Exp $
 * $Date: 2003/10/29 22:36:20 $
 */

#endif
