//-*-c++-*-
#ifndef INCLUDED_RawCamBehavior_h_
#define INCLUDED_RawCamBehavior_h_

#include "Behaviors/BehaviorBase.h"
#include "Shared/Config.h"

class Socket;
class FilterBankGenerator;
class FilterBankEvent;

//! Forwards images from camera over wireless
/*! The format used for serialization is basically defined by the
 *  subclass of FilterBankGenerator being used.  I suggest looking at
 *  that classes's documentation to determine the format used.
 *
 *  However, RawCamBehavior will add a few fields at the beginning of
 *  each packet to assist in processing the image stream.
 *
 *  I emphasize: <i>beginning</i> of each Vision packet, <i>before</i> the FilterBankGenerator header. 
 *  - <@c string:"TekkotsuImage">
 *  - <<tt>Config::vision_config::encoding_t</tt>: rawcam_encoding> <i>(expect single or multiple channels, 0 means color (3 channels), 1 means intensity (1 channel))</i>
 *  - <<tt>Config::vision_config::compression_t</tt>: rawcam_compression> <i>(0==none, 1==jpeg, 2==rle)</i>
 *  - <@c unsigned @c int: width> <i>(this is the width of the largest channel - note different channels can be sent at different resolutions!  Provides cheap "compression" of chromaticity channels)</i>
 *  - <@c unsigned @c int: height> <i>(similarly, height of largest channel)</i>
 *  - <@c unsigned @c int: timestamp> <i>(time image was taken, milliseconds since boot)</i>
 *  - <@c unsigned @c int: framenumber> <i>(incremented for each frame, so we can tell if/when we drop one)</i>
 *
 *  This is exactly the same protocol that is followed by the SegCamBehavior as well - the same code can parse either stream.
 */ 
class RawCamBehavior : public BehaviorBase {
public:
	//! constructor
	RawCamBehavior();

	static const unsigned int WIRELESS_BUFFER_SIZE=200000; //!< 200000 bytes for use up to 416x320 + 2*208x160 (double res Y, full res UV on ERS-7)

	virtual void DoStart();

	virtual void DoStop();

	virtual void processEvent(const EventBase& e);

	virtual std::string getName() const { return "RawCamServer"; }

	static std::string getClassDescription() {
		char tmp[20];
		sprintf(tmp,"%d",config->vision.rle_port);
		return std::string("Forwards images from camera over port ")+tmp;
	}
	
protected:
	//! opens a new packet, writes header info; returns true if open, false if otherwise open (check cur==NULL for error)
	/*! see the class documentation for RawCamBehavior for the protocol documentation */
	bool openPacket(FilterBankGenerator& fbkgen, unsigned int time, unsigned int layer);
	bool writeColor(const FilterBankEvent& fbke); //!< writes a color image
	bool writeSingleChannel(const FilterBankEvent& fbke); //!< writes a single channel
	void closePacket(); //!< closes and sends a packet, does nothing if no packet open

	Socket * visRaw; //!< socket for sending the image stream
	char * packet; //!< point to the current buffer being prepared to be sent
	char * cur; //!< current location within that buffer
	unsigned int avail; //!< the number of bytes remaining in the buffer

private:
	RawCamBehavior(const RawCamBehavior&); //!< don't call
	RawCamBehavior& operator=(const RawCamBehavior&); //!< don't call
};

/*! @file
 * @brief Describes RawCamBehavior, which forwards images from camera over wireless
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_1 $
 * $Revision: 1.8 $
 * $State: Exp $
 * $Date: 2004/02/05 23:33:41 $
 */

#endif
