#include "Controller.h"
#include "Motion/EmergencyStopMC.h"
#include "Motion/LedMC.h"
#include "Motion/MMAccessor.h"
#include "Shared/SharedObject.h"
#include "Shared/WorldState.h"
#include "Shared/get_time.h"
#include "SoundPlay/SoundManager.h"
#include "Events/TextMsgEvent.h"
#include "Shared/ERS210Info.h"
#include "Shared/ERS220Info.h"
#include "Shared/ERS7Info.h"
#include "Shared/string_util.h"
#include <sstream>

Controller * Controller::theOneController=NULL;

//these are given appropriate values in init once we know which model we're running on
EventBase Controller::nextItem;
EventBase Controller::prevItem;
EventBase Controller::nextItemFast;
EventBase Controller::prevItemFast;
EventBase Controller::selectItem;
EventBase Controller::cancel;

using namespace string_util;


void Controller::DoStart() {
	BehaviorBase::DoStart();
	sndman->LoadFile(config->controller.select_snd);
	sndman->LoadFile(config->controller.next_snd);
	sndman->LoadFile(config->controller.prev_snd);
	sndman->LoadFile(config->controller.read_snd);
	sndman->LoadFile(config->controller.cancel_snd);
	erouter->addListener(this,EventBase::estopEGID);
	// Turn on wireless
	gui_comm=wireless->socket(SocketNS::SOCK_STREAM, 2048, 32000);
	wireless->setReceiver(gui_comm->sock, gui_comm_callback);
	wireless->setDaemon(gui_comm,true);
	wireless->listen(gui_comm->sock, config->controller.gui_port);
	theOneController=this;
	SharedObject<LedMC> leds;
	leds->setWeights(~FaceLEDMask,0);
	leds->setWeights(FaceLEDMask,.75);
	display=motman->addMotion(leds,isControlling?MotionManager::kEmergencyPriority:MotionManager::kIgnoredPriority);
	reset();
}

void Controller::DoStop() {
	motman->removeMotion(display);
	//these two lines help prevent residual display in case that was the only MotionCommand using LEDs
	for(unsigned int i=LEDOffset; i<LEDOffset+NumLEDs; i++)
		motman->setOutput(NULL,i,0.f);
	display=MotionManager::invalid_MC_ID;
	sndman->ReleaseFile(config->controller.select_snd);
	sndman->ReleaseFile(config->controller.next_snd);
	sndman->ReleaseFile(config->controller.prev_snd);
	sndman->ReleaseFile(config->controller.read_snd);
	sndman->ReleaseFile(config->controller.cancel_snd);
	erouter->forgetListener(this);
	reset();
	gui_comm->printf("goodbye\n");
	wireless->setDaemon(gui_comm,false);
	wireless->close(gui_comm);
	theOneController=NULL;
	BehaviorBase::DoStop();
}

bool Controller::trapEvent(const EventBase& e) {
	if(!chkCmdStack())
		return false;
	last_time=cur_time;
	cur_time=get_time();
	//this will prevent inadvertant controller commands when you pick up an ERS-7
	if(state->buttons[nextItem.getSourceID()] && state->buttons[prevItem.getSourceID()] && state->buttons[selectItem.getSourceID()])
		return true;
	
	if(nextItem.sameGenSource(e)) {
		nextEv_val=e.getMagnitude();
		nextEv_dur=e.getDuration();
		if(nextEv_val==0 && prevEv_val==0)
			alreadyGotBoth=false;
		if(nextEv_val>.75 && prevEv_val>.75 && nextEv_dur<666 && prevEv_dur<666)
			if(alreadyGotBoth)
				return true;
			else {
				alreadyGotBoth=true;
				return setNext(cmdstack.top()->doReadStdIn());
			}
		if(e.getTypeID()==nextItem.getTypeID() && e.getDuration()<666)
			return setNext(cmdstack.top()->doNextItem());
		if(e.getTypeID()==nextItemFast.getTypeID() && e.getDuration()>666 && calcPulse(cur_time,last_time,static_cast<unsigned int>(50/e.getMagnitude())))
			return setNext(cmdstack.top()->doNextItem());
	}
	if(prevItem.sameGenSource(e)) {
		prevEv_val=e.getMagnitude();
		prevEv_dur=e.getDuration();
		if(nextEv_val==0 && prevEv_val==0)
			alreadyGotBoth=false;
		if(nextEv_val>.75 && prevEv_val>.75 && nextEv_dur<666 && prevEv_dur<666)
			if(alreadyGotBoth)
				return true;
			else {
				alreadyGotBoth=true;
				return setNext(cmdstack.top()->doReadStdIn());
			}
		if(e.getTypeID()==prevItem.getTypeID() && e.getDuration()<666)
			return setNext(cmdstack.top()->doPrevItem());
		if(e.getTypeID()==prevItemFast.getTypeID() && e.getDuration()>666 && calcPulse(cur_time,last_time,static_cast<unsigned int>(50/e.getMagnitude())))
			return setNext(cmdstack.top()->doPrevItem());
	}
	if(e.getDuration()>250) {
		if(e==selectItem)
			return setNext(cmdstack.top()->doSelect());
		if(e==cancel)
			return setNext(cmdstack.top()->doCancel());
	}
	return true;
}

void Controller::processEvent(const EventBase& event) {
	if(event.getTypeID()==EventBase::activateETID) { //estop just turned on
		if(!isControlling)
			activate();
	}	else { //estop just turned off
		if(isControlling)
			deactivate();
	}
}

void Controller::reset() {
	while(cmdstack.size()>1)
		pop();
	if(!cmdstack.empty()) {
		cmdstack.top()->deactivate();
		cmdstack.pop();
	}
	refresh();
}

void Controller::refresh() {
	if(!chkCmdStack())
		return;
	cmdstack.top()->refresh();
}

void Controller::push(ControlBase* c) {
	if(!chkCmdStack())
		return;
	cmdstack.top()->pause();
	cmdstack.push(c);
	theOneController->gui_comm->printf("push\n");
	setNext(cmdstack.top()->activate(display,gui_comm));
}

void Controller::pop() {
	cmdstack.top()->deactivate();
	cmdstack.pop();
	theOneController->gui_comm->printf("pop\n");
	refresh();
}

Controller& Controller::setRoot(ControlBase* r) {
	reset();
	root=r;
	refresh();
	return *this;
}

Controller& Controller::setEStopID(MotionManager::MC_ID estopid) {
	estop_id=estopid;
	if(static_cast<EmergencyStopMC*>(motman->peekMotion(estopid))->getStopped()) {
		if(!isControlling)
			activate();
	} else {
		if(isControlling)
			deactivate();
	}		
	return *this;
}

void Controller::loadGUI(const std::string& type, const std::string& name, unsigned int port, const std::vector<std::string>& args) {
	if(theOneController==NULL)
		return;
	std::stringstream ss;
	ss << "load\n" << type << '\n' << name << '\n' << port << '\n';
	for(unsigned int i=0; i<args.size(); i++) {
		ss << '"';
		for(unsigned int j=0; j<args[i].size(); j++) {
			if(args[i][j]=='\\' || args[i][j]=='"' || args[i][j]=='\n')
				ss << '\\';
			ss << args[i][j];
		}
		ss << "\" ";
	}
	ss << '\n';
	theOneController->gui_comm->write((const byte*)ss.str().c_str(),ss.str().size());
}

void Controller::closeGUI(const std::string& name) {
	if(theOneController==NULL)
		return;
	ASSERTRET(theOneController->gui_comm!=NULL,"null gui_comm");

	theOneController->gui_comm->printf("close\n%s\n",name.c_str());
}

int Controller::gui_comm_callback(char *buf, int bytes) {
	std::string s(buf,bytes);
	//	cout << "Controller Received: " << s << endl;
	if(theOneController==NULL)
		return 0;

	static std::string incomplete;

	//pass a line at a time to the controller
	while(s.size()>0) {
		unsigned int endline=s.find('\n');
		if(endline==std::string::npos) {
			incomplete+=s;
			return 0;
		}
		
		//strip a \r\n or a \n
		if(endline>0 && s[endline-1]=='\r')
			incomplete+=s.substr(0,endline-1);
		else
			incomplete+=s.substr(0,endline);
		
		//is now complete
		theOneController->takeLine(incomplete); 
		incomplete.erase();
		s=s.substr(endline+1);
	}
	
	return 0;
}

int Controller::console_callback(char *buf, int bytes) {
	std::string s(buf,bytes);
	//	cout << "Console Received: " << s << endl;
	if(theOneController==NULL)
		return 0;

	static std::string incomplete;

	//pass a line at a time to the controller
	while(s.size()>0) {
		unsigned int endline=s.find('\n');
		if(endline==std::string::npos) {
			incomplete+=s;
			return 0;
		}

		//strip a \r\n or a \n
		if(endline>0 && s[endline-1]=='\r')
			incomplete+=s.substr(0,endline-1);
		else
			incomplete+=s.substr(0,endline);
		
		//is now complete
		if(wireless->isConnected(theOneController->gui_comm->sock))
			erouter->postEvent(new TextMsgEvent(incomplete));
		else
			theOneController->takeLine(incomplete); 
		incomplete.erase();
		s=s.substr(endline+1);
	}
	
	return 0;
}

void Controller::init() {
	if(state->robotDesign & WorldState::ERS210Mask) {
		nextItem=EventBase(EventBase::buttonEGID,ERS210Info::HeadFrButOffset,EventBase::deactivateETID,0);
		prevItem=EventBase(EventBase::buttonEGID,ERS210Info::HeadBkButOffset,EventBase::deactivateETID,0);
		nextItemFast=EventBase(EventBase::buttonEGID,ERS210Info::HeadFrButOffset,EventBase::statusETID,666);
		prevItemFast=EventBase(EventBase::buttonEGID,ERS210Info::HeadBkButOffset,EventBase::statusETID,666);
		selectItem=EventBase(EventBase::buttonEGID,ERS210Info::ChinButOffset,EventBase::deactivateETID,250);
		cancel=EventBase(EventBase::buttonEGID,ERS210Info::BackButOffset,EventBase::deactivateETID,250);
	} else if(state->robotDesign & WorldState::ERS220Mask) {
		nextItem=EventBase(EventBase::buttonEGID,ERS220Info::TailLeftButOffset,EventBase::deactivateETID,0);
		prevItem=EventBase(EventBase::buttonEGID,ERS220Info::TailRightButOffset,EventBase::deactivateETID,0);
		//the 220 doesn't really support the next two because it's using boolean buttons
		//i'm using a "hack" on the 210 because the pressure sensitivity causes status
		//events to continually be sent but since this is just on/off, it only gets the
		//activate/deactivate.  To fix this, make these timers and do timer management
		//in processEvents()
		nextItemFast=EventBase(EventBase::buttonEGID,ERS220Info::TailLeftButOffset,EventBase::statusETID,666);
		prevItemFast=EventBase(EventBase::buttonEGID,ERS220Info::TailRightButOffset,EventBase::statusETID,666);
		selectItem=EventBase(EventBase::buttonEGID,ERS220Info::TailCenterButOffset,EventBase::deactivateETID,50);
		cancel=EventBase(EventBase::buttonEGID,ERS220Info::BackButOffset,EventBase::deactivateETID,50);
	} else if(state->robotDesign & WorldState::ERS7Mask) {
		nextItem=EventBase(EventBase::buttonEGID,ERS7Info::FrontBackButOffset,EventBase::deactivateETID,0);
		prevItem=EventBase(EventBase::buttonEGID,ERS7Info::RearBackButOffset,EventBase::deactivateETID,0);
		nextItemFast=EventBase(EventBase::buttonEGID,ERS7Info::FrontBackButOffset,EventBase::statusETID,500);
		prevItemFast=EventBase(EventBase::buttonEGID,ERS7Info::RearBackButOffset,EventBase::statusETID,500);
		selectItem=EventBase(EventBase::buttonEGID,ERS7Info::MiddleBackButOffset,EventBase::deactivateETID,25);
		cancel=EventBase(EventBase::buttonEGID,ERS7Info::HeadButOffset,EventBase::deactivateETID,25);
	} else {
		serr->printf("Controller: Unsupported model!\n");
	}
}

void Controller::takeLine(const std::string& s) {
	//	cout << "RECEIVED: " << s << endl;
	if(s.size()==0)
		return;
	// break s into a vector of arguments
	std::vector<std::string> args;
	std::vector<unsigned int> offsets;
	if(!string_util::parseArgs(s,args,offsets)) {
		serr->printf("Controller::takeLine(\"%s\") was malformed.\n",s.c_str());
		return;
	}
	if(args.size()==0 || offsets.size()==0)
		return;
	// now look through for a ';' (separates multiple commands)
	unsigned int last=offsets[0];
	for(unsigned int i=0; i<args.size(); i++) {
		if(args[i]==";") { // if we found a ';', recurse with substring
			takeLine(s.substr(last,offsets[i]-last));
			if(i+1==args.size()) // last arg is a ';'
				return;
			last=offsets[i+1];
		}
		if(args[i]=="\\;") // if we found a '\;', replace it with base ';'
			args[i]=";";
	}
	if(!chkCmdStack())
		return;
	if(args[0][0]!='!') {
		setNext(cmdstack.top()->takeInput(s));
	} else {
		if(last!=offsets[0]) { // only changes if we found a ';' - in that case, need to do last segment
			takeLine(s.substr(last));
		} else if(args[0]=="!refresh") {
			refresh();
		} else if(args[0]=="!reset") {
			reset();
		} else if(args[0]=="!cancel") {
			setNext(cmdstack.top()->doCancel());
		} else if(args[0]=="!select") {
			setNext(cmdstack.top()->doSelect());
		} else if(args[0]=="!next") {
			setNext(cmdstack.top()->doNextItem());
		} else if(args[0]=="!prev") {
			setNext(cmdstack.top()->doPrevItem());
		} else if(args[0]=="!msg") {
			if(offsets.size()>1)
				erouter->postEvent(new TextMsgEvent(s.substr(offsets[1])));
			else
				erouter->postEvent(new TextMsgEvent(""));
		} else if(args[0]=="!hello") {
			static unsigned int count=0;
			count++;
			theOneController->gui_comm->printf("hello\n%d\n",count);
		} else if(args[0]=="!root") {
			ControlBase * ret=root->takeInput(s.substr(offsets[1]));
			if(ret!=NULL)
				setNext(ret);
		} else if(args[0]=="!hilight") {
			std::vector<unsigned int> hilights;
			for(unsigned int i=1; i<args.size(); i++)
				hilights.push_back(atoi(args[i].c_str()));
			cmdstack.top()->setHilights(hilights);
		} else if(args[0]=="!input") {
			const std::vector<unsigned int>& hilights=cmdstack.top()->getHilights();
			const std::vector<ControlBase*>& slots=cmdstack.top()->getSlots();
			std::string in=s.substr(offsets[1]);
			for(unsigned int i=0; i<hilights.size(); i++)
				if(hilights[i]<slots.size() && slots[hilights[i]]!=NULL) {
					ControlBase * ret=slots[hilights[i]]->takeInput(in);
					if(ret!=NULL)
						setNext(ret);
				}
			refresh();
		} else if(args[0]=="!set") {
      setConfig(s.substr(offsets[1]).c_str());
    } else
			setNext(cmdstack.top()->takeInput(s));
	}
}

int Controller::setConfig(const char *str) {
	char buf[80];
	strncpy(buf, str, 79);
	char *value=index(buf, '=');
	char *key=index(buf, '.');
	if (key==NULL || value==NULL) return -1;
	if (key>=value) return -1;
	*key=0;
	key++;
	*value=0;
	value++;
	Config::section_t section=config->parseSection(buf);
	if (section==Config::sec_invalid) return -2;
	config->setValue(section, key, value, true);
	//void *val_set=config->setValue(section, key, value, true);
	// might want to catch setValue's return value and do
	// something special for some config values?
	// (such as reboot a subsystem to reload new settings)
	return 0;
}

bool Controller::setNext(ControlBase* next) {
	if(next==NULL)
		pop();
	else if(next!=cmdstack.top())
		push(next);
	return true;
}

void Controller::activate() {
	motman->setPriority(display,MotionManager::kEmergencyPriority);
	erouter->addTrapper(this,EventBase::buttonEGID);
	isControlling=true;
	if(!cmdstack.empty())
		cmdstack.top()->activate(display,gui_comm);
	else
		chkCmdStack();
}

void Controller::deactivate() {
	//these two lines help prevent residual display in case that was the only MotionCommand using LEDs
	motman->setPriority(display,MotionManager::kIgnoredPriority);
	isControlling=false;
	for(unsigned int i=LEDOffset; i<LEDOffset+NumLEDs; i++)
		motman->setOutput(NULL,i,0.f);
	erouter->removeTrapper(this);
	cmdstack.top()->pause();
}

bool Controller::chkCmdStack() {
	if(cmdstack.empty()) {
		if(root==NULL)
			return false;
		cmdstack.push(root);
		ControlBase * next = cmdstack.top()->activate(display,gui_comm);
		if(next==NULL)
			cout << "*** WARNING Controller root returned NULL on activate!" << endl;
		else if(next!=root)
			push(next);
	}
	return true;
}


/*! @file
 * @brief Implements Controller class, a behavior that should be started whenever the emergency stop goes on to provide menus for robot control
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_1 $
 * $Revision: 1.36 $
 * $State: Exp $
 * $Date: 2004/03/01 21:16:48 $
 */
