function [H] = ransacH(im1,im2)
% Performs RANSAC-based computation of a homography between two images.

% Configuration constants
% Number of iterations
iterations = 500;

% Predeclare F matrix and the best score, which should get lower and lower
% as F improves.
H = [];
score = inf;

% Run the furnished getCandidateCorrespondences method
[f1_x, f1_y, f2_x, f2_y] = ...
	getCandidateCorrespondences(im1, im2);

% Make sure we have enough correspondences
if length(f1_x) < 10
  error 'Not enough candidate correspondences'
end

% Store scaled versions of candidate correspondences--we use them a lot
scaled_im1_pts = img2math([f1_x f1_y]');
scaled_im2_pts = img2math([f2_x f2_y]');

% Do RANSAC--iterate
for i = 1:iterations
  % Choose the points we'll use to find this H.
  % To determine empirically: how many points is optimal?
  contestants = randperm(length(f1_x));
  contestants = contestants(1:10);

  % Construct the X and Y matrices for the selected points
  trial_im1 = [f1_x(contestants) f1_y(contestants)]';
  trial_im2 = [f2_x(contestants) f2_y(contestants)]';

  % Estimate H
  trial_H = computeH(trial_im1, trial_im2);

  % Compute Euclidean error of H.
  %errs = ([f2_x f2_y]' - math2img(trial_H*scaled_im1_pts)).^2;
  errs = (100*scaled_im2_pts - 100*math2math(trial_H*scaled_im1_pts)).^2;
  errs = errs(1,:) + errs(2,:);

  % Step 4: Eliminate outliers--pick the smallest 50% of elements.
  % Start by sorting
  errs = sortrows(errs');
  % Eliminate the larger values
  errs = errs(1:ceil(length(errs)*2/3));

  % Compute score
  trial_score = sum(errs);

  % See if we have a new winner
  if trial_score < score
    H = trial_H;
    score = trial_score;
    disp(sprintf('New best on iteration %d! Score: %f', i, score));
  end
end


%% Helper functions
 
function mp=img2math(ip)
% Convert image coordinates to the homogeneous coordinates we use for math
mp = [ip*0.01; ones(1, size(ip,2))];
 
function ip=math2img(mp)
% Convert homogeneous coordinates used for math to image-size coordinates
ip = round( mp(1:2,:)*100 ./ [mp(3,:);mp(3,:)] );

function mps=math2math(mp)
% Scale homogeneous coordinates so that their scale factor is 1
mps = mp ./ [mp(3,:);mp(3,:);mp(3,:)];
