#include "SegmentedColorGenerator.h"
#include "Events/EventRouter.h"
#include "Events/SegmentedColorFilterBankEvent.h"
#include "Wireless/Wireless.h"
#include "Shared/WorldState.h"

#include "Shared/debuget.h"

SegmentedColorGenerator::SegmentedColorGenerator(EventBase::EventGeneratorID_t gid, unsigned int sid, unsigned int mysid)
	: FilterBankGenerator("SegmentedColorGenerator",0,0,gid,sid,EventBase::visSegmentEGID,mysid), src(NULL), srcYChan(0), srcUChan(1), srcVChan(2), tmaps(), tmapNames(), numColors(0), colorNames()
{
	setNumImages(numLayers,numChannels);
}

SegmentedColorGenerator::SegmentedColorGenerator(EventBase::EventGeneratorID_t gid, unsigned int sid, unsigned int mysid, unsigned int syc, unsigned int suc, unsigned int svc)
	: FilterBankGenerator("SegmentedColorGenerator",0,0,gid,sid,EventBase::visSegmentEGID,mysid), src(NULL), srcYChan(syc), srcUChan(suc), srcVChan(svc), tmaps(), tmapNames(), numColors(0), colorNames()
{
	setNumImages(numLayers,numChannels);
}

SegmentedColorGenerator::~SegmentedColorGenerator() {
	for(unsigned int i=0; i<tmaps.size(); i++)
		delete [] tmaps[i];
	// todo - i think we're leaking the color names memory
}

void
SegmentedColorGenerator::processEvent(const EventBase& event) {
	const FilterBankEvent& fbkevent=dynamic_cast<const FilterBankEvent& >(event);
	src=fbkevent.getSource();
	frameNumber=src->getFrameNumber();
	if(src->getNumLayers()!=numLayers || src->getNumChannels()!=numChannels)
		setNumImages(src->getNumLayers(),1);
	if(src->getWidth(numLayers-1)!=getWidth(numLayers-1))
		setDimensions();

	invalidateCaches();
	framesProcessed++;
	erouter->postEvent(new SegmentedColorFilterBankEvent(this,getGeneratorID(),getSourceID(),this,getNumColors(),getColors(),&colorNames));
}

unsigned int
SegmentedColorGenerator::loadThresholdMap(const std::string& tm_file) {
  const unsigned int size = 1 << (BITS_Y + BITS_U + BITS_V);
	cmap_t * tmap = new cmap_t[size];
  if(!CMVision::LoadThresholdFile(tmap,NUM_Y,NUM_U,NUM_V,tm_file.c_str())) {
    serr->printf("  ERROR: Could not load threshold file '%s'.\n",tm_file.c_str());
		delete [] tmap;
		return -1U;
  }
	if(numColors!=0) {
		//we've already loaded color info, check against that for invalid indexes
		int remapped=CMVision::CheckTMapColors(tmap,NUM_Y,NUM_U,NUM_V,numColors,0);
		if(remapped>0)
			serr->printf("remapped %d colors in threshold file '%s'\n",remapped, tm_file.c_str());
	}

	tmapNames.push_back(tm_file);
	tmaps.push_back(tmap);
	setNumImages(numLayers,tmaps.size());
	return tmaps.size()-1;
}

bool
SegmentedColorGenerator::loadColorInfo(const std::string& col_file) {
	// todo - we're leaking the color names memory
  colorNames.clear();
  numColors=CMVision::LoadColorInformation(colors,MAX_COLORS,col_file.c_str(),colorNames);
  if(numColors > 0){
    sout->printf("  Loaded %d colors.\n",numColors);
  } else {
    serr->printf("  ERROR: Could not load colors file:%s\n", col_file.c_str());
		return false;
  }

	//we'd better check the already loaded thresholds (if any) for out of bound indexes
	for(unsigned int i=0; i<tmaps.size(); i++) {
		int remapped=CMVision::CheckTMapColors(tmaps[i],NUM_Y,NUM_U,NUM_V,numColors,0);
		if(remapped>0)
			serr->printf("remapped %d colors in threshold file '%s'\n",remapped, tmapNames[i].c_str());
	}
	return true;
}


unsigned int
SegmentedColorGenerator::getBinSize() const {
	unsigned int used=FilterBankGenerator::getBinSize();
	used+=strlen("SegColorImage")+LoadSave::stringpad;
	used+=widths[selectedSaveLayer]*heights[selectedSaveLayer];
	return used;
}

unsigned int
SegmentedColorGenerator::LoadBuffer(const char buf[], unsigned int len) {
	unsigned int origlen=len;
	unsigned int used;
	std::string tmp;
	if(0==(used=FilterBankGenerator::LoadBuffer(buf,len))) return 0;
	len-=used; buf+=used;
	if(0==(used=decode(tmp,buf,len))) return 0;
	len-=used; buf+=used;
	if(tmp!="SegColorImage") {
		serr->printf("Unhandled image type for SegmentedColorGenerator: %s",tmp.c_str());
		return 0;
	} else {
		// actual image
		used=widths[selectedSaveLayer]*heights[selectedSaveLayer];
		if(used>len)
			return 0;
		if(images[selectedSaveLayer][selectedSaveChannel]==NULL)
			images[selectedSaveLayer][selectedSaveChannel]=createImageCache(selectedSaveLayer,selectedSaveChannel);
		unsigned char* img=images[selectedSaveLayer][selectedSaveChannel];
		if(img==NULL)
			return 0;
		memcpy(img,buf,used);
		len-=used; buf+=used;

		// color table
		if(0==(used=decodeColors(buf,len))) return 0;
		len-=used; buf+=used;
		
		imageValids[selectedSaveLayer][selectedSaveChannel]=true;
		return origlen-len;	
	}
}

unsigned int
SegmentedColorGenerator::SaveBuffer(char buf[], unsigned int len) const {
	unsigned int origlen=len;
	unsigned int used;
	if(0==(used=FilterBankGenerator::SaveBuffer(buf,len))) return 0;
	len-=used; buf+=used;
	if(0==(used=encode("SegColorImage",buf,len))) return 0;
	len-=used; buf+=used;
	
	// actual image
	used=widths[selectedSaveLayer]*heights[selectedSaveLayer];
	if(used>len)
		return 0;
	unsigned char* img=getImage(selectedSaveLayer,selectedSaveChannel);
	if(img==NULL)
		return 0;
	memcpy(buf,img,used);
	len-=used; buf+=used;

	// color table
	if(0==(used=encodeColors(buf,len))) return 0;
	len-=used; buf+=used;
	
	return origlen-len;
}

unsigned int
SegmentedColorGenerator::encodeColors(char buf[], unsigned int len) const {
	unsigned int origlen=len;
	unsigned int used;
	if(0==(used=encode(numColors,buf,len))) return 0;
	len-=used; buf+=used;
	for(unsigned int i=0; i<numColors; i++) {
		if(0==(used=encode(colors[i].color.red,buf,len))) return 0;
		len-=used; buf+=used;
		if(0==(used=encode(colors[i].color.green,buf,len))) return 0;
		len-=used; buf+=used;
		if(0==(used=encode(colors[i].color.blue,buf,len))) return 0;
		len-=used; buf+=used;
	}
	return origlen-len;	
}

unsigned int
SegmentedColorGenerator::decodeColors(const char buf[], unsigned int len) {
	unsigned int origlen=len;
	unsigned int used;
	if(0==(used=decode(numColors,buf,len))) return 0;
	len-=used; buf+=used;
	for(unsigned int i=0; i<numColors; i++) {
		if(0==(used=decode(colors[i].color.red,buf,len))) return 0;
		len-=used; buf+=used;
		if(0==(used=decode(colors[i].color.green,buf,len))) return 0;
		len-=used; buf+=used;
		if(0==(used=decode(colors[i].color.blue,buf,len))) return 0;
		len-=used; buf+=used;
	}
	return origlen-len;	
}

void
SegmentedColorGenerator::setDimensions() {
	for(unsigned int i=0; i<numLayers; i++) {
		widths[i]=src->getWidth(i);
		heights[i]=src->getHeight(i);
		skips[i]=0;
		strides[i]=widths[i];
	}	
}

void
SegmentedColorGenerator::setNumImages(unsigned int nLayers, unsigned int /*nChannels*/) {
	FilterBankGenerator::setNumImages(nLayers,tmaps.size());
}

unsigned char *
SegmentedColorGenerator::createImageCache(unsigned int layer, unsigned int /*chan*/) const {
	return new unsigned char[widths[layer]*heights[layer]];
}

void
SegmentedColorGenerator::calcImage(unsigned int layer, unsigned int chan) const {
	PROFSECTION("SegmentedColorGenerator::calcImage(...)",state->mainProfile);
	CMVision::image_yuv<const cmap_t> img;
	img.buf_y=src->getImage(layer,srcYChan);
	img.buf_u=src->getImage(layer,srcUChan);
	img.buf_v=src->getImage(layer,srcVChan);
	img.width=getWidth(layer);
	img.height=getHeight(layer);
  img.row_stride=src->getStride(layer);

  CMVision::ThresholdImageYUVPlanar<cmap_t,CMVision::image_yuv<const cmap_t>,const cmap_t,BITS_Y,BITS_U,BITS_V>(images[layer][chan],img,tmaps[chan]);
	imageValids[layer][chan]=true;
}

/*! @file
 * @brief Implements SegmentedColorGenerator, which generates FilterBankEvents indexed color images based on a color threshold file
 * @author alokl (Creator)
 * @author ejt (reorganized)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_0 $
 * $Revision: 1.5 $
 * $State: Exp $
 * $Date: 2004/01/16 07:39:51 $
 */

