//-*-c++-*-
#ifndef INCLUDED_FilterBankGenerator_h_
#define INCLUDED_FilterBankGenerator_h_

#include "Events/EventGeneratorBase.h"
#include "Shared/LoadSave.h"

//! Abstract base class for generators of FilterBankEvents
/*! This is needed to provide an interface for the FilterBankEvent to
 *  call back when the actual image data is requested from it.  This
 *  facilitates lazy calculation of image data...  no sense in
 *  processing layers or channels which aren't actually going to be
 *  used...
 *
 *  Also this way we save on allocating/deallocating large memory
 *  blocks on each event... the buffers allocated here can be reused
 *  frame to frame.
 *
 *  Larger layer indicies generally indicate higher resolution images
 *  in a scaling pyramid, but you are free to store your own data
 *  however you wish.
 *
 *  For serializing, FilterBankGenerators inherit from the LoadSave
 *  interface.  Only one image (selected with setLoadSaveImage()) of
 *  the bank will loaded or saved at a time.  The default LoadSave
 *  will store some header information for you:
 *  
 *  - Creator code: string ("\008FBkImage\0")
 *  - Width: unsigned int
 *  - Height: unsigned int
 *  - Layer: unsigned int
 *  - Channel: unsigned int
 *  
 *  However, you will need to override the LoadSave functions (listed
 *  below) to provide your own code for interpreting the image data
 *  itself.  You may also want to add a few extra functions to allow
 *  users to set compression/data format parameters.
 *
 *  If you're doing fancy memory stuff, you probably want to override
 *  the freeCaches() and destruct() functions so that the default
 *  implementation won't try to free something it shouldn't.  Don't
 *  forget to call them from your own destructor though, otherwise
 *  your versions won't get called before the default implementation's
 *  does.
 *
 *  @see RawCameraGenerator
 */
class FilterBankGenerator : public EventGeneratorBase, public LoadSave {
public:
	//! constructor, calls setNumImages(1,1)
	FilterBankGenerator()
		: EventGeneratorBase(), numLayers(0), numChannels(0), widths(NULL), heights(NULL), skips(NULL), strides(NULL), increments(NULL), images(NULL), imageValids(NULL), selectedSaveLayer(0), selectedSaveChannel(0), frameNumber(0), framesProcessed(0)
	{
		setNumImages(1,1);
	}

	//! constructor
	FilterBankGenerator(unsigned int layers, unsigned int channels)
		: EventGeneratorBase(), numLayers(0), numChannels(0), widths(NULL), heights(NULL), skips(NULL), strides(NULL), increments(NULL), images(NULL), imageValids(NULL), selectedSaveLayer(0), selectedSaveChannel(0), frameNumber(0), framesProcessed(0)
	{
		setNumImages(layers,channels);
	}

	//! constructor
	FilterBankGenerator(const std::string& name,unsigned int layers, unsigned int channels, EventBase::EventGeneratorID_t srcgid, unsigned int srcsid, EventBase::EventGeneratorID_t mgid, unsigned int msid)
		: EventGeneratorBase(name,mgid,msid,srcgid,srcsid), numLayers(0), numChannels(0), widths(NULL), heights(NULL), skips(NULL), strides(NULL), increments(NULL), images(NULL), imageValids(NULL), selectedSaveLayer(0), selectedSaveChannel(0), frameNumber(0), framesProcessed(0)
	{
		setNumImages(layers,channels);
	}

	//! destructor
	~FilterBankGenerator() {
		freeCaches();
		destruct();
	}

	//! returns the number of image layers (e.g. different resolutions available)
	virtual unsigned int getNumLayers() const { return numLayers; }

	//! returns the number of channels per image (e.g. Y, U, or V components)
	virtual unsigned int getNumChannels() const { return numChannels; }
	
	//! returns pointer to the beginning of the image data for the specified layer and channel
	/*! this will cause the data to be calculated and cached if it's not already available */
	virtual unsigned char * getImage(unsigned int layer, unsigned int channel) const;

	//! returns width (in samples) of the image in a given layer
	unsigned int getWidth(unsigned int layer) const { return widths[layer]; }

	//! returns height (in samples) of the image in a given layer
	unsigned int getHeight(unsigned int layer) const { return heights[layer]; }
	
	//! returns the bytes to skip from the one-past-end of a row to get the beginning of the next
	unsigned int getSkip(unsigned int layer) const { return skips[layer]; }
	
	//! returns the bytes to skip from the beginning of one row to get the beginning of the next
	/*! This is just for convenience; the stride is just the skip plus the width, but it's precomputed for you for speed and clarity */
	unsigned int getStride(unsigned int layer) const { return strides[layer]; }

	//! returns the increment (in bytes) to use to go from one sample to the next
	unsigned int getIncrement(unsigned int layer) const { return increments[layer]; }
	
	//! returns the frame number of the current frame, see #frameNumber
	unsigned int getFrameNumber() const { return frameNumber; }
	
	//! returns the number of frames processed, see #framesProcessed
	unsigned int getFramesProcessed() const { return framesProcessed; }
	
	//! deletes storage of cached images and marks it invalid
	/*! you should override this if the images cache pointer isn't actually an array of bytes... 
	 *  Don't forget to call it in your subclass's destructor or your version won't get called... */
	virtual void freeCaches();

	//! marks all of the cached images as invalid (but doesn't free their memory)
	/*! You probably want to call this right before you send the FilterBankEvent */
	virtual void invalidateCaches();

	//! default implementation ignore events, just so you don't have to define it if you don't receive events
	virtual void processEvent(const EventBase & /*event*/) {}
	
	//!@name LoadSave interface

	virtual unsigned int getBinSize() const;

	virtual unsigned int LoadBuffer(const char buf[], unsigned int len);

	virtual unsigned int SaveBuffer(char buf[], unsigned int len) const;

	//! not actually part of the LoadSave interface, but allows you to select which image of the bank will be saved
	/*! when loading, the saved image's layer and channel will reset this */
	virtual void selectSaveImage(unsigned int layer, unsigned int channel) { selectedSaveLayer=layer; selectedSaveChannel=channel; }

	virtual unsigned int getSelectedSaveLayer() const { return selectedSaveLayer; } //!< returns layer to be saved, or layer of last image loaded
	virtual unsigned int getSelectedSaveChannel() const { return selectedSaveChannel; } //!< returns channel to be saved, or channel of last image loaded

	//@}


protected:
	//! resizes the filter bank information storage area, you should override this to do your setup and call it from your constructor
	/*! In general, it isn't expected that FilterBankGenerator's should
	 *  necessarily be dynamically resizeable (although it would be
	 *  nice), which is why this isn't public.  If yours is, just add
	 *  some pubic accessor functions which call this.  In general, the
	 *  included subclasses should be able to handle being resized, but
	 *  there's no reason to do so since the system won't be changing
	 *  its available resolutions at run time. */
	virtual void setNumImages(unsigned int nLayers, unsigned int nChannels);

	//! create new image data storage area for the cache - this is only called when the corresponding entry in images is NULL
	/*! You should return the pointer you want stored in images to be
	 *  returned by any calls to getFirstRow.  Interpretation of the
	 *  data it points to is dependant on the the generator which
	 *  creates it */
	virtual unsigned char * createImageCache(unsigned int layer, unsigned int channel) const=0;

	//! should calculate new image data, only called when imageValids indicates the image being requested is dirty
	/*! This is where you'll want to put your user-specific code for calculating the image data */
	virtual void calcImage(unsigned int layer, unsigned int channel) const=0;

	//! deletes the arrays
	virtual void destruct();

	unsigned int numLayers;   //!< current number of layers available
	unsigned int numChannels; //!< current number of channels available

	unsigned int * widths;    //!< an array of size numLayers, width (in samples) in pixels of each layer
	unsigned int * heights;   //!< an array of size numLayers, height (in samples) in pixels of each layer
	unsigned int * skips;     //!< an array of size numLayers, skip (in bytes) from row end to next row begin
	unsigned int * strides;   //!< an array of size numLayers, stride (in bytes) from a given column in one row to the same column in the next row
	unsigned int * increments;//!< an array of size numLayers, increment (in bytes) to use to get from one sample to the next
	
	mutable unsigned char *** images; //!< an array [numLayers][numChannels], stores pointer to cached image data
	mutable bool ** imageValids;      //!< an array [numLayers][numChannels], entry is true if cached data is still valid

	unsigned int selectedSaveLayer;   //!< layer to be manipulated with the LoadSave functions
	unsigned int selectedSaveChannel; //!< channel to be manipulated with the LoadSave functions

	//! the current frame number - subclasses will need to set to the source's frameNumber when they receive a new frame (probably from processEvent())	
	/*! The idea is to use this as a unique serial number for each
	 *	frame.  That way you can know if the current image in different
	 *	generators is actually the same camera image before you try to
	 *	compare or combine them.
	 *
	 *  You could also figure out the number of dropped frames by
	 *  subtracting framesProcessed from this value.  Give some leeway
	 *  however, because it takes the first 40-70 frames just to boot
	 *  up, so there's no way they can be processed.
	 */
	unsigned int frameNumber; 
	unsigned int framesProcessed; //!< subclasses should increment this any time they make a new filter bank available (probably by throwing an event)

private:
	FilterBankGenerator(const FilterBankGenerator& fbk); //!< don't call
	const FilterBankGenerator& operator=(const FilterBankGenerator& fbk); //!< don't call
};

/*! @file
 * @brief Describes abstract base class for generators of FilterBankEvents
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_0 $
 * $Revision: 1.9 $
 * $State: Exp $
 * $Date: 2004/01/08 02:22:34 $
 */

#endif
