//-*-c++-*-
#ifndef INCLUDED_TailWagMC_h
#define INCLUDED_TailWagMC_h

#include "Motion/MotionCommand.h"
#include "Motion/MotionManager.h"
#include "Shared/get_time.h"
#include "math.h"
#include "Shared/ERS210Info.h"

//! A simple motion command for wagging the tail - you can specify period, magnitude, and tilt
class TailWagMC : public MotionCommand {
 public:
	//!constructor
	TailWagMC() : period(500), magnitude(22*3.141592/180), active(false), tilt() { }
	//!destructor
	virtual ~TailWagMC() {}
	virtual int updateOutputs() {
		if(active && state->robotDesign&WorldState::ERS210Mask) {
			for(unsigned int i=0; i<NumFrames; i++)
				pans[i].set(sin((2*M_PI*(get_time()+i*FrameTime))/period)*magnitude); //bug fix thanks L.A.Olsson@herts.ac.uk
			motman->setOutput(this,ERS210Info::TailOffset+PanOffset,pans);
			motman->setOutput(this,ERS210Info::TailOffset+TiltOffset,tilt);
			return 1;
		} else
			return 0;
	}
	virtual int isDirty() { return active; }
	virtual int isAlive() { return true; }

	void setPeriod(unsigned int p) { period=p; } //!< sets the period of time between swings, in milliseconds
	unsigned int getPeriod() { return period; } //!< gets the period of time between swings, in milliseconds
	void setMagnitude(double mag) { magnitude=mag; } //!< sets the magnitude of swings, in radians
	double getMagnitude() { return magnitude; } //!< gets the magnitude of swings, in radians
	void setTilt(double r) { tilt.set(r,1); }  //!< sets the tilt of the tail while wagging, in radians
	void unsetTilt() { tilt.unset(); } //!< makes the tilt control unspecified, will let something else control tilt
	double getTilt() { return tilt.value; }  //!< sets the tilt of the tail while wagging, in radians
	void setActive(bool a) { active=a; } //!< turns the tail wagger on or off
	bool getActive() { return active; } //!< returns true if this is currently trying to wag the tail
	
 protected:
	unsigned int period; //!< period of time between swings, in milliseconds
	double magnitude; //!< magnitude of swings, in radians
	bool active; //!< true if this is currently trying to wag the tail
	OutputCmd tilt; //!< holds current setting for the tilt joint
	OutputCmd pans[NumFrames]; //!< holds commands for planning ahead the wagging
};

/*! @file
 * @brief Defines TailWagMC, which will wag the tail on a ERS-210 robot.
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_0 $
 * $Revision: 1.5 $
 * $State: Rel $
 * $Date: 2003/09/25 15:27:23 $
 */

#endif

