//-*-c++-*-
#ifndef INCLUDED_PostureMC_h
#define INCLUDED_PostureMC_h

#include "MotionCommand.h"
#include "PostureEngine.h"
#include "MotionManager.h"

//! a MotionCommand shell for PostureEngine
class PostureMC : public MotionCommand, public PostureEngine {
public:
	//!constructor
	PostureMC() : MotionCommand(),PostureEngine(),dirty(true),tolerance(.01) { }
	//!constructor, loads from @a filename
	PostureMC(const char* filename) : MotionCommand(),PostureEngine(filename),dirty(true),tolerance(.01) { }
	//!destructor
	virtual ~PostureMC() {}
	
	//!@name New Stuff
	/*!@brief call this if you call PostureEngine::setOutputCmd(), that doesn't know about dirty flags */
	/* This is the downside of making setOutputCmd() not virtual - if you pass this around as just a
	 * PostureEngine, calls made to that won't be able to update the dirty flag automatically.  However,
	 * if the object is typed as a PostureMC at the time the call is made, then you don't have to worry.\n
	 * These are also not virtual - otherwise you'd have to still check the motion out and it would make
	 * this all pointless! @return @c *this */
	PostureMC& setDirty(bool d=true) { dirty=d; return *this; } 
	bool isDirty() const {return dirty;} //!< if you want to check the dirty flag
	virtual PostureMC& setTolerance(float t) { tolerance=t; return *this; } //!< sets #tolerance, returns @c *this
	virtual float getTolerance() { return tolerance; } //!< returns #tolerance
	//@}


	//!@name MotionCommand Stuff
	virtual int updateOutputs() {
		for(unsigned int i=0; i<NumOutputs; i++)
			if(cmds[i].weight>0)
				motman->setOutput(this,i,cmds[i]);
		bool r=dirty; dirty=false; return r;
	}
	virtual int isDirty() { return dirty; }

	//! returns non-zero (true) if PostureEngine::maxdiff() between this and the current position is over #tolerance
	/*! This is handy so you can set to have the robot go to a position and then automatically remove
	 *  the MotionCommand when it gets there - but beware fighting Postures which average out and neither
	 *  succeeds */
	virtual int isAlive() {
		if(dirty || getAutoPrune()==false) //doesn't bother computing if we're staying alive anyway
			return true;
		PostureEngine tmp; 
		tmp.takeSnapshot(); 
		return (maxdiff(tmp)>tolerance);
	}
	//@}

	//!Had to override stuff to manage a dirty flag
	//!@name PostureEngine Stuff
	inline virtual void takeSnapshot() { dirty=true; PostureEngine::takeSnapshot(); }
	inline virtual void clear() { dirty=true; PostureEngine::clear(); }
	inline virtual PostureMC& setOverlay(const PostureEngine& pe) { dirty=true; PostureEngine::setOverlay(pe); return *this; }
	inline virtual PostureMC& setUnderlay(const PostureEngine& pe) { dirty=true; PostureEngine::setUnderlay(pe); return *this; }
	inline virtual PostureMC& setAverage(const PostureEngine& pe,float w=0.5) { dirty=true; PostureEngine::setAverage(pe,w); return *this; }
	inline virtual PostureMC& setCombine(const PostureEngine& pe) { dirty=true; PostureEngine::setCombine(pe); return *this; }
	inline PostureMC& setOutputCmd(unsigned int i, const OutputCmd& c) { dirty=true; PostureEngine::setOutputCmd(i,c); return *this; }
	inline virtual unsigned int LoadBuffer(const char buf[], unsigned int len) { dirty=true; return PostureEngine::LoadBuffer(buf,len); }
	//@}

protected:
	bool dirty; //!< true if changes have been made since last updateOutputs()
	float tolerance; //!< when autopruning, if the maxdiff() of this posture and the robot's current position is below this value, isAlive() will be false, defaults to 0.01 (5.7 degree error)

	//!defines Registerable stuff
	/*	REGDEFCUSTOMMEM(PostureMC) {
		MotionCommand::registerDataRegions(regions);
		registerRegion(regions,&cmds,&cmds,sizeof(cmds));//for PostureEngine, LoadSave doesn't have any (nonstatic) members
		registerRegion(regions,&dirty,&tolerance,sizeof(tolerance));
		}*/
};

/*! @file
 * @brief Defines PostureMC, a MotionCommand shell for PostureEngine
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_0 $
 * $Revision: 1.5 $
 * $State: Exp $
 * $Date: 2004/01/06 01:01:50 $
 */

#endif
