//-*-c++-*-
#ifndef INCLUDED_AlanBehavior_h_
#define INCLUDED_AlanBehavior_h_

#include "Behaviors/BehaviorBase.h"
#include "Motion/MotionManager.h"
#include "Motion/MMAccessor.h"
#include "Shared/SharedObject.h"
#include "Shared/WorldState.h"
#include "Events/EventRouter.h"
#include "Motion/PostureMC.h"
#include "Shared/ERS210Info.h"

//! just a little demo behavior which lifts a leg higher as more pressure is put on a head button
/*! Based on an idea from Alan Chun-ho Ho for a basic demo program */
class AlanBehavior : public BehaviorBase {
public:
	AlanBehavior()
		: BehaviorBase(), pose_id(MotionManager::invalid_MC_ID)
	{}
	
	virtual void DoStart() {
		//call superclass first for housekeeping:
		BehaviorBase::DoStart();

		//now do your code:
		
		// creates a PostureMC class to move the joint(s) and adds it to global MotionManager
		pose_id=motman->addMotion(SharedObject<PostureMC>());
		// subscribe to sensor updated events through the global EventRouter
		erouter->addListener(this,EventBase::sensorEGID,SensorSourceID::UpdatedSID);
	}
	
	virtual void DoStop() {
		//do your code first:
		motman->removeMotion(pose_id);  // removes your posture controller
		erouter->forgetListener(this); // stops getting events (and timers, if we had any)

		//but don't forget to call superclass at the end:
		BehaviorBase::DoStop();
	}
	
	virtual void processEvent(const EventBase& event) {
		// to be more general, let's check that it's the right event:
		if(event.getGeneratorID()==EventBase::sensorEGID) {
			//only really works on the 210 model - the others don't have a proper pressure sensor
			//(the 220's antenna-thing is close, but doesn't give a continuous range)
			if( !(state->robotDesign&WorldState::ERS210Mask) )
				return;
			//we'll need to specify the ERS210Info namespace below when
			//referencing the button offsets so that this will compile for
			//the ERS-7 as well (which lacks front and back head buttons),
			//but for your own code, you could leave it off if you weren't
			//worried about compiling for other models.

			// "checks out" the posture motion command from MotionManager
			// (this is the PostureMC we created in DoStart())
			MMAccessor<PostureMC> pose_mc(pose_id);
			
			//Joint offsets are defined in ERS210Info.h, ERS220Info.h and ERS2xxInfo.h
			unsigned int joint=LFrLegOffset+RotatorOffset;
			
			//state is a global instantiation of WorldState, kept up to date by framework;
			//pressure is in range 0 to 1 - we use the pressure on the front head button here
			float pressure=state->buttons[ERS210Info::HeadFrButOffset];
			std::cout << "HeadFrBut Pressure: " << pressure << std::endl;
			
			//outputRanges is a constant table, also defined in ERS210Info.h or ERS220Info.h
			float angle=outputRanges[joint][MaxRange]*pressure;

			// now send the joint angle to the posture motion command
			pose_mc->setOutputCmd(joint,angle);

			//let's do the whole thing again with the other head button for the other leg:
			// (cutting out a some of the intermediary steps this time)
			joint=RFrLegOffset+RotatorOffset;
			pose_mc->setOutputCmd(joint,outputRanges[joint][MaxRange]*state->buttons[ERS210Info::HeadBkButOffset]);

			// notice that there's no "check in" for pose_mc
			// MMAccessor's destructor does this automatically

		} else {
			//should never happen
			cout << "Unhandled Event:" << event.getName() << endl;
		}
	}
	
	virtual std::string getName() const {
		// Name is used for menus, or debugging.
		return "AlanBehavior";
	}
	
	static std::string getClassDescription() {
		// This string will be shown by the HelpControl or by the tooltips of the Controller GUI
		return "Lifts the left/right front legs higher as more pressure is applied to the front/back head buttons";
	}
	
protected:
	MotionManager::MC_ID pose_id; //!< ID of PostureMC, set up in DoStart() and used in processEvent()
};

/*! @file
 * @brief Defines AlanBehavior, a little demo behavior which lifts a leg higher as more pressure is put on the back head button
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_0 $
 * $Revision: 1.5 $
 * $State: Exp $
 * $Date: 2003/12/23 06:33:42 $
 */

#endif
