#include "PostureEditor.h"
#include "Motion/MMAccessor.h"
#include "ValueEditControl.h"
#include "NullControl.h"

PostureEditor::PostureEditor()
	: ControlBase("Posture Editor","Allows you to load, save, and numerically edit the posture"), 
		poseMC(), poseID(MotionManager::invalid_MC_ID), lastSlot(NULL), loadPose(NULL), savePose(NULL)
{
	// add load and save menus
	pushSlot(loadPose=new FileInputControl("Load Posture","Select a posture to open",config->motion.root));
	loadPose->setFilter("*.pos");
	pushSlot(savePose=new StringInputControl("Save Posture","Please enter the filename to save to (in "+config->motion.root+")"));

	// add submenu for weight editors
	ControlBase * weights;
	pushSlot(weights=new ControlBase("Weights","Set the weights for outputs"));
	for(unsigned int i=0; i<NumOutputs; i++) {
		std::string tmp(outputNames[i]);
		weights->pushSlot(new ValueEditControl<float>(tmp,&poseMC->getOutputCmd(i).weight));
	}

	pushSlot(NULL); // a separator for clarity

	// add actual value editors
	for(unsigned int i=0; i<NumOutputs; i++) {
		std::string tmp(outputNames[i]);
		pushSlot(new ValueEditControl<float>(tmp,&poseMC->getOutputCmd(i).value));
	}
}

ControlBase *
PostureEditor::activate(MotionManager::MC_ID disp_id, Socket * gui) {
	// start off with current pose
	poseMC->takeSnapshot();
	// clear the LEDs though
	for(unsigned int i=LEDOffset; i<LEDOffset+NumLEDs; i++)
		poseMC->setOutputCmd(i,0);
	// add it to motion manager above estop
	poseID=motman->addMotion(poseMC,MotionManager::kEmergencyPriority+1,false);
	// call super class
	return ControlBase::activate(disp_id,gui);
}

void
PostureEditor::refresh() {
	if(lastSlot==loadPose) {
		// we just got back from the load menu
		MMAccessor<PostureMC>(poseID)->LoadFile(loadPose->getLastInput().c_str());
	} else if(lastSlot==savePose || savePose->getLastInput().size()>0) {
		// we just got back from the save menu
		MMAccessor<PostureMC>(poseID)->SaveFile((config->motion.root+"/"+savePose->getLastInput()).c_str());
		savePose->takeInput("");
	}
	lastSlot=NULL;
	ControlBase::refresh();
}

void
PostureEditor::deactivate() {
	motman->removeMotion(poseID);
	poseID=MotionManager::invalid_MC_ID;
	ControlBase::deactivate();
}

ControlBase*
PostureEditor::doSelect() {
	// record the option that is being selected, in case it's the load or save
	lastSlot=options[hilights.front()];
	// but do what we'd normally do (select that option)
	return ControlBase::doSelect();
}


/*! @file
 * @brief Describes PostureEditor, which allows numeric control of joints and LEDs
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_0 $
 * $Revision: 1.2 $
 * $State: Exp $
 * $Date: 2004/01/11 07:30:02 $
 */
