function stabilize2(path, outpath)
% Performs image stabilization on images in directory indicated by path.

% Configuration constants:
% Stabilized image size constants. Make sure they'll fit everything
S_cols = 450;
S_rows = 350;
% P factor (think PID controller) for enforcing zero rotation
R_Pfactor = 1/18;
% P factor for enforcing constant vertical image registration
V_Pfactor = 1/25;
% P factor for enforcing constant horizontal image registration
H_Pfactor = 1/25;

% Output image counter. Starts at 0
outcount = 0;

% Get list of PNG files in path. The dir function will sort them by
% keyframe number.
Pngs = dir([path '/*.png']);

% Load the first image in the sequence
im1 = imread([path '/' Pngs(1).name]);
% Isolate a single channel and convert it to doubles
im1 = double(im1(:,:,1));
% Show image #1
S = register(im1,S_cols,S_rows,0,0);
figure; colormap gray; imagesc(S); axis equal; drawnow;
% Write this first image
imwrite(S,sprintf('%s/out%.5d.png',outpath,outcount)); outcount=outcount+1;
%subplot(1,2,1); colormap gray; imagesc(im1); axis equal;
%subplot(1,2,2); colormap gray; imagesc(im1); axis equal; drawnow;
%pause

% Initialize the homography we're computing throughout the sequence to I
H = eye(3);

% Loop through all the remaining images
for png = Pngs(2:length(Pngs))'
  % Load current image. Same procedure as before
  im2 = imread([path '/' png.name]); im2 = double(im2(:,:,1));

  % Compute H matrix between both images. Note that we don't carry over
  % the scaled image from before--we just multiply this H with the
  % H we're carrying through the whole sequence. By doing this, we
  % avoid image degradation problems that come from warping
  local_H = ransacH(im2, im1);
  H = H*local_H;

  % Scale H to avoid numerical instability and facilitate calculations below.
  H = H / H(3,3);

  % LONG-TERM ERROR CORRECTION
  % 1. Rotation
  % Isolate rotation error from H matrix.
  R_err = -atan2(H(2,1),H(1,1));
  % Compute PID correction for error. Right now it's just P
  R_PID = R_err * R_Pfactor;
  % Correct with the PID correction
  R_correct = [ cos(R_PID) -sin(R_PID) 0; sin(R_PID) cos(R_PID) 0; 0 0 1 ];
  H = R_correct*H;

  % 2. Vertical displacement
  % Isolate vertical displacement from H matrix.
  V_err = -H(2,3);
  % Compute PID correction for error. Right now it's just P
  V_PID = V_err * V_Pfactor;
  % Correct with the PID correction
  H(2,3) = H(2,3) + V_PID;

  % 3. Horizontal displacement
  % Isolate horizontal displacement from H matrix.
  H_err = -H(1,3);
  % Compute PID correction for error. Right now it's just P
  H_PID = H_err * H_Pfactor;
  % Correct with the PID correction
  H(1,3) = H(1,3) + H_PID;

  % Now show the warped image
  Iwarp = warpImage(im2,H);
  %subplot(1,2,1); colormap gray; imagesc(im2); axis equal;
  %subplot(1,2,2); colormap gray; imagesc(Iwarp); axis equal; drawnow;
  S = register(Iwarp,S_cols,S_rows,H(1,3)*100,H(2,3)*100);
  colormap gray; imagesc(S); axis equal; drawnow;
  % Write this image
  imwrite(S,sprintf('%s/out%.5d.png',outpath,outcount)); outcount=outcount+1;

  % Stash im2 in im1 for the next iteration
  im1 = im2;
end


% Helper functions

% Image registration
function Iout = register(Iin,cols,rows,center_x,center_y)
% Initialize everything to black
Iout = zeros(rows,cols);
% Save rows/2 and cols/2
r_half = round(rows/2);
c_half = round(cols/2);
% Save Iin dims and dims/2
[irows, icols] = size(Iin);
ir_half = round(irows/2);
ic_half = round(icols/2);
% Compute ranges for image transfer
crange = (1:icols) + c_half-ic_half + round(center_x);
rrange = (1:irows) + r_half-ir_half + round(center_y);
% Register
Iout(rrange,crange) = Iin;
Iout = uint8(Iout);
