function m = WalkCalibration(base_file_name,outname)
% WALKCALIBRATION Takes a set of actual and commanded speeds and
% performs linear least squares on a set of basis functions to
% determine proper calibration.
%
% Pass the base name of the input data set as first input and an
% output filename (if desired)
% Data files are broken into 6 categories:
% Forward-Strafe  (fs)
% Forward-Rotate  (fr)
% Strafe-Rotate   (sr)
% Backward-Rotate (br)
% Backward-Strafe (bs)
% Pure Rotation   (rr)
%
% So the file name for each should be BASEfs.txt, BASEfr.txt...
%
% The fs,fr,sr,rr categories are used to compute a forward matrix,
% and the sr,br,bs,rr categories are used to compute a backward
% matrix (since we assume left/right similarity, but
% forward/backward may not be a linear transition (and doesn't
% appear to be)
%
% First 3 columns of each file is the actual measured speed in
% forward, strafe, and rotation.  Columns 4-6 are the commanded
% values.

dat{1}=load(strcat(base_file_name,'fs.txt'));
dat{2}=load(strcat(base_file_name,'fr.txt'));
dat{3}=load(strcat(base_file_name,'sr.txt'));
dat{4}=load(strcat(base_file_name,'br.txt'));
dat{5}=load(strcat(base_file_name,'bs.txt'));
dat{6}=load(strcat(base_file_name,'rr.txt'));

forward=[ dat{1} ; dat{2} ; dat{3} ; dat{6} ];
backward=[ dat{3} ; dat{4} ; dat{5} ; dat{6} ];

disp('Forward Matrix:');
intend=forward(:,1:3);
cmd=forward(:,4:6);
data=mungeRawData(intend,cmd);
m{1}=doLLS(data);

disp('Errors:');
calcmd=m{1}*data(1:3:size(data,1),1:(size(data,2)-1)/3)';
err=(cmd-calcmd');
err=[intend err (sum((cmd'-calcmd).^2)')];
%disp(sortrows(err,7));
disp(sprintf('Total Error=%g;    N=%g;    Avg. Err=%g',sum(err(:,7)),size(intend,1),sum(err(:,7))/size(intend,1)));

disp('Backward Matrix:');
intend=backward(:,1:3);
cmd=backward(:,4:6);
data=mungeRawData(intend,cmd);
m{2}=doLLS(data);

disp('Errors:');
calcmd=m{2}*data(1:3:size(data,1),1:(size(data,2)-1)/3)';
err=(cmd-calcmd');
err=[intend err (sum((cmd'-calcmd).^2)')];
%disp(sortrows(err,7));
disp(sprintf('Total Error=%g;    N=%g;    Avg. Err=%g',sum(err(:,7)),size(intend,1),sum(err(:,7))/size(intend,1)));

disp(' ');
disp('Final Forward Matrix:');
disp(m{1});
disp('Final Backward Matrix:');
disp(m{2});

if nargin>1
	disp(sprintf('Saving to %s',outname));
	forwC=m{1};
	backC=m{2};
	save(outname,'forwC','backC','-ASCII');
end


function data = mungeRawData(itd,cmd)
abs_sr=abs(itd(:,2:3));
cross_mult=itd.*circshift(itd,[0 1]);
%ang=atan2(itd(:,2),itd(:,1));
ang=atan2(itd(:,2),abs(itd(:,1)));
spd=itd(:,1).^2+itd(:,2).^2;

row=[itd abs_sr ang spd cross_mult ones(size(itd,1),1) ];

data=zeros(3*size(row,1),size(row,2));
data(1:3:size(data,1),:)=row;
data=[ data circshift(data,1) circshift(data,2) reshape(cmd',size(data,1),1) ];



function ans = doLLS(data)
orig=data;

% nevermind about this - I guess it doesn't make a difference after
% all...
% % we scale by the measurement error for better accuracy
% % I figure my measurements are accurate to within 2.5 mm/s, or 1 degree/s
%scale=[ 2.5 2.5 1/180*pi 1 ];
%tmp=repmat(scale,size(data,1),3);
%tmp=[tmp repmat(scale(1:3)',size(data,1)/3,1)];
%data=data./tmp;
%disp([ getIntended(data./tmp)' getCommanded(data./tmp)' ]);

%now we pull out the big gun:
ans=data(:,1:size(data,2)-1)\data(:,end);
ans=reshape(ans,(size(data,2)-1)/3,3)';

% or you could do it the hard way:
%A=data(:,1:12);
%b=data(:,end);
%[U,S,V]=svd(A,0);
%c = U'*b;
%y = c./diag(S);
%x = V*y;
%e = A*x - b;
%disp(sprintf('Least Squares error is: %g',norm(e)));
%ans=reshape(x,4,3)';

% nevermind...
% % and don't forget to unscale
%tmp=[ scale/scale(1) ; scale/scale(2) ; scale/scale(3) ];
%ans=ans./tmp;

%for visualization
t1=getIntended(orig)';
t2=getCommanded(orig)';
% see predictions
%visPredict(t1,t2,ans);
% see errors
%visError(t1,t2,ans);

function ans = getIntended(data)
ans=data(1:3:size(data,1),1:((size(data,2)-1)/3))';

function ans = getCommanded(data)
ans=reshape(data(:,end),3,size(data,1)/3);


function visPredict(t1,t2,ans)
visPredicts(t1,t2,ans,[1 2 3]);
visPredicts(t1,t2,ans,4:7)
visPredicts(t1,t2,ans,8:size(t1,2))

function visError(t1,t2,ans)
visErrors(t1,t2,ans,[1 2 3]);
visErrors(t1,t2,ans,4:7);
visErrors(t1,t2,ans,8:size(t1,2));

function visPredicts(t1,t2,ans,jrange)
figure;
for j=jrange
	for i=1:3,
		subplot(3,size(jrange,2),(j-min(jrange)+1)+(i-1)*size(jrange,2));
		mn=min(t1(:,j));
		mx=max(t1(:,j));
		if(mn==mx)
			x=[mn-1 mx+1];
			y=[ ans(i,j) ans(i,j) ];
		else
			x=[mn mx];
			y=x*ans(i,j);
		end
		plot(t1(:,j),t2(:,i)-(ans(i,:)*t1')'+t1(:,j)*ans(i,j),'b.',x,y,'g-');
	end
end

function visErrors(t1,t2,ans,jrange)
figure;
for j=jrange,
	for i=1:3,
		subplot(3,size(jrange,2),(j-min(jrange)+1)+(i-1)*size(jrange,2));
		mn=min(t1(:,j));
		mx=max(t1(:,j));
		if(mn==mx)
			x=[mn-1 mx+1];
		else
			x=[mn mx];
		end
		plot(t1(:,j),t2(:,i)-(ans(i,:)*t1')','r.',x,[0 0],'g-');
	end
end
